#include "delete.h"
#include "dbhelpers.h"

#include <yandex/maps/wiki/common/batch.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/groupedit/session.h>
#include <yandex/maps/wiki/groupedit/object.h>
#include <yandex/maps/wiki/groupedit/relation.h>
#include <maps/libs/log8/include/log8.h>

namespace fs = std::filesystem;
namespace rev = maps::wiki::revision;
namespace mwg = maps::wiki::groupedit;

namespace maps {
namespace wiki {
namespace importer {

namespace {

const std::string GROUP_DELETED_ACTION = "group-deleted";

void addToDelete(
    const mwg::Object& object,
    const EditorConfig& config,
    ObjectIds& toDeleteIds)
{
    const auto& category = config.category(object.category());

    ObjectIds toDeleteTableAttrsIds;
    for (const auto& rel : object.relations()) {
        if (rel.type() == mwg::Relation::Type::Slave) {
            const auto& role = category.slaveRole(rel.role());
            if (role.tableRow()) {
                toDeleteTableAttrsIds.insert(rel.otherId());
            }
        }
    }
    toDeleteIds.insert(object.id());
    toDeleteIds.insert(toDeleteTableAttrsIds.begin(), toDeleteTableAttrsIds.end());
}

} // namespace

CommitIds deleteRelations(
    const rev::RevisionIds& relations,
    TaskParams& params)
{
    if (relations.empty()) {
        return {};
    }

    rev::RevisionsGateway gateway(*params.mainTxn);

    CommitIds commitIds;

    common::applyBatchOp<rev::RevisionIds>(relations, COMMIT_BATCH_SIZE, [&](const rev::RevisionIds& batch)
    {
        std::vector<revision::RevisionsGateway::NewRevisionData> preparedRelations;
        for (const auto& revisionId : batch)
        {
            revision::ObjectRevision::Data data;
            data.deleted = true;
            preparedRelations.emplace_back(revisionId, std::move(data));
        }
        commitIds.push_back(gateway.createCommit(preparedRelations, common::ROBOT_UID, params.globalAttrs).id());
    });

    return commitIds;
}

CommitIds deleteObjectsWithTableAttributes(
    const ObjectIds& objectIds,
    TaskParams& params,
    MessageReporter& messageReporter)
{
    if (objectIds.empty()) {
        return {{}, {}};
    }

    ObjectIds toDeleteIds;

    mwg::Session session(*params.mainTxn, params.branchId);
    session.query(objectIds).visit(
        [&](const mwg::Object& object) {
            addToDelete(object, params.editorConfig, toDeleteIds);
        });

    ObjectIds deletedObjectIds;

    auto commitIds = session.query(toDeleteIds).update(
        GROUP_DELETED_ACTION, common::ROBOT_UID, [&](mwg::Object& object)
        {
            deletedObjectIds.emplace(object.id());
            object.setDeleted();
        });

    if (deletedObjectIds.size() != toDeleteIds.size()) {
        messageReporter.error() << "Not all objects are deleted";

        ObjectIds difference;
        std::set_difference(
            toDeleteIds.begin(), toDeleteIds.end(),
            deletedObjectIds.begin(), deletedObjectIds.end(),
            std::inserter(difference, difference.end()));

        for (auto id : difference) {
            messageReporter.error() << "Object " << id << " was not deleted";
        }

        return {{}, {}};
    }

    return CommitIds(commitIds.begin(), commitIds.end());
}

} // importer
} // wiki
} // maps
