#include "edit.h"

#include <yandex/maps/wiki/common/batch.h>
#include <yandex/maps/wiki/common/geom.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/configs/editor/categories.h>
#include <yandex/maps/wiki/configs/editor/master_role.h>
#include <yandex/maps/wiki/configs/editor/slave_role.h>
#include <yandex/maps/wiki/groupedit/session.h>
#include <yandex/maps/wiki/groupedit/object.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/log8/include/log8.h>

namespace mwg = maps::wiki::groupedit;
namespace rev = maps::wiki::revision;

namespace maps {
namespace wiki {
namespace importer {

namespace {

const std::string GROUP_MOVED_ACTION = "group-moved";
const std::string GROUP_MOD_ATTRS_ACTION = "group-modified-attributes";

constexpr size_t OBJECT_EDIT_BATCH_SIZE = 100;

void validateNewRelations(const EditorConfig& editorConfig, const mwg::Object& object)
{
    const auto& categoryId = object.category();
    const auto& category = editorConfig.categories()[categoryId];
    const auto& masterRoles = category.masterRoles();

    for (const auto& relation : object.relations()) {
        if (relation.isDeleted()) {
            continue; // skip check for previous category
        }

        const auto& roleId = relation.role();
        const auto& otherCategoryId = relation.otherCategory();

        if (relation.type() == mwg::Relation::Type::Master) {
            bool found = masterRoles.end() != std::find_if(
                masterRoles.begin(),
                masterRoles.end(),
                [&](const auto& role) {
                    return role.roleId() == roleId && role.categoryId() == otherCategoryId;
                }
            );
            REQUIRE(found, "Invalid master relation: "
                    << otherCategoryId << "," << roleId << "," << categoryId);
        } else {
            bool found =
                category.isSlaveRoleDefined(roleId) &&
                category.slaveRole(roleId).categoryId() == otherCategoryId;
            REQUIRE(found, "Invalid slave relation: "
                    << categoryId << "," << roleId << "," << otherCategoryId);
        }
    }
}

} // namespace

CommitIds editObjectAttributes(
    const Objects& objects,
    TaskParams& params,
    MessageReporter& messageReporter)
{
    ObjectIds objectIds;
    std::map<TObjectId, ObjectPtr> dbIdToObject;
    std::map<TObjectId, ID> dbIdToFeatureId;
    for (const auto& object : objects) {
        objectIds.emplace(object->dbId());
        dbIdToObject.emplace(object->dbId(), object);
        dbIdToFeatureId.emplace(object->dbId(), object->id());
    }

    ObjectIds updatedObjectIds;
    CommitIds commitIds;

    mwg::Session session(*params.mainTxn, params.branchId);
    common::applyBatchOp(objectIds,
        OBJECT_EDIT_BATCH_SIZE,
        [&](const ObjectIds& batchObjectIds) {
            auto newModAttrsCommitIds = session.query(batchObjectIds).update(
                GROUP_MOD_ATTRS_ACTION, common::ROBOT_UID,
                [&](mwg::Object& prevObject) {
                    try {
                        auto& nextObject = dbIdToObject.at(prevObject.id());

                        updatedObjectIds.emplace(nextObject->dbId());

                        if (prevObject.category() != nextObject->category().id()) {
                            // Currently only POI categories have kind=poi.
                            // Other categories have empty kind and should not be converted
                            REQUIRE(!nextObject->category().kind().empty()
                                && params.editorConfig.category(prevObject.category()).kind() == nextObject->category().kind(),
                                "Attempt to change category "
                                << prevObject.category() << " of kind " << params.editorConfig.category(prevObject.category()).kind()
                                << " to another category "
                                << nextObject->category().id() << " of kind " << nextObject->category().kind());
                            prevObject.changeCategory(nextObject->category().id());
                            validateNewRelations(params.editorConfig, prevObject);
                        }

                        for (const auto& attribute : nextObject->attributesToAdd()) {
                            if (attribute.first.starts_with("cat:")) {
                                continue;
                            }
                            prevObject.setAttribute(attribute.first, attribute.second);
                        }
                        for (const auto& attribute : nextObject->attributesToDelete()) {
                            REQUIRE(!attribute.starts_with("cat:"), "Can't delete category: " << attribute);
                            prevObject.removeAttribute(attribute);
                        }
                    } catch (const maps::Exception& e) {
                        messageReporter.error(dbIdToFeatureId[prevObject.id()])
                            << "Object " << prevObject.id() << ". " << e.what();
                    }
                });
            commitIds.insert(commitIds.end(), newModAttrsCommitIds.begin(), newModAttrsCommitIds.end());

            auto newMovedCommitIds = session.query(batchObjectIds).update(
                GROUP_MOVED_ACTION, common::ROBOT_UID,
                [&](mwg::Object& prevObject) {
                    try {
                        auto& nextObject = dbIdToObject.at(prevObject.id());

                        updatedObjectIds.emplace(nextObject->dbId());

                        if (prevObject.geometryWkb() && !nextObject->wkb().empty()) {
                            prevObject.setGeometryWkb(
                                geolib3::WKB::toString<geolib3::SimpleGeometryVariant>(
                                    geolib3::convertGeodeticToMercator(
                                        geolib3::WKB::read<geolib3::SimpleGeometryVariant>(
                                            nextObject->wkb()))));
                        }
                    } catch (const maps::Exception& e) {
                        messageReporter.error(dbIdToFeatureId[prevObject.id()]) << e.what();
                    }
                });
            commitIds.insert(commitIds.end(), newMovedCommitIds.begin(), newMovedCommitIds.end());
        });

    if (updatedObjectIds.size() != objectIds.size()) {
        messageReporter.error() << "Not all objects are updated";

        ObjectIds difference;
        std::set_difference(
            objectIds.begin(), objectIds.end(),
            updatedObjectIds.begin(), updatedObjectIds.end(),
            std::inserter(difference, difference.end()));

        for (auto id : difference) {
            messageReporter.error(dbIdToFeatureId[id])
                << "Object " << id << " was not updated";
        }

        return {};
    }

    if (messageReporter.hasErrors()) {
        return {};
    }

    return commitIds;
}

} // importer
} // wiki
} // maps
