#pragma once

#include "common.h"
#include "editor_config.h"
#include "rubric_matcher.h"
#include "object.h"

namespace maps {
namespace wiki {
namespace importer {

/**
Contains callbacks for processing fields and geometry of the gdal feature
*/
class FeatureProcessor
{
public:
    using FieldProcessor = std::function<void(ObjectPtr&, const std::string&)>;
    using GeometryProcessor = std::function<Objects(ObjectPtr&, const OGRGeometry*)>;

    FeatureProcessor(const EditorConfig& editorConfig, const cfg::Category& category, const RubricsMatcher& rubricsMatcher);

    const cfg::Category& category() const { return category_; }

    Objects processGeometry(ObjectPtr& object, const OGRGeometry*) const;

    void processField(
        ObjectPtr& object,
        const std::string& fieldName,
        const std::string& value) const;

public:
    void setGeometryProcessor(const GeometryProcessor& processor);

    bool hasFieldProcessor(const std::string& fieldName) const;

    void setFieldProcessor(
        const std::string& fieldName,
        const FieldProcessor& processor);
public:
    // Useful helper methods to quickly create a field processor for popular cases

    void addNormalField(
        const std::string& fieldName,
        const std::string& attributeId,
        OGRFieldType fieldType);

    void addMasterField(
        const std::string& fieldName,
        const std::string& roleId,
        const cfg::Category& masterCategory);

    void addFutureMasterField(
        const std::string& fieldName,
        const std::string& roleId,
        const std::string& masterLayer);

    void addMasterIndoorLevel(
        ObjectPtr& object,
        const std::string& role,
        ID masterIndoorLevelId) const;

    void addDbIdField(
        const std::string& fieldName);

    void addTempBldComplexField(
        const std::string& fieldName);

    void addIndoorRubricField(
        const std::string& fieldName,
        const std::string& attrName);

    void addMasterPoiIdsField(
        const std::string& fieldName);

    /**
    Try match pattern to find fields:
        nm_0_ru_l, nm_1_ru_l, nm_0_fr, etc.
    */
    bool tryAddNameField(
        const std::string& fieldName,
        OGRFieldType fieldType);

    /**
    Try match pattern to find fields:
        d_nm_0_ru, d_nm_1_ru, d_nm_0_fr, etc.
    */
    bool tryDeleteNameField(
        const std::string& fieldName,
        OGRFieldType fieldType);

    /**
    Try match one of the patterns:
        <role_id>_<slave_attribute>
        <role_id>_<master_category>
    */
    bool tryParseRelation(
        const std::string& fieldName);

private:
    const EditorConfig& editorConfig_;
    const cfg::Category& category_;
    const RubricsMatcher& rubricsMatcher_;
    std::map<std::string, FieldProcessor> fieldProcessors_;
    GeometryProcessor geometryProcessor_;
};

} // importer
} // wiki
} // maps
