#include "gdal_helpers.h"
#include "layer_processor.h"

#include <maps/libs/log8/include/log8.h>

#include <vector>

namespace fs = std::filesystem;

namespace maps {
namespace wiki {
namespace importer {

namespace {

using GDALDatasetHolder = std::unique_ptr<GDALDataset>;

const std::string SHAPEFILE_EXTENSION = ".shp";
const std::string GEOJSON_EXTENSION = ".geojson";
const std::string GEOJSON_ALT_EXTENSION = ".json";
const std::string MIF_EXTENSION = ".mif";
const std::string DBF_EXTENSION = ".dbf";
const std::string CSV_EXTENSION = ".csv";

const std::vector<std::string> SUPPORTED_EXTENSIONS {
    SHAPEFILE_EXTENSION,
    GEOJSON_EXTENSION,
    GEOJSON_ALT_EXTENSION,
    MIF_EXTENSION,
    CSV_EXTENSION
};

std::map<std::string, GDALDatasetHolder> findDataSources(const fs::path& dataDir)
{
    std::map<std::string, GDALDatasetHolder> dataSources;

    for (fs::directory_iterator it(dataDir); it != fs::directory_iterator(); ++it) {
        for (const auto& ext : SUPPORTED_EXTENSIONS) {
            if (it->path().extension() != ext) {
                continue;
            }
            GDALDatasetHolder dataSource(
                GDALDataset::Open(it->path().string().c_str(), FALSE));
            if (!dataSource) {
                continue;
            }
            auto layerName = it->path().stem().string();
            dataSources.emplace(layerName, std::move(dataSource));
        }
    }

    for (fs::directory_iterator it(dataDir); it != fs::directory_iterator(); ++it) {
        if (it->path().extension() != DBF_EXTENSION) {
            continue;
        }
        auto layerName = it->path().stem().string();
        if (dataSources.count(layerName)) {
            continue;
        }
        GDALDatasetHolder dataSource(
            GDALDataset::Open(it->path().string().c_str(), FALSE));
        if (dataSource) {
            dataSources.emplace(layerName, std::move(dataSource));
        }
    }
    return dataSources;
}

} // namespace

Objects gdal2objects(
    const fs::path& dataDir,
    Action action,
    ObjectsCache& cache,
    const EditorConfig& editorConfig,
    const ImportConfig& importConfig,
    MessageReporter& messageReporter)
{
    // Support GetSpatialRef without GDAL data.
    CPLSetConfigOption("USE_OSR_FIND_MATCHES", "NO");

    OGRRegisterAll();

    auto dataSources = findDataSources(dataDir);
    if (dataSources.empty()) {
        messageReporter.error() << "Can't open data file from dir " << dataDir;
        return {};
    }

    Objects objects;

    for (const auto& pair : dataSources) {
        const auto& layerName = pair.first;
        const auto& dataSource = pair.second;

        if (dataSource->GetLayerCount() != 1) {
            messageReporter.error() << "Wrong layer count for data source " << layerName;
            continue;
        }

        auto* layer = dataSource->GetLayer(0);
        if (!layer)
        {
            messageReporter.error() << "Wrong layer " << layerName;
            continue;
        }

        LayerProcessor processor(action, cache, editorConfig, importConfig, layerName);

        try {
            processor.findAndPrepareCategories(*layer);
        }  catch (const std::exception& e) {
            messageReporter.error() << e.what();
            return {};
        }

        layer->ResetReading();
        OGRFeaturePtr ogrFeature(layer->GetNextFeature());
        while (ogrFeature) {
            try {
                objects.splice(objects.end(), processor.prepare(ogrFeature));
            } catch (const std::exception& e) {
                messageReporter.error(ID{layerName, ogrFeature->GetFID()}) << e.what();
            }
            ogrFeature.reset(layer->GetNextFeature());
        }
    }
    return objects;
}

} // importer
} // wiki
} // maps
