#include "import_config.h"

#include <maps/libs/xml/include/xml.h>

#include <library/cpp/resource/resource.h>

namespace maps {
namespace wiki {
namespace importer {

namespace {

const std::string IMPORT_CONFIG_RESOURCE_KEY = "/wiki-import-worker-config";

const std::unordered_set<std::string> EMPTY_FIELD_SET;

} // namespace

ImportConfig::ImportConfig()
{
    auto configContent = NResource::Find(IMPORT_CONFIG_RESOURCE_KEY);

    xml3::Doc doc(configContent, xml3::Doc::Source::String);

    auto attributeAliasesNodes = doc.nodes("/import/attribute-aliases/attribute");
    for (size_t i = 0; i < attributeAliasesNodes.size(); ++i) {
        auto alias = attributeAliasesNodes[i].attr<std::string>("field-name");
        auto attributeName = attributeAliasesNodes[i].attr<std::string>("name");
        auto categoryId = attributeAliasesNodes[i].attr<std::string>("category", {});

        if (!categoryId.empty()) {
            categoryAttributeAliases_[categoryId].emplace(std::move(alias), std::move(attributeName));
        } else {
            auto ret = commonAttributeAliases_.emplace(std::move(alias), std::move(attributeName));
            REQUIRE(ret.second, "Duplicate alias '" << ret.first->first << "'");
        }
    }

    auto masterRelationNodes = doc.nodes("/import/master-relation-aliases/master-relation");
    for (size_t i = 0; i < masterRelationNodes.size(); ++i) {
        auto alias = masterRelationNodes[i].attr<std::string>("field-name");
        auto role = masterRelationNodes[i].attr<std::string>("role");
        auto slaveCategoryId = masterRelationNodes[i].attr<std::string>("slave-category");
        auto masterCategoryId = masterRelationNodes[i].attr<std::string>("master-category");

        auto ret = categoryRelationAliases_[slaveCategoryId].emplace(
            std::move(alias),
            MasterRelationInfo{std::move(masterCategoryId), std::move(role)});
        REQUIRE(ret.second, "Duplicate master relation '" << ret.first->first << "'");
    }

    auto futureRelationNodes = doc.nodes("/import/future-master-relation-aliases/future-master-relation");
    for (size_t i = 0; i < futureRelationNodes.size(); ++i) {
        auto alias = futureRelationNodes[i].attr<std::string>("field-name");
        auto role = futureRelationNodes[i].attr<std::string>("role");
        auto slaveCategoryId = futureRelationNodes[i].attr<std::string>("slave-category");
        auto masterLayer = futureRelationNodes[i].attr<std::string>("master-layer");

        auto ret = categoryFutureRelationAliases_[slaveCategoryId].emplace(
            std::move(alias),
            FutureMasterRelationInfo{std::move(masterLayer), std::move(role)});
        REQUIRE(ret.second, "Duplicate future master relation '" << ret.first->first << "'");
    }

    auto futureNameRelationNodes = doc.nodes("/import/future-layer-name-relations/future-layer-name-relation");
    for (size_t i = 0; i < futureNameRelationNodes.size(); ++i) {
        auto role = futureNameRelationNodes[i].attr<std::string>("role");
        auto slaveCategoryId = futureNameRelationNodes[i].attr<std::string>("slave-category");
        auto masterLayer = futureNameRelationNodes[i].attr<std::string>("master-layer");

        auto ret = layerNameFutureRelations_.emplace(
            std::move(slaveCategoryId),
            FutureMasterRelationInfo{std::move(masterLayer), std::move(role)});
        REQUIRE(ret.second, "Duplicate future master relation '" << ret.first->first << "'");
    }

    auto requiredFieldNodes = doc.nodes("/import/required-fields/category");
    for (size_t i = 0; i < requiredFieldNodes.size(); ++i) {
        auto categoryId = requiredFieldNodes[i].attr<std::string>("id");
        auto fieldNodes = requiredFieldNodes[i].nodes("field");
        for (size_t j = 0; j < fieldNodes.size(); ++j) {
            auto fieldName = fieldNodes[j].attr<std::string>("name");

            auto ret = categoryRequiredFields_[categoryId].insert(std::move(fieldName));
            REQUIRE(ret.second, "Duplicate field '" << *ret.first << "'");
        }
    }

    auto deleteCategoryNodes = doc.nodes("/import/delete/allowed-categories/category");
    for (size_t i = 0; i < deleteCategoryNodes.size(); ++i) {
        auto categoryId = deleteCategoryNodes[i].value<std::string>();
        allowedCategoryIdsToDelete_.insert(std::move(categoryId));
    }
}

std::string ImportConfig::getAttributeName(
    const std::string& categoryId,
    const std::string& fieldName) const
{
    auto catIt = categoryAttributeAliases_.find(categoryId);
    if (catIt != categoryAttributeAliases_.end()) {
        const auto& aliases = catIt->second;
        auto it = aliases.find(fieldName);
        if (it != aliases.end()) {
            return it->second;
        }
    }

    auto it = commonAttributeAliases_.find(fieldName);
    if (it != commonAttributeAliases_.end()) {
        return it->second;
    }
    return fieldName;
}

std::optional<MasterRelationInfo> ImportConfig::getMasterRelation(
    const std::string& slaveCategoryId,
    const std::string& fieldName) const
{
    auto catIt = categoryRelationAliases_.find(slaveCategoryId);
    if (catIt == categoryRelationAliases_.end()) {
        return std::nullopt;
    }
    const auto& aliases = catIt->second;
    auto it = aliases.find(fieldName);
    if (it == aliases.end()) {
        return std::nullopt;
    }
    return it->second;
}

std::optional<FutureMasterRelationInfo> ImportConfig::getFutureMasterRelation(
    const std::string& slaveCategoryId,
    const std::string& fieldName) const
{
    auto catIt = categoryFutureRelationAliases_.find(slaveCategoryId);
    if (catIt == categoryFutureRelationAliases_.end()) {
        return std::nullopt;
    }
    const auto& aliases = catIt->second;
    auto it = aliases.find(fieldName);
    if (it == aliases.end()) {
        return std::nullopt;
    }
    return it->second;
}

std::optional<FutureMasterRelationInfo> ImportConfig::getIndoorFutureMasterRelationInfo(
    const std::string& slaveCategoryId) const
{
    auto it = layerNameFutureRelations_.find(slaveCategoryId);
    if (it == layerNameFutureRelations_.end()) {
        return std::nullopt;
    }
    return it->second;
}
const std::unordered_set<std::string>& ImportConfig::requiredFields(
    const std::string& categoryId) const
{
    auto catIt = categoryRequiredFields_.find(categoryId);
    if (catIt == categoryRequiredFields_.end()) {
        return EMPTY_FIELD_SET;
    }
    return catIt->second;
}

bool ImportConfig::isCategoryAllowedToDelete(
    const std::string& categoryId) const
{
    return allowedCategoryIdsToDelete_.count(categoryId) > 0;
}

} // importer
} // wiki
} // maps
