#pragma once

#include "common.h"
#include "editor_config.h"
#include "import_config.h"
#include "rubric_matcher.h"
#include "object.h"
#include "feature_processor.h"

namespace maps {
namespace wiki {
namespace importer {

struct OGRFeatureDeleter
{
    void operator()(OGRFeature* ogrFeature)
    {
        OGRFeature::DestroyFeature(ogrFeature);
    }
};

using OGRFeaturePtr = std::unique_ptr<OGRFeature, OGRFeatureDeleter>;

enum class WorkMode
{
    Category,
    Poi,
    Ad,
    Addr,
    Rd,
    Contour,
    ContourElement,
    IndoorLevel,
    IndoorPoi
};

class LayerProcessor
{
public:
    LayerProcessor(
        Action action,
        ObjectsCache& cache,
        const EditorConfig& editorConfig,
        const ImportConfig& importConfig,
        const std::string& layerName);

    void findAndPrepareCategories(OGRLayer& layer);

    Objects prepare(const OGRFeaturePtr& ogrFeature) const;

private:
    std::string featureCategoryId(const OGRFeaturePtr& ogrFeature) const;

    void createFeatureProcessor(
        const std::string& categoryId,
        OGRFeatureDefn* featureDefn);

    const FeatureProcessor& featureProcessor(const std::string& categoryId) const;

    void makeFieldProcessor(
        FeatureProcessor& processor,
        OGRFieldDefn* fieldDefn) const;

    void validateFields(
        const FeatureProcessor& processor,
        OGRFeatureDefn* featureDefn) const;

    void makeGeometryProcessorForAddAction(
        FeatureProcessor& processor,
        OGRFeatureDefn* featureDefn) const;

    void makeGeometryProcessorForEditAction(
        FeatureProcessor& processor,
        OGRFeatureDefn* featureDefn) const;

    void makeGeometryProcessorForPointAd(
        FeatureProcessor& processor,
        OGRFeatureDefn* featureDefn) const;

    void makeGeometryProcessorForContourObjects(
        FeatureProcessor& processor,
        OGRFeatureDefn* featureDefn) const;

    void makeGeometryProcessorForContourElementObjects(
        FeatureProcessor& processor,
        OGRFeatureDefn* featureDefn) const;

    ObjectsCache& cache_;
    const EditorConfig& editorConfig_;
    const ImportConfig& importConfig_;
    Action action_;
    WorkMode mode_;
    RubricsMatcher rubricsMatcher_;
    std::string layerName_;
    std::optional<LevelId> masterIndoorLevelId_;

    std::map<std::string, FeatureProcessor> categoryIdToFeatureProcessor_;
};

} // importer
} // wiki
} // maps
