#include "object.h"

#include <maps/libs/log8/include/log8.h>

namespace maps {
namespace wiki {
namespace importer {

ObjectsCache::~ObjectsCache()
{
    for (const auto& object : objects_) {
        object->clear();
    }
}

ObjectPtr ObjectsCache::makeObject(ID id, TObjectId dbId, const cfg::Category& category)
{
    auto object = ObjectPtr(new Object{std::move(id), dbId, category, *this});
    objects_.push_back(object);
    idToObject_.emplace(object->id(), object);
    return object;
}

ObjectPtr ObjectsCache::getOrCreateJunction(ID id, const OGRPoint& point, const cfg::Category& category)
{
    auto it = junctions_.find(point);
    if (it != junctions_.end()) {
        return it->second;
    }
    auto object = junctions_.emplace(point, makeObject(std::move(id), EMPTY_DB_ID, category)).first->second;
    object->setWkb(getWkb(&point));
    return object;
}

ObjectPtr ObjectsCache::getOrCreateObject(ID id, const cfg::Category& category)
{
    auto object = getObject(id, category);
    if (object) {
        return object;
    }
    return makeObject(std::move(id), EMPTY_DB_ID, category);
}

ObjectPtr ObjectsCache::getObject(ID id, const cfg::Category& category) const
{
    Objects candidates;

    auto [begin, end] = idToObject_.equal_range(id);
    for (auto it = begin; it != end; ++it) {
        if (it->second->category().id() == category.id()) {
            candidates.push_back(it->second);
        }
    }

    REQUIRE(candidates.size() <= 1, "There are too many objects with id " << id);

    if (!candidates.empty()) {
        return candidates.front();
    }
    return ObjectPtr();
}

Object::Object(ID id, TObjectId dbId, const cfg::Category& category, ObjectsCache& cache)
    : cache_(cache)
    , category_(category)
    , id_(std::move(id))
    , state_(dbId == EMPTY_DB_ID ? ObjectState::New : ObjectState::Existing)
    , isModified_(false)
    , isSkipped_(false)
    , dbId_(dbId)
{
    attributesToAdd_.emplace("cat:" + category_.id(), "1");
}

void Object::addAttribute(std::string id, std::string value)
{
    isModified_ = true;
    attributesToAdd_.emplace(std::move(id), std::move(value));
}

void Object::deleteAttribute(std::string id)
{
    isModified_ = true;
    attributesToDelete_.insert(std::move(id));
}

Relation& Object::addSlave(std::string roleId, const cfg::Category& slaveCategory)
{
    isModified_ = true;
    slaveRelations_.emplace_back(
        Relation{std::move(roleId),
            cache_.makeObject(id_, EMPTY_DB_ID, slaveCategory)});
    return slaveRelations_.back();
}

Relation& Object::addSlave(std::string roleId, const ObjectPtr& slave)
{
    isModified_ = true;
    slaveRelations_.emplace_back(Relation{std::move(roleId), slave});
    return slaveRelations_.back();
}

void Object::deleteSlave(DeleteRelation relation)
{
    isModified_ = true;
    deleteSlaveRelations_.push_back(std::move(relation));
}

Relation& Object::addMaster(std::string roleId, TObjectId masterDbId, const cfg::Category& masterCategory)
{
    isModified_ = true;
    masterRelations_.emplace_back(
        Relation{std::move(roleId),
            cache_.makeObject(id_, masterDbId, masterCategory)});
    return masterRelations_.back();
}

Relation& Object::addMaster(std::string roleId, const ObjectPtr& master)
{
    isModified_ = true;
    masterRelations_.emplace_back(Relation{std::move(roleId), master});
    return masterRelations_.back();
}

void
Object::deleteMaster(const std::string& roleId, TObjectId masterDbId)
{
    isModified_ = true;
    mastersToDelete_.emplace_back(DeleteMasterRelation {roleId, masterDbId});
}

void Object::addFutureMaster(std::string roleId, ID masterId)
{
    isModified_ = true;
    futureMasterRelations_.emplace_back(FutureRelation{std::move(roleId), std::move(masterId)});
}

void Object::clear()
{
    slaveRelations_.clear();
    masterRelations_.clear();
}

} // importer
} // wiki
} // maps
