#include <maps/wikimap/mapspro/services/tasks_misc/src/import_worker/tests/helpers.h>
#include <maps/wikimap/mapspro/services/tasks_misc/src/import_worker/lib/gdal_helpers.h>
#include <maps/wikimap/mapspro/services/tasks_misc/src/import_worker/lib/json_helpers.h>
#include <maps/wikimap/mapspro/services/tasks_misc/src/import_worker/lib/message_reporter.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>

#include <filesystem>
#include <sstream>

namespace fs = std::filesystem;

namespace maps {
namespace wiki {
namespace importer {
namespace test {

namespace {

void generateFakeDbIds(const ObjectsCache& cache)
{
    TObjectId dbId = EMPTY_DB_ID;

    for (const auto& object : cache.objects()) {
        if (object->dbId() == EMPTY_DB_ID) {
            object->setDbId(++dbId, ObjectState::New);
        }
    }
}

Objects processObjectGraph(const ObjectsCache& cache)
{
    Objects objectsToAdd;

    for (const auto& object : cache.objects()) {
        if (object->state() == ObjectState::New) {
            objectsToAdd.push_back(object);
        }
    }

    return objectsToAdd;
}

void badTest(const fs::path& dataDir, size_t expectedErrorSize)
{
    EditorConfig editorConfig(EDITOR_CONFIG_PATH);
    ImportConfig importConfig;
    MessageReporter messageReporter;
    ObjectsCache cache;
    gdal2objects(dataDir, Action::Add, cache, editorConfig, importConfig, messageReporter);
    UNIT_ASSERT_VALUES_EQUAL(messageReporter.size(), expectedErrorSize);
}

json::Value goodTest(const fs::path& dataDir, const StringMap& globalAttrs)
{
    EditorConfig editorConfig(EDITOR_CONFIG_PATH);
    ImportConfig importConfig;
    MessageReporter messageReporter;
    ObjectsCache cache;

    gdal2objects(dataDir, Action::Add, cache, editorConfig, importConfig, messageReporter);
    generateFakeDbIds(cache);
    auto objectsToAdd = processObjectGraph(cache);

    std::ostringstream oss;
    objects2json(objectsToAdd, oss, globalAttrs, messageReporter);

    checkMessageReporter(messageReporter);

    return json::Value::fromString(oss.str());
}

} // namespace

Y_UNIT_TEST_SUITE(json) {

Y_UNIT_TEST(test_gdal2json_empty_dir)
{
    badTest(dataPath("empty_dir"), 1);
}

Y_UNIT_TEST(test_gdal2json_wrong_sref)
{
    badTest(dataPath("shape_mercator_sref"), 1);
}

Y_UNIT_TEST(test_gdal2json_geojson_bad_category)
{
    badTest(dataPath("geojson_bad_category"), 1);
}

Y_UNIT_TEST(test_gdal2json_geojson_bad_geometry)
{
    badTest(dataPath("geojson_bad_geometry"), 1);
}

Y_UNIT_TEST(test_gdal2json_geojson_bad_geometry_size)
{
    badTest(dataPath("geojson_bad_geometry_size"), 1);
}

Y_UNIT_TEST(test_gdal2json_geojson_bad_attribute)
{
    badTest(dataPath("geojson_bad_attribute"), 1);
}

Y_UNIT_TEST(test_gdal2json_geojson_bad_attribute_value)
{
    badTest(dataPath("geojson_bad_attribute_value"), 1);
}

Y_UNIT_TEST(test_gdal2json_geojson_bad_locale)
{
    badTest(dataPath("geojson_bad_locale"), 1);
}

Y_UNIT_TEST(test_gdal2json)
{
    const StringMap globalAttrs = {
        {"source", "import"},
        {"task_id", "test task"}
    };

    auto json = goodTest(dataPath("shape"), globalAttrs);

    // Check global attributes
    UNIT_ASSERT(json.hasField("attributes"));
    for (auto&& ga: globalAttrs) {
        UNIT_ASSERT_STRINGS_EQUAL(
            json["attributes"][ga.first].as<std::string>(),
            ga.second
        );
    }

    // Check objects
    UNIT_ASSERT(json.hasField("objects"));
    const auto& objects = json["objects"];
    UNIT_ASSERT_VALUES_EQUAL(objects.size(), 2);

    UNIT_ASSERT(objects.hasField("1"));
    UNIT_ASSERT(objects["1"].hasField("attributes"));
    UNIT_ASSERT(objects["1"].hasField("geometry"));

    UNIT_ASSERT(objects.hasField("1"));
    UNIT_ASSERT(objects["2"].hasField("attributes"));
    UNIT_ASSERT(objects["2"].hasField("geometry"));

    UNIT_ASSERT_STRINGS_EQUAL(
            objects["1"]["attributes"]["cat:aoi"].as<std::string>(),
            "1");
    UNIT_ASSERT_STRINGS_EQUAL(
            objects["1"]["attributes"]["aoi:name"].as<std::string>(),
            "cleanspecial_test");
    UNIT_ASSERT_STRINGS_EQUAL(
            objects["1"]["attributes"]["aoi:type"].as<std::string>(),
            "2");
    UNIT_ASSERT_STRINGS_EQUAL(
            objects["1"]["geometry"]["type"].as<std::string>(),
            "Polygon");
    UNIT_ASSERT(objects["1"]["geometry"].hasField("coordinates"));

    UNIT_ASSERT_STRINGS_EQUAL(
            objects["2"]["attributes"]["cat:aoi"].as<std::string>(),
            "1");
    UNIT_ASSERT_STRINGS_EQUAL(
            objects["2"]["attributes"]["aoi:name"].as<std::string>(),
            "miplot_test");
    UNIT_ASSERT_STRINGS_EQUAL(
            objects["2"]["attributes"]["aoi:type"].as<std::string>(),
            "2");
    UNIT_ASSERT_STRINGS_EQUAL(
            objects["2"]["geometry"]["type"].as<std::string>(),
            "Polygon");
    UNIT_ASSERT(objects["2"]["geometry"].hasField("coordinates"));
}

Y_UNIT_TEST(test_gdal2json_geojson_poi)
{
    auto json = goodTest(dataPath("geojson_poi"), {});

    UNIT_ASSERT(json.hasField("objects"));
    const auto& objects = json["objects"];
    UNIT_ASSERT_VALUES_EQUAL(objects.size(), 7); // 2 poi, 5 names

    UNIT_ASSERT(objects.hasField("1"));
    UNIT_ASSERT(objects["1"].hasField("attributes"));
    UNIT_ASSERT(objects["1"].hasField("geometry"));
    UNIT_ASSERT(objects["1"].hasField("relations"));

    const auto& relations = objects["1"]["relations"];
    UNIT_ASSERT_VALUES_EQUAL(relations.size(), 5);

    UNIT_ASSERT(relations[0].hasField("attributes"));
    UNIT_ASSERT(relations[0].hasField("slave"));
    UNIT_ASSERT(relations[1].hasField("attributes"));
    UNIT_ASSERT(relations[1].hasField("slave"));
    UNIT_ASSERT(relations[2].hasField("attributes"));
    UNIT_ASSERT(relations[2].hasField("slave"));
    UNIT_ASSERT(relations[3].hasField("attributes"));
    UNIT_ASSERT(relations[3].hasField("slave"));

    UNIT_ASSERT(objects.hasField("2"));
    UNIT_ASSERT_STRINGS_EQUAL(
        objects["2"]["attributes"]["poi_nm:lang"].as<std::string>(),
        "ru");

    UNIT_ASSERT(objects.hasField("3"));
    UNIT_ASSERT_STRINGS_EQUAL(
        objects["3"]["attributes"]["poi_nm:lang"].as<std::string>(),
        "ru");

    UNIT_ASSERT(objects.hasField("4"));
    UNIT_ASSERT_STRINGS_EQUAL(
        objects["4"]["attributes"]["poi_nm:lang"].as<std::string>(),
        "ru");

    UNIT_ASSERT(objects.hasField("5"));
    UNIT_ASSERT_STRINGS_EQUAL(
        objects["5"]["attributes"]["poi_nm:lang"].as<std::string>(),
        "en");

    UNIT_ASSERT(objects.hasField("6"));
    UNIT_ASSERT_STRINGS_EQUAL(
        objects["6"]["attributes"]["poi_nm:lang"].as<std::string>(),
        "sr-Latn");

    UNIT_ASSERT(objects.hasField("7"));
    UNIT_ASSERT(objects["7"].hasField("attributes"));
    UNIT_ASSERT(objects["7"].hasField("geometry"));
    UNIT_ASSERT(!objects["7"].hasField("relations"));
}

Y_UNIT_TEST(test_gdal2json_geojson_ad)
{
    auto json = goodTest(dataPath("geojson_ad"), {});

    UNIT_ASSERT(json.hasField("objects"));
    const auto& objects = json["objects"];
    UNIT_ASSERT_VALUES_EQUAL(objects.size(), 7); //2 ad, 2 centers, 3 names

    //first ad
    UNIT_ASSERT(objects.hasField("1"));
    UNIT_ASSERT(objects["1"].hasField("attributes"));
    UNIT_ASSERT(!objects["1"].hasField("geometry")); //ad has no geometry
    UNIT_ASSERT(objects["1"].hasField("relations"));

    const auto& relations = objects["1"]["relations"];
    UNIT_ASSERT_VALUES_EQUAL(relations.size(), 3); //2 names, 1 center

    UNIT_ASSERT(relations[0].hasField("attributes"));
    UNIT_ASSERT(relations[0].hasField("slave"));
    UNIT_ASSERT(relations[1].hasField("attributes"));
    UNIT_ASSERT(relations[1].hasField("slave"));

    UNIT_ASSERT(objects.hasField("2")); //ad center

    UNIT_ASSERT(objects.hasField("3"));
    UNIT_ASSERT_STRINGS_EQUAL(
        objects["3"]["attributes"]["ad_nm:lang"].as<std::string>(),
        "ru");
    UNIT_ASSERT(objects["3"]["attributes"].hasField("ad_nm:is_local"));

    UNIT_ASSERT(objects.hasField("4"));
    UNIT_ASSERT_STRINGS_EQUAL(
        objects["4"]["attributes"]["ad_nm:lang"].as<std::string>(),
        "ru");
    UNIT_ASSERT(objects["4"]["attributes"].hasField("ad_nm:is_local"));

    //second ad
    UNIT_ASSERT(objects.hasField("5"));
    UNIT_ASSERT(objects["5"].hasField("attributes"));
    UNIT_ASSERT(!objects["5"].hasField("geometry"));
    UNIT_ASSERT(objects["5"].hasField("relations"));

    UNIT_ASSERT(objects.hasField("6")); //ad center

    UNIT_ASSERT(objects.hasField("7"));
    UNIT_ASSERT_STRINGS_EQUAL(
        objects["7"]["attributes"]["ad_nm:lang"].as<std::string>(),
        "ru");
    UNIT_ASSERT(!objects["7"]["attributes"].hasField("ad_nm:is_local"));
}

Y_UNIT_TEST(test_gdal2json_geojson_ad_multipolygon)
{
    auto json = goodTest(dataPath("geojson_ad_multipolygon"), {});

    UNIT_ASSERT(json.hasField("objects"));
    const auto& objects = json["objects"];
    //8051 vertices are recursively splited into 32 parts
    //with limits of 500 vertices and 2000km
    //1 ad, 1 ad_cnt, 1 ad_fc, 32 ad_el, 32 ad_jc
    UNIT_ASSERT_VALUES_EQUAL(objects.size(), 67);
}

Y_UNIT_TEST(test_gdal2json_geojson_ad_el)
{
    auto json = goodTest(dataPath("geojson_ad_el"), {});

    UNIT_ASSERT(json.hasField("objects"));
    const auto& objects = json["objects"];
    //7813 vertices are recursively splited into 32 parts
    //with limits of 500 vertices and 2000km
    //32 ad_el, 33 ad_jc
    UNIT_ASSERT_VALUES_EQUAL(objects.size(), 65);
}

Y_UNIT_TEST(test_gdal2json_geojson_ad_el_long)
{
    auto json = goodTest(dataPath("geojson_ad_el_long"), {});

    UNIT_ASSERT(json.hasField("objects"));
    const auto& objects = json["objects"];
    //Line of length 2679237m is splited into 3 parts
    //3 ad_el, 4 ad_jc
    UNIT_ASSERT_VALUES_EQUAL(objects.size(), 7);
}

Y_UNIT_TEST(test_gdal2json_shape_ad_el_connected)
{
    auto json = goodTest(dataPath("shape_ad_el_connected"), {});

    UNIT_ASSERT(json.hasField("objects"));
    const auto& objects = json["objects"];
    //6 ad_el connected with 4 common ad_jc
    UNIT_ASSERT_VALUES_EQUAL(objects.size(), 10);
}

Y_UNIT_TEST(test_gdal2json_mif_addr)
{
    auto json = goodTest(dataPath("mif_addr"), {});

    UNIT_ASSERT(json.hasField("objects"));
    const auto& objects = json["objects"];
    UNIT_ASSERT_VALUES_EQUAL(objects.size(), 4);

    UNIT_ASSERT(objects.hasField("1"));
    UNIT_ASSERT(objects["1"].hasField("attributes"));
    UNIT_ASSERT(objects["1"].hasField("geometry"));
    UNIT_ASSERT(objects["1"].hasField("relations"));

    const auto& relations = objects["1"]["relations"];
    UNIT_ASSERT_VALUES_EQUAL(relations.size(), 2);

    UNIT_ASSERT(relations[0].hasField("attributes"));
    UNIT_ASSERT(relations[0].hasField("slave"));

    UNIT_ASSERT(objects.hasField("2")); //addr_nm

    UNIT_ASSERT(objects.hasField("3"));
    UNIT_ASSERT(objects["3"].hasField("attributes"));
    UNIT_ASSERT(objects["3"].hasField("geometry"));
    UNIT_ASSERT(objects["3"].hasField("relations"));

    UNIT_ASSERT(objects.hasField("4")); //addr_nm
}

Y_UNIT_TEST(test_gdal2json_rd)
{
    auto json = goodTest(dataPath("dbf_rd"), {});

    UNIT_ASSERT(json.hasField("objects"));
    const auto& objects = json["objects"];
    UNIT_ASSERT_VALUES_EQUAL(objects.size(), 4); //2 rd and 2 name

    UNIT_ASSERT(objects.hasField("1"));
    UNIT_ASSERT(objects["1"].hasField("attributes"));
    UNIT_ASSERT(!objects["1"].hasField("geometry"));
    UNIT_ASSERT(objects["1"].hasField("relations"));

    UNIT_ASSERT(objects.hasField("2"));
    UNIT_ASSERT_STRINGS_EQUAL(
        objects["2"]["attributes"]["rd_nm:lang"].as<std::string>(),
        "ru");

    UNIT_ASSERT(objects.hasField("3"));
    UNIT_ASSERT(objects["3"].hasField("attributes"));
    UNIT_ASSERT(!objects["3"].hasField("geometry"));
    UNIT_ASSERT(objects["3"].hasField("relations"));

    UNIT_ASSERT(objects.hasField("4"));
    UNIT_ASSERT_STRINGS_EQUAL(
        objects["4"]["attributes"]["rd_nm:lang"].as<std::string>(),
        "ru");
}

Y_UNIT_TEST(test_gdal2json_geojson_bld)
{
    auto json = goodTest(dataPath("geojson_bld"), {});

    UNIT_ASSERT(json.hasField("objects"));
    const auto& objects = json["objects"];
    UNIT_ASSERT_VALUES_EQUAL(objects.size(), 1);
}

} // Y_UNIT_TEST_SUITE

} // test
} // importer
} // wiki
} // maps
