#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/tools/grinder/worker/include/api.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/mongo/include/init.h>
#include <yandex/maps/shell_cmd.h>
#include <yandex/maps/shellcmd/logging_ostream.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/common/retry_duration.h>
#include <yandex/maps/wiki/tasks/task_logger.h>
#include <yandex/maps/wiki/revision/branch_manager.h>

#include <memory>
#include <optional>
#include <string>
#include <utility>

namespace common = maps::wiki::common;
namespace mlog = maps::log8;
namespace rev = maps::wiki::revision;
namespace worker = maps::grinder::worker;

namespace {

constexpr size_t WORKER_CONCURRENCY = 1;

} // namespace

namespace maps::wiki::vrevisions_refresh {

const std::string EDITOR_TOOL_PATH =
    "/usr/lib/yandex/maps/wiki/bin/wiki-editor-tool";
constexpr int EDITOR_TOOL_EXIT_CODE_LOCKED = 1;
constexpr rev::Branch::LockId GLOBAL_BRANCH_LOCK_ID = 0;

class EditorToolError : public maps::Exception
{
    using maps::Exception::Exception;
};

enum class Action
{
    Refresh,
    CreateStable
};

Action actionFromString(const std::string& value)
{
    static const std::map<std::string, Action> nameToAction = {
        {"refresh", Action::Refresh},
        {"create-stable", Action::CreateStable}
    };
    const auto itr = nameToAction.find(value);
    REQUIRE(itr != nameToAction.end(), "Invalid action: " + value);
    return itr->second;
}

std::string actionToString(Action action)
{
    switch (action) {
        case Action::Refresh: return "refresh";
        case Action::CreateStable: return "create-stable";
    }
    throw maps::RuntimeError()
            << "Invalid action value: " << static_cast<int>(action);
}

std::ostream& operator<<(std::ostream& os, Action action)
{
    return os << actionToString(action);
}

std::istream& operator>>(std::istream& is, Action& action)
{
    std::string str;
    is >> str;
    action = actionFromString(str);
    return is;
}

struct TaskParams
{
    explicit TaskParams(const worker::Task& task)
    {
        const auto& args = task.args();
        taskId = args["taskId"].as<uint64_t>();
        branchId = args["branchId"].as<rev::DBID>();
        action = actionFromString(args["action"].as<std::string>());
        stages = args["stages"].as<std::string>();
        categoryGroups = args["categoryGroups"].as<std::string>();
        isExclusive = args["isExclusive"].as<int>();
        uid = args["uid"].as<uint64_t>();
    }

    uint64_t taskId;
    rev::DBID branchId;
    Action action;
    std::string stages;
    std::string categoryGroups;
    bool isExclusive;
    uint64_t uid;
};

std::ostream& operator<< (std::ostream& out, const TaskParams& taskParams)
{
    out << "taskId: " << taskParams.taskId
        << ", branch: " << taskParams.branchId
        << ", action: " << taskParams.action
        << ", stages: " << taskParams.stages
        << ", category-groups: " << taskParams.categoryGroups
        << ", is-exclusive: " << (taskParams.isExclusive ? "true" : "false");

    return out;
}

std::string getVrevisionsRefreshCmd(
    const TaskParams& taskParams,
    const std::string& configPath)
{
    std::ostringstream cmd;

    cmd << EDITOR_TOOL_PATH
        << " --log-level=info"
        << " --all-objects=1"
        << " --branch=" << taskParams.branchId
        << " --action=" << taskParams.action
        << " --user-id=" << taskParams.uid
        << " --stages=" << taskParams.stages
        << " --category-groups=" << taskParams.categoryGroups
        << " --set-progress-state=" << taskParams.isExclusive
        << " --branch-exclusive-lock=" << taskParams.isExclusive
        << " --config=" << configPath;

    return cmd.str();
}

std::optional<rev::BranchState> setBranchState(
    pgpool3::Pool& pgPool,
    rev::DBID branchId,
    rev::BranchState targetBranchState)
{
    return common::retryDuration([&]() -> std::optional<rev::BranchState> {
        auto txn = pgPool.masterWriteableTransaction();
        rev::BranchManager branchManager(*txn);
        auto branch = branchManager.load(branchId);
        auto prevBranchState = branch.state();
        if (branch.state() != targetBranchState) {
            if (!branch.tryLock(*txn, GLOBAL_BRANCH_LOCK_ID,
                                revision::Branch::LockType::Shared)) {
                return std::nullopt;
            }
            branch = branchManager.load(branchId);
            prevBranchState = branch.state();
            if (branch.state() != targetBranchState) {
                if (!branch.setState(*txn, targetBranchState)) {
                    return std::nullopt; // concurrent change without lock?
                }
                txn->commit();
            }
        }
        return prevBranchState;
    });
}

std::optional<rev::DBID> branchIdByType(
    pgpool3::Pool& pgPool,
    rev::BranchType branchType)
{
    if (branchType == rev::BranchType::Trunk) {
        return rev::TRUNK_BRANCH_ID;
    }

    auto branches = common::retryDuration([&] {
        auto txn = pgPool.masterReadOnlyTransaction();
        return rev::BranchManager(*txn).load({{branchType, 1}});
    });

    if (branches.empty()) {
        return std::nullopt;
    }
    return branches.begin()->id();
}

/// Fixes damage done by failed 'vrevisions refresh' operation.
/// - If the action is 'create stable', set state of a stable branch
///   to Unavailable and state of approved branch to Normal
/// - Otherwise set \p branch state to unavailable
void unprogressBranch(pgpool3::Pool& pgPool, rev::DBID branchId, Action action)
{
    WARN() << "Unprogress changes done on the branch";
    std::optional<rev::DBID> optBranchId = branchId;
    if (action == Action::CreateStable) {
        auto approvedBranchId = branchIdByType(pgPool, rev::BranchType::Approved);
        REQUIRE(approvedBranchId, "Approved branch does not exist");
        setBranchState(pgPool, *approvedBranchId, rev::BranchState::Normal);
        optBranchId = branchIdByType(pgPool, rev::BranchType::Stable);
    }
    if (optBranchId) {
        setBranchState(pgPool, *optBranchId, rev::BranchState::Unavailable);
    }
}

void runVrevisionsRefresh(
    pgpool3::Pool& pgPool,
    const std::string& configPath,
    const worker::Task& task)
{
    TaskParams taskParams(task);
    tasks::TaskPgLogger logger(pgPool, taskParams.taskId);
    logger.logInfo() << "Task started. Grinder task id: " << task.id();
    INFO() << "Received task: " << taskParams.taskId
           << ". Grinder task id: " << task.id();

    bool branchLocked = false;

    auto cleanup = [&]() {
        if (taskParams.isExclusive && !branchLocked) {
            unprogressBranch(pgPool, taskParams.branchId, taskParams.action);
        }
    };

    try {
        logger.logInfo() << "Task parameters: " << taskParams;

        shell::stream::LoggingOutputStream loggedOut(
            [](const std::string& s){
                INFO() << "stdout: " << s;
            });
        shell::stream::LoggingOutputStream loggedErr(
            [](const std::string& s){
                INFO() << "stderr: " << s;
            });

        shell::ShellCmd shellCmd(
            getVrevisionsRefreshCmd(taskParams, configPath), loggedOut, loggedErr);
        auto exitCode = shellCmd.run();

        if (exitCode) {
            if (exitCode == EDITOR_TOOL_EXIT_CODE_LOCKED) {
                branchLocked = true;
                ERROR() << "Branch locked: already in progress";
                throw EditorToolError() << "Branch " << taskParams.branchId
                    << ": already in progress mode";
            } else {
                ERROR() << "Editor tool error, exit code: " << exitCode;
                throw EditorToolError()
                    << "Editor tool error, exit code: " << exitCode;
            }
        }
    } catch (const std::exception& e) {
        logger.logError() << "Task failed: " << e.what();
        cleanup();
        throw;
    } catch (...) {
        cleanup();
        throw;
    }
    INFO() << "Task " << taskParams.taskId << " finished";
    logger.logInfo() << "Task finished";
}

} // maps::wiki::vrevisions_refresh

int main(int argc, char* argv[]) try {
    maps::cmdline::Parser parser;
    auto workerConfig = parser.string("config")
        .help("path to worker configuration");
    auto syslogTag = parser.string("syslog-tag")
        .help("redirect log output to syslog with given tag");
    auto grinderConfig = parser.string("grinder-config")
        .help("path to grinder configuration file");

    parser.parse(argc, argv);

    if (syslogTag.defined()) {
        mlog::setBackend(mlog::toSyslog(syslogTag));
    }

    maps::mongo::init();

    std::unique_ptr<common::ExtendedXmlDoc> configDocPtr;
    if (workerConfig.defined()) {
        configDocPtr.reset(new common::ExtendedXmlDoc(workerConfig));
    } else {
        configDocPtr = common::loadDefaultConfig();
    }

    worker::Options workerOpts;

    if (grinderConfig.defined()) {
        workerOpts.setConfigLocation(grinderConfig);
    }

    const std::string configPath = configDocPtr->sourcePaths()[0];

    common::PoolHolder dbHolder(*configDocPtr, "core", "grinder");
    auto& pgPool = dbHolder.pool();

    using namespace maps::wiki::vrevisions_refresh;

    workerOpts.on("vrevisions_refresh",
            [&](const worker::Task& task) {
                    runVrevisionsRefresh(pgPool, configPath, task);
            })
        .setConcurrencyLevel(WORKER_CONCURRENCY);

    worker::run(std::move(workerOpts));

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    ERROR() << "Worker failed: " << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    ERROR() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
