#include "acl_roles_dumper.h"

#include <maps/libs/chrono/include/time_point.h>
#include <yandex/maps/wiki/common/tskv_logger.h>

#include <fstream>
#include <string>
#include <vector>

namespace maps::wiki::acl_roles_dumper {

namespace {

using Uid = std::uint64_t;
using UserRolePair = std::pair<Uid, std::string>;
using UserRolePairs = std::vector<UserRolePair>;
using Messages = std::vector<common::TskvMessage>;

namespace tskv::key {
const std::string PUID = "puid";
const std::string ROLE = "role";
const std::string UNIXTIME = "unixtime";
} // namespace tskv::key


common::TskvMessage
message(
    const std::string& unixtime,
    const Uid uid,
    const std::string& role)
{
    common::TskvMessage message;

    message.setParam(tskv::key::UNIXTIME, unixtime);
    message.setParam(tskv::key::PUID, std::to_string(uid));
    message.setParam(tskv::key::ROLE, role);

    return message;
}


UserRolePairs
getUserRolePairs(pqxx::transaction_base& txn)
{
    // Almost everyone has the `common` role, so do not dump it.
    const auto userToRoleQuery =
        // Roles assigned on users
        "SELECT acl.user.uid, acl.role.name\n"
        "FROM acl.user\n"
        "JOIN acl.policy ON acl.user.id = acl.policy.agent_id\n"
        "JOIN acl.role ON acl.policy.role_id = acl.role.id\n"
        "WHERE acl.user.status = 'active'\n"
        "    AND acl.role.name <> 'common'\n"

        "UNION DISTINCT\n"

        // Roles assigned on groups
        "SELECT acl.user.uid, acl.role.name\n"
        "FROM acl.user\n"
        "JOIN acl.group_user ON acl.user.id = acl.group_user.user_id\n"
        "JOIN acl.policy ON acl.group_user.group_id = acl.policy.agent_id\n"
        "JOIN acl.role ON acl.policy.role_id = acl.role.id\n"
        "WHERE acl.user.status = 'active'\n"
        "    AND acl.role.name <> 'common'\n";

    UserRolePairs result;
    const auto rows = txn.exec(userToRoleQuery);
    result.reserve(rows.size());

    for (const auto& row: rows) {
        result.emplace_back(row[0].as<Uid>(), row[1].as<std::string>());
    }
    return result;
}


Messages
userRolePairsToMessages(const UserRolePairs& userRolePairs)
{
    const auto now = std::to_string(chrono::sinceEpoch<std::chrono::seconds>());

    Messages result;
    result.reserve(userRolePairs.size());
    for (auto&& [uid, role]: userRolePairs) {
        result.emplace_back(message(now, uid, role));
    }
    return result;
}


void
dump(
    const Messages& messages,
    const std::string& fileName)
{
    std::ofstream file(fileName, std::ios_base::trunc);
    REQUIRE(file, "Can't open file '" << fileName << "'");

    for (const auto& message: messages) {
        file << message.toString() << "\n";
    }
}

} // namespace


void
dump(
    pqxx::transaction_base& txn,
    const std::string& dumpFileName)
{
    dump(
        userRolePairsToMessages(getUserRolePairs(txn)),
        dumpFileName
    );
}

} // namespace maps::wiki::acl_roles_dumper
