#include <maps/wikimap/mapspro/libs/revision_meta/include/approved_queue.h>

#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <iostream>

namespace common = maps::wiki::common;

namespace maps::wiki {

const std::string PREAPPROVED_QUEUE_SIZE = "preapproved-queue-size";
const std::string APPROVED_QUEUE_SIZE = "approved-queue-size";
const std::string LABELS_QUEUE_SIZE = "labels-queue-size";
const std::string BBOXES_QUEUE_SIZE = "bboxes-queue-size";
const std::string PREAPPROVED_QUEUE_AGE = "preapproved-queue-age";

const std::vector<std::string> MONITORING_TYPES = {
    PREAPPROVED_QUEUE_SIZE,
    APPROVED_QUEUE_SIZE,
    LABELS_QUEUE_SIZE,
    BBOXES_QUEUE_SIZE,
    PREAPPROVED_QUEUE_AGE
};

const std::map<std::string, revision_meta::ApprovedQueueMode> MONITORING_TYPE_TO_MODE = {
    { PREAPPROVED_QUEUE_SIZE, revision_meta::ApprovedQueueMode::PreApprove },
    { APPROVED_QUEUE_SIZE, revision_meta::ApprovedQueueMode::ViewAttrs },
    { LABELS_QUEUE_SIZE, revision_meta::ApprovedQueueMode::Labels },
    { BBOXES_QUEUE_SIZE, revision_meta::ApprovedQueueMode::Bboxes },
};

void checkQueueSize(
    pgpool3::Pool& pool,
    revision_meta::ApprovedQueueMode mode,
    size_t warnLimit,
    size_t critLimit)
{
    auto txn = pool.masterReadOnlyTransaction();
    auto size = revision_meta::ApprovedQueue(*txn, mode).size();

    if (size > critLimit) {
        std::cout << "2;Error: commit queue size " << size << std::endl;
    } else if (size > warnLimit) {
        std::cout << "1;Warning: commit queue size " << size << std::endl;
    } else {
        std::cout << "0;OK: commit queue size " << size << std::endl;
    }
}

void checkPreApprovedQueueAge(
    pgpool3::Pool& pool,
    size_t warnLimit,
    size_t critLimit)
{
    auto txn = pool.masterReadOnlyTransaction();
    const auto maxAge = static_cast<size_t>(
        revision_meta::PreApprovedQueue(*txn).oldestCommitAge().count()
    );

    if (maxAge > critLimit) {
        std::cout << "2;Error: oldest commit age " << maxAge << " seconds" << std::endl;
    } else if (maxAge > warnLimit) {
        std::cout << "1;Warning: oldest commit age " << maxAge << " seconds" << std::endl;
    } else {
        std::cout << "0;OK: oldest commit age " << maxAge << " seconds" << std::endl;
    }
}

} //namespace maps::wiki

int main(int argc, char* argv[])
try {
    maps::log8::setLevel(maps::log8::Level::FATAL);

    maps::cmdline::Parser parser;
    auto configPath = parser
        .file("config")
        .help("Path to services.xml");
    auto monitoringType = parser
        .string("type")
        .help("Monitoring type: " + common::join(maps::wiki::MONITORING_TYPES, '|'))
        .required();
    auto warnLimit = parser
        .size_t("warn")
        .help("Warning limit: commit count or the number of seconds of the oldest commit")
        .required();
    auto critLimit = parser
        .size_t("crit")
        .help("Critical limit: commit count or the number of seconds of the oldest commit")
        .required();
    parser.parse(argc, argv);

    auto configDocPtr =
        configPath.defined()
        ? std::make_unique<common::ExtendedXmlDoc>(configPath)
        : common::loadDefaultConfig();

    common::PoolHolder coreDbHolder(*configDocPtr, "core", "grinder");

    if (monitoringType == maps::wiki::PREAPPROVED_QUEUE_AGE) {
        maps::wiki::checkPreApprovedQueueAge(
            coreDbHolder.pool(),
            warnLimit,
            critLimit);
    } else {
        auto it = maps::wiki::MONITORING_TYPE_TO_MODE.find(monitoringType);
        if (it != maps::wiki::MONITORING_TYPE_TO_MODE.end()) {
            maps::wiki::checkQueueSize(
                coreDbHolder.pool(),
                it->second,
                warnLimit,
                critLimit);
        } else {
            std::cout << "2;Error: wrong monitoring type " << monitoringType;
        }
    }

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    std::cout << "2;Error: " << e.what() << std::endl;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    std::cout << "2;Error: " << e.what() << std::endl;
    return EXIT_FAILURE;
}
