#include "contour_object.h"
#include "magic_strings.h"
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/geolib/include/multipolygon.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/tile/include/const.h>
#include <maps/libs/tile/include/coord.h>
#include <maps/libs/tile/include/utils.h>
#include <yandex/maps/wiki/geom_tools/polygonal/builder.h>
#include <yandex/maps/wiki/geom_tools/polygonal/partition.h>

#include <boost/algorithm/string/predicate.hpp>
#include <string>

namespace maps {
namespace wiki {
namespace contours {
namespace {

constexpr size_t VERTICES_GROWTH_FACTOR = 2;

const revision_meta::TStringSet isPartOfRegionCategories {
    category::AD,
    category::AD_NEUTRAL
};

std::string getValueBySuffix
    ( revision_meta::TAttributesMap attrs
    , const std::string& suffix
    )
{
    for (const auto& attr: attrs) {
        if (boost::algorithm::ends_with(attr.first, ":" + suffix)) {
            return attr.second;
        }
    }
    return std::string();
}

template <typename T>
std::string toSqlValue(const T& val, pqxx::transaction_base& work)
{
    std::stringstream stream;
    stream.precision(12);
    stream << val;
    return work.esc(stream.str());
}

std::string toSqlValue
    ( const revision_meta::TAttributesMap& attrs
    , pqxx::transaction_base& work
    )
{
    std::stringstream stream;
    bool first = true;
    for (const auto& attr: attrs) {
        if (!first) {
            stream << ',';
        }
        first = false;
        stream << work.quote(attr.first) << ',' << work.quote(attr.second);
    }

    auto result = stream.str();
    if (result.empty()) {
        return std::string();
    }
    return "hstore(ARRAY[" + result + "])";
}

std::string toSqlValue(const geolib3::Polygon2& polygon, pqxx::transaction_base& work)
{
    std::stringstream stream;
    geolib3::WKB::write(polygon, stream);
    auto wkb = stream.str();
    REQUIRE(wkb.size(), "empty geometry");
    return "ST_GeomFromWkb('" + work.esc_raw(wkb) + "',3395)";
}

std::string toSqlStatement(const std::map<std::string, std::string>& namedValues)
{
    std::stringstream names;
    std::stringstream values;
    bool first = true;
    for (const auto& namedValue: namedValues) {
        if (!namedValue.second.empty()) {
            if (!first) {
                names << ',';
                values << ',';
            }
            first = false;
            names << namedValue.first;
            values << namedValue.second;
        }
    }
    std::stringstream result;
    result
        << "INSERT INTO " << table::CONTOUR_OBJECTS
        << "(" << names.str() << ") VALUES(" << values.str() << ");\n";
    return result.str();
}

class GeomToolsException : public Exception {};

/**
 * \brief compensated summation, significantly reduces the numerical error
 * \see http://en.wikipedia.org/wiki/Kahan_summation_algorithm
 */
double kahanSum(const std::vector<double>& vals)
{
    double result = 0.;
    double err = 0.;
    for (const auto& val: vals) {
        const double item = val - err;
        const double sum = result + item;
        err = (sum - result) - item;
        result = sum;
    }
    return result;
}

size_t getPointsNumber(const geom_tools::GeolibPolygonVector& polygons)
{
    size_t result = 0;
    for (const auto& polygon: polygons) {
        result += polygon.totalPointsNumber();
    }
    return result;
}

geom_tools::GeolibPolygonVector simplify
    ( const geom_tools::GeolibPolygonVector& polygons
    , Zoom zoom
    )
{
    geom_tools::GeolibPolygonVector result;
    result.reserve(polygons.size());
    const auto halfPixelSize = tile::zoomToResolution(zoom) / 2.;
    for (const auto& polygon: polygons) {
        result.push_back(geolib3::simplify(polygon, halfPixelSize));
    }
    return result;
}

} // namespace

void ContourObject::reset()
{
    objectId_ = 0;
    commitId_ = 0;
    categoryId_.clear();
    domainAttrs_.clear();
    serviceAttrs_.clear();
    polygons_.clear();
    areasCache_.clear();
    zoomRange_ = ZoomRange{1, 0}; // invalid
    pyramid_.clear();
}

void ContourObject::sync(const revision_meta::Object& obj)
{
    if (commitId_ < obj.commitId()) {
        commitId_ = obj.commitId();
    }
}

template <typename Functor>
auto ContourObject::geomToolsCall(Functor&& f) const try {
    return f();
} catch (const GeomToolsException&) {
    throw;
} catch (const std::exception& e) {
    throw GeomToolsException() << e.what();
}

geolib3::LinearRing2 ContourObject::getRing
    ( revision_meta::Snapshot& snapshot
    , revision_meta::TObjectId ringId
    , double tolerance
    )
{
    auto linestringIds = snapshot.slaves(ringId);
    geom_tools::WikiGeomPtrVector linestrings;
    linestrings.reserve(linestringIds.size());
    for (auto linestringId: linestringIds) {
        const auto& linestringObj = snapshot.existingObjectRevision(linestringId);
        sync(linestringObj);
        linestrings.push_back(&linestringObj.geom());
    }
    return geomToolsCall([&]{
        return geom_tools::LinearRingBuilder::build(linestrings, tolerance);
    });
}

geom_tools::GeolibPolygonVector ContourObject::getPolygons
    ( revision_meta::Snapshot& snapshot
    , double tolerance
    )
{
    auto ringIds = snapshot.slaves(objectId_);
    geom_tools::GeolibLinearRingVector shells;
    shells.reserve(ringIds.size());
    geom_tools::GeolibLinearRingVector holes;
    holes.reserve(ringIds.size());
    for (auto ringId: ringIds) {
        const auto& ringObj = snapshot.existingObjectRevision(ringId);
        sync(ringObj);
        if (getValueBySuffix(ringObj.attrs(), attr::IS_INTERIOR) == service_attrs::FLAG_TRUE) {
            holes.push_back(getRing(snapshot, ringId, tolerance));
        } else {
            shells.push_back(getRing(snapshot, ringId, tolerance));
        }
    }
    return geomToolsCall([&]{
        return geom_tools::PolygonBuilder::build
            (shells, holes, geom_tools::ValidateResult::Yes);
    });
}

ContourObject::Level ContourObject::makeLevel(Zoom zoom) const
{
    Level result;
    result.zoomRange.zmin = zoom;
    result.zoomRange.zmax = zoom;
    result.polygons = geomToolsCall([&]{ return simplify(polygons_, zoom); });
    result.originalAreas = areasCache_;
    result.pointsNumber = getPointsNumber(result.polygons);
    return result;
}

void
ContourObject::calculateServiceAttributes(revision_meta::Snapshot& snapshot)
{
    if (!isPartOfRegionCategories.count(categoryId_)) {
        return;
    }
    bool regionsFound = false;
    for (const auto masterId : snapshot.masters(objectId_)) {
        const auto& master = snapshot.existingObjectRevision(masterId);
        sync(master);
        if (master.categoryId() == category::REGION) {
            regionsFound = true;
            serviceAttrs_.insert(
                {
                    service_attrs::IS_PART_OF_REGION_PREFIX + std::to_string(masterId),
                    service_attrs::FLAG_TRUE
                });
        }
    }
    if (regionsFound) {
        serviceAttrs_.insert(
            {
                service_attrs::IS_PART_OF_REGION,
                service_attrs::FLAG_TRUE
            });
    }
}

ContourObject::Level ContourObject::splitLevel
    ( const ContourObject::Level& level
    , const Params& params
    ) const
{
    const auto halfTileSize
        = tile::zoomToResolution(level.zoomRange.zmax)
        * tile::TILE_SIZE / 2.
        ;
    Level result;
    result.zoomRange = level.zoomRange;
    result.pointsNumber = level.pointsNumber;
    for (size_t idx = 0; idx != level.polygons.size(); ++idx) {
        auto partition = geomToolsCall([&]{
            return geom_tools::partition
                ( { level.polygons[idx] }
                , params.minVerticesCount
                , halfTileSize
                , params.tolerance
                );
        });
        for (auto& polygon: partition) {
            result.polygons.push_back(std::move(polygon));
            result.originalAreas.push_back(level.originalAreas[idx]);
        }
    }
    return result;
}

void ContourObject::addLevel(const Level& level, const Params& params)
{
    pyramid_.push_back(splitLevel(level, params));
}

// make sure that pyramid doesn't contain holes
void ContourObject::correctPyramid()
{
    if (pyramid_.empty()) {
        return;
    }
    for (size_t i = 1; i < pyramid_.size(); ++i) {
        pyramid_[i - 1].zoomRange.zmin = pyramid_[i].zoomRange.zmax + 1;
    }
    pyramid_.back().zoomRange.zmin = zoomRange_.zmin;
}

void ContourObject::load
    ( revision_meta::Snapshot& snapshot
    , revision_meta::TObjectId id
    , const Params& params
    ) try
{
    reset();

    objectId_ = id;
    auto obj = snapshot.tryGetObjectRevision(id);
    if (!obj || obj->state() == revision_meta::ObjectState::Deleted) {
        DEBUG() << "deleted: " << id;
        return;
    }
    sync(*obj);
    categoryId_ = obj->categoryId();
    domainAttrs_ = obj->attrs();
    polygons_ = getPolygons(snapshot, params.tolerance);
    for (const auto& polygon: polygons_) {
        areasCache_.push_back(polygon.area());
    }
    zoomRange_ = GeneralizationLoader::zoomRange(categoryId_, kahanSum(areasCache_));
    calculateServiceAttributes(snapshot);
    addLevel(makeLevel(zoomRange_.zmax), params);
    for ( auto zoom = zoomRange_.zmax - 1, zoomEnd = zoomRange_.zmin - 1
        ; zoom > zoomEnd
        ; --zoom
        ) {
        const auto level = makeLevel(zoom);
        /**
         * Since the amount of points hasn't decreased dramatically
         * with zoom decrease, it's better to reuse the previous
         * (more detailed) partition, than to generate a new one.
         */
        if (pyramid_.back().pointsNumber > VERTICES_GROWTH_FACTOR * level.pointsNumber) {
            addLevel(level, params);
            --zoom; // Heuristic: reduce number of levels that are built
        }
        if (level.pointsNumber <= params.minVerticesCount) {
            break;
        }
    }
    correctPyramid();
} catch (const GeomToolsException& e) {
    correctPyramid();
    if (pyramid_.empty()) {
        WARN() << "object id = " << id << ": " << e.what();
        serviceAttrs_.insert({service_attrs::INVALID, service_attrs::FLAG_TRUE});
        serviceAttrs_.insert({service_attrs::ERROR, e.what()});
    }
}

std::vector<std::string> ContourObject::dumpToSql(pqxx::transaction_base& work) const
{
    std::vector<std::string> result;
    result.emplace_back(
        "DELETE FROM " + table::CONTOUR_OBJECTS +
        " WHERE " + column::OBJECT_ID + " = " + std::to_string(objectId_) + ";");

    if (isDeleted()) {
        return result;
    }

    if (pyramid_.empty()) { // invalid object
        result.emplace_back(toSqlStatement(
            { {column::COMMIT_ID, toSqlValue(commitId_, work)}
            , {column::DOMAIN_ATTRS, toSqlValue(domainAttrs_, work)}
            , {column::OBJECT_ID, toSqlValue(objectId_, work)}
            , {column::SERVICE_ATTRS, toSqlValue(serviceAttrs_, work)}
            , {column::ZMAX, toSqlValue(zoomRange_.zmax, work)}
            , {column::ZMIN, toSqlValue(zoomRange_.zmin, work)}
            }));
    } else {
        for (const auto& level: pyramid_) {
            for (size_t idx = 0; idx != level.polygons.size(); ++idx) {
                std::map<std::string, std::string> namedValues {
                      {column::AREA, toSqlValue(level.originalAreas[idx], work)}
                    , {column::COMMIT_ID, toSqlValue(commitId_, work)}
                    , {column::DOMAIN_ATTRS, toSqlValue(domainAttrs_, work)}
                    , {column::OBJECT_ID, toSqlValue(objectId_, work)}
                    , {column::THE_GEOM, toSqlValue(level.polygons[idx], work)}
                    , {column::ZMAX, toSqlValue(level.zoomRange.zmax, work)}
                    , {column::ZMIN, toSqlValue(level.zoomRange.zmin, work)}
                    };
                if (!serviceAttrs_.empty()) {
                    namedValues.insert({column::SERVICE_ATTRS, toSqlValue(serviceAttrs_, work)});
                }
                result.emplace_back(toSqlStatement(namedValues));
            }
        }
    }
    return result;
}

size_t ContourObject::polygonCount() const
{
    size_t result = 0;
    for (const auto& level: pyramid_) {
        result += level.polygons.size();
    }
    return result;
}

bool ContourObject::isValid() const
{
    auto itr = serviceAttrs_.find(service_attrs::INVALID);
    return itr == serviceAttrs_.end() || itr->second != service_attrs::FLAG_TRUE;
}

size_t getPolygonCount(const ContourObjects& objs)
{
    size_t result = 0;
    for (const auto& obj: objs) {
        result += obj.polygonCount();
    }
    return result;
}

size_t getInvalidCount(const ContourObjects& objs)
{
    return std::count_if(objs.begin(), objs.end(), [](const ContourObject& obj){
        return !obj.isValid();
    });
}

size_t getDeletedCount(const ContourObjects& objs)
{
    return std::count_if(objs.begin(), objs.end(), [](const ContourObject& obj){
        return obj.isDeleted();
    });
}

std::vector<std::string> dumpToSql(const ContourObjects& objs, pqxx::transaction_base& work)
{
    if (objs.empty()) {
        return {};
    }

    if (objs.size() == 1) {
        return objs.begin()->dumpToSql(work);
    }

    std::vector<std::string> result;
    for (const auto& obj: objs) {
        for (auto& data: obj.dumpToSql(work)) {
            result.emplace_back(std::move(data));
        }
    }
    return result;
}

std::string sqlSelectCount()
{
    return "SELECT COUNT(*) FROM " + table::CONTOUR_OBJECTS;
}

std::string sqlSelectObjectIds()
{
    return "SELECT DISTINCT " + column::OBJECT_ID +
        " FROM " + table::CONTOUR_OBJECTS;
}

std::string sqlDelete(const revision_meta::TObjectIdSet& ids)
{
    std::stringstream result;
    result
        << "DELETE FROM " << table::CONTOUR_OBJECTS
        << " WHERE " << column::OBJECT_ID << " IN (";
    bool first = true;
    for (auto id: ids) {
        if (!first) {
            result << ',';
        }
        first = false;
        result << id;
    }
    result << ");";
    return result.str();
}

std::string sqlSelectArea()
{
    return "SELECT " + column::AREA + " FROM " + table::CONTOUR_OBJECTS;
}

} // contours
} // wiki
} // maps
