#include "generalization.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/tile/include/coord.h>
#include <maps/libs/tile/include/utils.h>

#include <library/cpp/testing/unittest/env.h>

#include <algorithm>

namespace maps {
namespace wiki {
namespace contours {
namespace {

const std::string ATTR_HEIGHT = "height";
const std::string ATTR_ID = "id";
const std::string ATTR_WIDTH = "width";
const std::string ATTR_ZMAX = "zmax";
const std::string ATTR_ZMIN = "zmin";
const std::string CATEGORY_OTHER = "other";
const std::string XML = "/usr/share/yandex/maps/wiki/editor/generalizations.xml";
const std::string XPATH_POLYGON = "polygon";
const std::string XPATH_GABARITS = "gabarits";
const std::string XPATH_GENERALIZATIONS = "/generalizations/generalization";
const std::string XPATH_MIN_FRACTION = "min-fraction";
const std::string XPATH_MIN_SIZE = "min-size";
const std::string XPATH_WINDOW = "window";

} // namespace

Generalization::Generalization
    ( const ZoomRange& zoomRange
    , const xml3::Node& gabarits
    )
    : zoomRange_(zoomRange)
    , window_
        { gabarits.node(XPATH_WINDOW).attr<double>(ATTR_WIDTH)
        , gabarits.node(XPATH_WINDOW).attr<double>(ATTR_HEIGHT)
        }
    , minSize_
        { gabarits.node(XPATH_MIN_SIZE).attr<double>(ATTR_WIDTH)
        , gabarits.node(XPATH_MIN_SIZE).attr<double>(ATTR_HEIGHT)
        }
    , minFraction_{gabarits.node(XPATH_MIN_FRACTION).value<double>()}
{
}

/**
 * \see https://git.yandex.ru/gitweb/maps/mapscore.git?a=blob;f=wikimap/mapspro/services/editor/src/configs/generalization.cpp
 */
ZoomRange Generalization::zoomRange(double area) const
{
    double rMax = tile::zoomToResolution(maps::tile::MAX_ZOOM);
    double rMin = tile::zoomToResolution(0.);
    rMax *= rMax;
    rMin *= rMin;
    auto zmin = std::max
        ( tile::MAX_ZOOM -
            ::log(area / (minSize_.width * minSize_.height * rMax))
            / (2. * ::log(2.)) + 0.5
        , static_cast<double>(zoomRange_.zmin)
        );
    auto zmax = std::max(zmin, static_cast<double>(zoomRange_.zmax));
    return ZoomRange{static_cast<Zoom>(zmin), static_cast<Zoom>(zmax)};
}

GeneralizationLoader::GeneralizationLoader()
{
    std::string path = FromYaTest()
        ? ArcadiaSourceRoot() + "/maps/wikimap/mapspro/cfg/editor/generalizations.xml"
        : XML;

    xml3::Doc doc(path);
    auto generalizations = doc.nodes(XPATH_GENERALIZATIONS);
    for (size_t i = 0; i != generalizations.size(); ++i) {
        const auto generalization = generalizations[i];
        const auto category = generalization.attr<std::string>(ATTR_ID);

        const auto& polygon = generalization.node(XPATH_POLYGON, true);
        if (polygon.isNull()) {
            continue;
        }
        ZoomRange zoomRange
            { polygon.attr<Zoom>(ATTR_ZMIN)
            , polygon.attr<Zoom>(ATTR_ZMAX)
            };

        const auto gabarits = polygon.node(XPATH_GABARITS, true);
        if (gabarits.isNull()) {
            continue;
        }
        DEBUG() << "generalization: " << category;
        categoryToGeneralization_.insert(
            { category
            , Generalization(zoomRange, gabarits)
            });
    }
}

ZoomRange GeneralizationLoader::zoomRange(const std::string& category, double area)
{
    static GeneralizationLoader loader;
    auto itr = loader.categoryToGeneralization_.find(category);
    if (itr == loader.categoryToGeneralization_.end()) {
        itr = loader.categoryToGeneralization_.find(CATEGORY_OTHER);
    }
    auto& gen = itr->second;
    return gen.zoomRange(area);
}

} // contours
} // wiki
} // maps
