#include "loader.h"
#include "magic_strings.h"
#include "revision_meta/historical_snapshot.h"
#include "revision_meta/snapshot.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/xml/include/xml.h>
#include <yandex/maps/wiki/common/revision_utils.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>

#include <library/cpp/testing/unittest/env.h>

namespace maps {
namespace wiki {
namespace contours {

Loader::Loader
    ( common::PoolHolder& pool
    , revision_meta::TCommitId commitId
    , const Params& params
    )
    : pool_(pool)
    , commitId_(commitId)
    , params_(params)
{
    std::string path = FromYaTest()
        ? ArcadiaSourceRoot() + "/maps/wikimap/mapspro/cfg/editor/contour_objects.xml"
        : xml::CONTOUR_OBJECTS_CONFIG_PATH;

    xml3::Doc doc(path);
    auto contourObjects = doc.nodes(xml::xpath::COUNTOUR_OBJECTS);
    for (size_t i = 0; i != contourObjects.size(); ++i) {
        const auto& contourObject = contourObjects[i];
        DEBUG()
            << "contour objects: "
            << contourObject.attr<std::string>(xml::attr::CATEGORY_ID);
        contourObjectCategoryIds_.insert(contourObject.attr<std::string>(xml::attr::CATEGORY_ID));
        categories_.addCategory
            ( contourObject.attr<std::string>(xml::attr::CATEGORY_ID)
            , revision_meta::CategoryType::Complex
            );

        const auto& contour = contourObject.node(xml::xpath::COUNTOUR);
        categories_.addCategory
            ( contour.attr<std::string>(xml::attr::CATEGORY_ID)
            , revision_meta::CategoryType::Complex
            );
        categories_.addRelation
            ( contourObject.attr<std::string>(xml::attr::CATEGORY_ID)
            , contour.attr<std::string>(xml::attr::CATEGORY_ID)
            , contour.attr<std::string>(xml::attr::ROLE_ID)
            );

        const auto& linearElement = contour.node(xml::xpath::LINEAR_ELEMENT);
        categories_.addCategory
            ( linearElement.attr<std::string>(xml::attr::CATEGORY_ID)
            , revision_meta::CategoryType::Geom
            );
        categories_.addRelation
            ( contour.attr<std::string>(xml::attr::CATEGORY_ID)
            , linearElement.attr<std::string>(xml::attr::CATEGORY_ID)
            , linearElement.attr<std::string>(xml::attr::ROLE_ID)
            );
    }
    dependFromMasterContourObjectCategoryIds_.insert(category::AD);
    dependFromMasterContourObjectCategoryIds_.insert(category::AD_NEUTRAL);
    categories_.addCategory(category::REGION, revision_meta::CategoryType::Geom);
    categories_.addRelation(category::REGION, category::AD, role::ASSIGNED);
    categories_.addRelation(category::REGION, category::AD_NEUTRAL, role::ASSIGNED_AD_NEUTRAL);
}

revision_meta::TObjectIdSet Loader::allObjectIds() const
{
    auto txn = common::getReadTransactionForCommit
        ( pool_.pool()
        , revision::TRUNK_BRANCH_ID
        , commitId_
        , [](const std::string& msg) { DEBUG() << msg; }
        );
    revision::RevisionsGateway gw(*txn);
    auto snapshot = gw.snapshot(commitId_);
    auto revIds = snapshot.revisionIdsByFilter
        ( categories_.categoriesFilter
            ( revision_meta::CategoryType::Complex
            , contourObjectCategoryIds_
            )
        && revision::filters::ObjRevAttr::isNotDeleted()
        && revision::filters::ObjRevAttr::isNotRelation()
        );

    revision_meta::TObjectIdSet result;
    std::transform
        ( revIds.begin()
        , revIds.end()
        , std::inserter(result, result.end())
        , [](const revision_meta::TRevisionId& revId) { return revId.objectId(); }
        );
    return result;
}

revision_meta::TObjectIdSet
Loader::affectedObjectIds(revision_meta::TCommitId minCommitId) const
{
    auto txn = common::getReadTransactionForCommit
        ( pool_.pool()
        , revision::TRUNK_BRANCH_ID
        , commitId_
        , [](const std::string& msg) { DEBUG() << msg; }
        );
    revision_meta::HistoricalSnapshot snapshot
        ( minCommitId
        , commitId_
        , *txn
        , revision::TRUNK_BRANCH_ID
        , categories_
        );
    auto affectedBySlaves = snapshot.affectedObjects
        ( revision_meta::RelationType::Master
        , [this](const revision_meta::TCategoryId& catId) {
            return contourObjectCategoryIds_.count(catId);
        });
    auto affectedByMasters = snapshot.affectedObjects
        ( revision_meta::RelationType::Slave
        , [this](const revision_meta::TCategoryId& catId) {
            return dependFromMasterContourObjectCategoryIds_.count(catId);
        });
    affectedBySlaves.insert(affectedByMasters.cbegin(), affectedByMasters.cend());
    return affectedBySlaves;
}

ContourObjects Loader::load(const revision_meta::TObjectIdSet& ids) const
{
    auto txn = common::getReadTransactionForCommit
        ( pool_.pool()
        , revision::TRUNK_BRANCH_ID
        , commitId_
        , [](const std::string& msg) { DEBUG() << msg; }
        );
    revision_meta::Snapshot snapshot
        (commitId_, *txn, revision::TRUNK_BRANCH_ID, categories_);
    snapshot.loadObjects(ids);
    snapshot.loadSlavesRecursive(ids);
    snapshot.loadMasters(ids);
    ContourObjects result;
    result.reserve(ids.size());
    for (auto id: ids) {
        ContourObject obj;
        obj.load(snapshot, id, params_);
        result.push_back(std::move(obj));
    }
    return result;
}

} // contours
} // wiki
} // maps
