#include "categories.h"

#include "relation.h"

#include <maps/libs/common/include/exception.h>

#include <unordered_map>
#include <stack>

namespace maps {
namespace wiki {
namespace contours {
namespace revision_meta {

// Impl

class Categories::Impl {
public:
    void addCategory(const TCategoryId& categoryId, CategoryType type);

    void addRelation(
        const TCategoryId& masterCategoryId,
        const TCategoryId& slaveCategoryId,
        const TRoleId& roleId);

    CategoryType type(const TCategoryId& categoryId) const;

    bool isDefined(const TCategoryId& categoryId) const;

    bool hasSlaves(const TCategoryId& categoryId) const;
    bool hasMasters(const TCategoryId& categoryId) const;

    revision::filters::ProxyFilterExpr objectsFilter(CategoryType type) const;
    revision::filters::ProxyFilterExpr relationsFilter() const;

    revision::filters::ProxyFilterExpr categoriesFilter(CategoryType type) const;
    revision::filters::ProxyFilterExpr categoriesFilter(CategoryType type, const TStringSet& categoryIds) const;

    revision::filters::ProxyFilterExpr
    slaveRelationsFilter(const TCategoryId& categoryId) const;

    revision::filters::ProxyFilterExpr
    masterRelationsFilter(const TCategoryId& categoryId) const;

private:
    struct CategoryData;

    CategoryData& categoryData(const TCategoryId& categoryId);
    const CategoryData& categoryData(const TCategoryId& categoryId) const;

    struct RelationTypeKey
    {
        bool operator < (const RelationTypeKey& other) const
        {
            int cmp;
            if ((cmp = masterCategoryId.compare(other.masterCategoryId)) != 0) {
                return cmp < 0;
            }
            if ((cmp = slaveCategoryId.compare(other.slaveCategoryId)) != 0) {
                return cmp < 0;
            }
            if ((cmp = roleId.compare(other.roleId)) != 0) {
                return cmp < 0;
            }
            return false;
        }

        TCategoryId masterCategoryId;
        TCategoryId slaveCategoryId;
        TCategoryId roleId;
    };

    typedef std::set<RelationTypeKey> RelationTypeKeySet;
    typedef std::map<TRoleId, TCategoryIdSet> RelatedCategoriesByRoleMap;

    struct CategoryData
    {
        CategoryType type;
        RelatedCategoriesByRoleMap masterRelations;
        RelatedCategoriesByRoleMap slaveRelations;
    };

    std::map<TCategoryId, CategoryData> categoriesData_;

    RelationTypeKeySet relations_;
};

Categories::Impl::CategoryData&
Categories::Impl::categoryData(const TCategoryId& categoryId)
{
    auto it = categoriesData_.find(categoryId);
    REQUIRE(it != categoriesData_.end(),
        "Category " << categoryId << " is not defined");
    return it->second;
}

const Categories::Impl::CategoryData&
Categories::Impl::categoryData(const TCategoryId& categoryId) const
{
    auto it = categoriesData_.find(categoryId);
    REQUIRE(it != categoriesData_.end(),
        "Category " << categoryId << " is not defined");
    return it->second;
}

namespace {

TCategoryIdSet
plainCategoryIdsToCanonical(const TCategoryIdSet& plainIds)
{
    TCategoryIdSet canonicalIds;
    for (const auto& catId : plainIds) {
        canonicalIds.insert("cat:" + catId);
    }
    return canonicalIds;
}

} // namespace

void
Categories::Impl::addCategory(const TCategoryId& categoryId, CategoryType type)
{
    categoriesData_.insert({categoryId, CategoryData{type, {}, {}}});
}

void
Categories::Impl::addRelation(
    const TCategoryId& masterCategoryId,
    const TCategoryId& slaveCategoryId,
    const TRoleId& roleId)
{
    CategoryData& masterCat = categoryData(masterCategoryId);
    CategoryData& slaveCat = categoryData(slaveCategoryId);
    relations_.insert({masterCategoryId, slaveCategoryId, roleId});
    masterCat.slaveRelations[roleId].insert(slaveCategoryId);
    slaveCat.masterRelations[roleId].insert(masterCategoryId);
}

CategoryType
Categories::Impl::type(const TCategoryId& categoryId) const
{
    return categoryData(categoryId).type;
}

bool
Categories::Impl::isDefined(const TCategoryId& categoryId) const
{
    return categoriesData_.count(categoryId);
}

bool
Categories::Impl::hasSlaves(const TCategoryId& categoryId) const
{
    return !categoryData(categoryId).slaveRelations.empty();
}

bool
Categories::Impl::hasMasters(const TCategoryId& categoryId) const
{
    return !categoryData(categoryId).masterRelations.empty();
}

revision::filters::ProxyFilterExpr
Categories::Impl::objectsFilter(CategoryType type) const
{
    namespace rf = revision::filters;

    rf::ProxyFilterExpr filter = categoriesFilter(type) && rf::ObjRevAttr::isNotRelation();
    if (type != CategoryType::Geom) {
        return std::move(filter) && !rf::Geom::defined();
    }

    return filter;
}

revision::filters::ProxyFilterExpr
Categories::Impl::categoriesFilter(CategoryType type) const
{
    namespace rf = revision::filters;

    TCategoryIdSet categoryIds;

    for (const auto& catData : categoriesData_) {
        if (this->type(catData.first) == type) {
            categoryIds.insert(catData.first);
        }
    }

    if (categoryIds.empty()) {
        return rf::False();
    }

    return rf::Attr::definedAny(plainCategoryIdsToCanonical(categoryIds));
}

revision::filters::ProxyFilterExpr
Categories::Impl::categoriesFilter(CategoryType type, const TStringSet& categoryIds) const
{
    namespace rf = revision::filters;

    TCategoryIdSet catIds = categoryIds;

    for (const auto& catId : categoryIds) {
        if (this->type(catId) != type) {
            catIds.erase(catId);
        }
    }

    if (categoryIds.empty()) {
        return rf::False();
    }

    return rf::Attr::definedAny(plainCategoryIdsToCanonical(catIds));
}

revision::filters::ProxyFilterExpr
Categories::Impl::relationsFilter() const
{
    namespace rf = revision::filters;

    ASSERT(!relations_.empty());

    std::unordered_map<
        std::string,
        std::unordered_map<TCategoryId, TCategoryIdSet>
    > relationsMap;

    for (const auto& rKey : relations_) {
        relationsMap[rKey.roleId][rKey.masterCategoryId].insert(rKey.slaveCategoryId);
    }

    std::unique_ptr<rf::ProxyFilterExpr> filter;
    for (const auto& roleMap : relationsMap) {
        std::unique_ptr<rf::ProxyFilterExpr> rolesFilter;
        for (const auto& masterToSlavesMap : roleMap.second) {
            auto categoriesFilter =
                rf::Attr(REL_MASTER) == masterToSlavesMap.first &&
                rf::Attr(REL_SLAVE).in(masterToSlavesMap.second);
            if (rolesFilter) {
                *rolesFilter |= std::move(categoriesFilter);
            } else {
                rolesFilter.reset(new rf::ProxyFilterExpr(std::move(categoriesFilter)));
            }
        }

        *rolesFilter &= rf::Attr(REL_ROLE) == roleMap.first;
        if (filter) {
            *filter |= std::move(*rolesFilter);
        } else {
            filter.swap(rolesFilter);
        }
    }

    return std::move(*filter) &&
        rf::Attr::definedAll(TStringList{REL_MASTER, REL_SLAVE, REL_ROLE});
}

revision::filters::ProxyFilterExpr
Categories::Impl::slaveRelationsFilter(const TCategoryId& categoryId) const
{
    namespace rf = revision::filters;

    const CategoryData& category = categoryData(categoryId);
    if (category.slaveRelations.empty()) {
        return rf::False();
    }

    std::unique_ptr<rf::ProxyFilterExpr> rolesFilter;
    for (const auto& slaveRel : category.slaveRelations) {
        auto roleFilter =
            rf::Attr(REL_ROLE) == slaveRel.first &&
            rf::Attr(REL_SLAVE).in(slaveRel.second);
        if (rolesFilter) {
            *rolesFilter |= std::move(roleFilter);
        } else {
            rolesFilter.reset(new rf::ProxyFilterExpr(std::move(roleFilter)));
        }
    }
    return std::move(*rolesFilter) && rf::Attr(REL_MASTER) == categoryId;
}

revision::filters::ProxyFilterExpr
Categories::Impl::masterRelationsFilter(const TCategoryId& categoryId) const
{
    namespace rf = revision::filters;

    const CategoryData& category = categoryData(categoryId);
    if (category.masterRelations.empty()) {
        return rf::False();
    }

    std::unique_ptr<rf::ProxyFilterExpr> rolesFilter;
    for (const auto& masterRel : category.masterRelations) {
        auto roleFilter =
            rf::Attr(REL_ROLE) == masterRel.first &&
            rf::Attr(REL_MASTER).in(masterRel.second);
        if (rolesFilter) {
            *rolesFilter |= std::move(roleFilter);
        } else {
            rolesFilter.reset(new rf::ProxyFilterExpr(std::move(roleFilter)));
        }
    }
    return std::move(*rolesFilter) && rf::Attr(REL_SLAVE) == categoryId;
}

// Categories

Categories::Categories()
    : impl_(new Impl())
{}

Categories::~Categories() {}

Categories::Categories(const Categories& oth)
    : impl_(new Impl(*oth.impl_))
{}

Categories& Categories::operator = (const Categories& oth)
{
    *impl_ = *oth.impl_;
    return *this;
}

void
Categories::addCategory(const TCategoryId& categoryId, CategoryType type)
{ impl_->addCategory(categoryId, type); }

void
Categories::addRelation(
    const TCategoryId& masterCategoryId,
    const TCategoryId& slaveCategoryId,
    const TRoleId& roleId)
{
    impl_->addRelation(masterCategoryId, slaveCategoryId, roleId);
}

bool Categories::isComplex(const TCategoryId& categoryId) const
{ return impl_->type(categoryId) == CategoryType::Complex; }

bool Categories::isGeom(const TCategoryId& categoryId) const
{ return impl_->type(categoryId) == CategoryType::Geom; }

bool Categories::isAttr(const TCategoryId& categoryId) const
{ return impl_->type(categoryId) == CategoryType::Attr; }

CategoryType Categories::type(const TCategoryId& categoryId) const
{ return impl_->type(categoryId); }

bool Categories::isDefined(const TCategoryId& categoryId) const
{ return impl_->isDefined(categoryId); }

bool Categories::hasSlaves(const TCategoryId& categoryId) const
{ return impl_->hasSlaves(categoryId); }

bool Categories::hasMasters(const TCategoryId& categoryId) const
{ return impl_->hasMasters(categoryId); }

revision::filters::ProxyFilterExpr Categories::objectsFilter(CategoryType type) const
{ return impl_->objectsFilter(type); }

revision::filters::ProxyFilterExpr Categories::categoriesFilter(CategoryType type) const
{ return impl_->categoriesFilter(type); }

revision::filters::ProxyFilterExpr
Categories::categoriesFilter(CategoryType type, const TStringSet& categoryIds) const
{ return impl_->categoriesFilter(type, categoryIds); }

revision::filters::ProxyFilterExpr Categories::relationsFilter() const
{ return impl_->relationsFilter(); }

revision::filters::ProxyFilterExpr
Categories::slaveRelationsFilter(const TCategoryId& categoryId) const
{ return impl_->slaveRelationsFilter(categoryId); }

revision::filters::ProxyFilterExpr
Categories::masterRelationsFilter(const TCategoryId& categoryId) const
{ return impl_->masterRelationsFilter(categoryId); }

} // namespace revision_meta
} // namespace contours
} // namespace wiki
} // namespace maps
