#include "historical_snapshot.h"

#include "revisions_loader.h"

namespace maps {
namespace wiki {
namespace contours {
namespace revision_meta {

namespace {

TObjectIdSet& operator |=(TObjectIdSet& lhs, const TObjectIdSet& rhs)
{
    std::copy(rhs.begin(), rhs.end(), std::inserter(lhs, lhs.end()));
    return lhs;
}

TObjectIdSet operator -(const TObjectIdSet& lhs, const TObjectIdSet& rhs)
{
    TObjectIdSet result;
    std::set_difference(
        lhs.begin(), lhs.end(),
        rhs.begin(), rhs.end(),
        std::inserter(result, result.end()));
    return result;
}

TObjectIdSet toObjectIds(
    const TRevisionList& objs,
    TCategoryPredicate filter = [](const TCategoryId&){ return true; })
{
    TObjectIdSet result;
    for (const auto& rev: objs) {
        Object obj(rev);
        if (filter(obj.categoryId())) {
            result.insert(obj.objectId());
        }
    }
    return result;
}

TObjectIdSet toObjectIds(
    const TRevisionList& rels,
    RelationType relationType,
    TCategoryPredicate filter = [](const TCategoryId&){ return true; })
{
    TObjectIdSet result;
    for (const auto& rev: rels) {
        Relation rel(rev);
        if (RelationType::Master == relationType) {
            if (filter(rel.masterCategoryId())) {
                result.insert(rel.masterId());
            }
        } else {
            if (filter(rel.slaveCategoryId())) {
                result.insert(rel.slaveId());
            }
        }
    }
    return result;
}

} // namespace

class HistoricalSnapshot::Impl {
public:
    Impl
        ( TCommitId minCommitId
        , TCommitId maxCommitId
        , Transaction& txn
        , TBranchId branchId
        , const Categories& categories
        )
        : minCommitId_(minCommitId)
        , maxCommitId_(maxCommitId)
        , loader_(new RevisionsLoader(txn, branchId, categories))
    {}

    TCommitId minCommitId() const { return minCommitId_; }
    TCommitId maxCommitId() const { return maxCommitId_; }

    TObjectIdSet affectedObjects
        ( RelationType relationType
        , TCategoryPredicate filter
        )
    {
        auto objs = loader_->loadObjects(
            loader_->loadObjectIds(minCommitId_, maxCommitId_));
        auto rels = loader_->loadRelations(minCommitId_, maxCommitId_);

        auto wave = toObjectIds(objs);
        wave |= toObjectIds(rels, relationType);

        auto result = affectedObjectsRecursive(
            std::move(wave),
            relationType,
            filter);
        result |= toObjectIds(objs, filter);
        result |= toObjectIds(rels, relationType, filter);
        return result;
    }

private:
    TCommitId minCommitId_;
    TCommitId maxCommitId_;
    std::unique_ptr<RevisionsLoader> loader_;

    TObjectIdSet affectedObjectsRecursive
        ( TObjectIdSet&& wave
        , RelationType relationType
        , TCategoryPredicate filter
        )
    {
        TObjectIdSet visitedObjects = wave;
        TObjectIdSet affectedObjects;
        while (!wave.empty()) {
            auto rels = loader_->loadNotDeletedRelations(
                relationType,
                wave,
                maxCommitId_);

            wave = toObjectIds(rels, relationType) - visitedObjects;
            visitedObjects |= wave;

            affectedObjects |= toObjectIds(rels, relationType, filter);
        }
        return affectedObjects;
    }

};

HistoricalSnapshot::HistoricalSnapshot
    ( TCommitId minCommitId
    , TCommitId maxCommitId
    , Transaction& txn
    , TBranchId branchId
    , const Categories& categories
    )
    : impl_(new Impl(minCommitId, maxCommitId, txn, branchId, categories))
{}

TCommitId HistoricalSnapshot::minCommitId() const { return impl_->minCommitId(); }
TCommitId HistoricalSnapshot::maxCommitId() const { return impl_->maxCommitId(); }

TObjectIdSet HistoricalSnapshot::affectedObjects
    ( RelationType relationType
    , TCategoryPredicate filter
    )
{
    return impl_->affectedObjects(relationType, filter);
}

MOVABLE_PIMPL_DEFINITIONS(HistoricalSnapshot)

} // namespace revision_meta
} // namespace contours
} // namespace wiki
} // namespace maps
