#include "snapshot_impl.h"

#include "snapshot_helpers.h"

#include <maps/libs/common/include/exception.h>

namespace maps {
namespace wiki {
namespace contours {
namespace revision_meta {

Snapshot::Impl::Impl
    ( TCommitId headCommitId
    , Transaction& txn
    , TBranchId branchId
    , const Categories& categories
    )
    : headCommitId_(headCommitId)
    , loader_(new RevisionsLoader(txn, branchId, categories))
{}

boost::optional<TCommitId>
Snapshot::Impl::headObjectCommitId(TObjectId objectId, TCommitId commitId) const
{
    return utils::headCommitId(objectCommitIds_, objectId, commitId);
}

boost::optional<TCommitId>
Snapshot::Impl::headRelationCommitId(TObjectId relationId, TCommitId commitId) const
{
    return utils::headCommitId(relationCommitIds_, relationId, commitId);
}

boost::optional<TCommitId>
Snapshot::Impl::maxObjectCommitId(TObjectId objectId) const
{
    auto it = objectCommitIds_.find(objectId);
    if (it == objectCommitIds_.end()) {
        return boost::none;
    }
    REQUIRE(!it->second.empty(), "Object " << objectId << " has empty revisions set");
    return *std::prev(it->second.end());
}


const Object*
Snapshot::Impl::tryGetObjectRevision(TObjectId objectId, TCommitId maxCommitId) const
{
    auto headCommitId = headObjectCommitId(objectId, maxCommitId);
    if (!headCommitId) {
        return nullptr;
    }
    auto revId = TRevisionId(objectId, *headCommitId);
    auto it = objects_.find(revId);
    REQUIRE(it != objects_.end(), "Revision " << revId << " not found");
    return it->second.get();
}

const Object&
Snapshot::Impl::objectRevision(TObjectId objectId, TCommitId maxCommitId) const
{
    auto headRevPtr = tryGetObjectRevision(objectId, maxCommitId);
    REQUIRE(headRevPtr,
            "Object " << objectId << " does not exist at commit " << maxCommitId);
    return *headRevPtr;
}

const Object&
Snapshot::Impl::existingObjectRevision(TObjectId objectId, TCommitId maxCommitId) const
{
    const auto& rev = objectRevision(objectId, maxCommitId);
    REQUIRE(rev.state() != ObjectState::Deleted,
            "Object " << objectId << " is deleted at commit " << maxCommitId);
    return rev;
}

const Relation*
Snapshot::Impl::tryGetRelationRevision(TObjectId relId, TCommitId maxCommitId) const
{
    auto headCommitId = headRelationCommitId(relId, maxCommitId);
    if (!headCommitId) {
        return nullptr;
    }
    auto revId = TRevisionId(relId, *headCommitId);
    auto it = relations_.find(revId);
    REQUIRE(it != relations_.end(), "Revision " << revId << " not found");
    return it->second.get();
}

const Relation&
Snapshot::Impl::relationRevision(TObjectId relId, TCommitId maxCommitId) const
{
    auto headRelPtr = tryGetRelationRevision(relId, maxCommitId);
    REQUIRE(headRelPtr,
            "Relation " << relId << " does not exist at commit " << maxCommitId);
    return *headRelPtr;
}

void
Snapshot::Impl::addObject(const ObjectPtr& object)
{
    ObjectPtr obj = object;
    TRevisionId curRevId(obj->objectId(), obj->commitId());
    objects_.insert({curRevId, obj});
    objectCommitIds_[obj->objectId()].insert(obj->commitId());
}

void
Snapshot::Impl::addRelation(const RelationPtr& relation)
{
    TRevisionId curRevId(relation->relationId(), relation->commitId());
    relations_.insert({curRevId, relation});
    relationCommitIds_[relation->relationId()].insert(relation->commitId());
    relationsByMasters_[relation->masterId()].insert(relation->relationId());
    relationsBySlaves_[relation->slaveId()].insert(relation->relationId());
}

Snapshot::Impl::RelationSet
Snapshot::Impl::relations(
    RelationType type, TObjectId objectId, TCommitId commitId) const
{
    auto& relationsByObjects = type == RelationType::Master
        ? relationsBySlaves_
        : relationsByMasters_;

    auto relationsIt = relationsByObjects.find(objectId);
    if (relationsIt == relationsByObjects.end()) {
        return {};
    }

    RelationSet result;
    for (auto relationId : relationsIt->second) {
        auto relationPtr = tryGetRelationRevision(relationId, commitId);
        if (!relationPtr || relationPtr->state() == ObjectState::Deleted) {
            continue;
        }
        const auto relativeId = type == RelationType::Master
            ? relationPtr->masterId()
            : relationPtr->slaveId();
        auto relativePtr = tryGetObjectRevision(relativeId, commitId);
        if (relativePtr && relativePtr->state() != ObjectState::Deleted) {
            REQUIRE(result.insert(relationPtr).second,
                "Relation " << relationPtr->relationId()
                    << " has already been added for " << objectId << " at " << commitId);
        }
    }
    return result;
}

TObjectIdSet
Snapshot::Impl::allRelativeIds(
    RelationType type, const TObjectIdSet& objectIds) const
{
    auto& relationsByObjects = type == RelationType::Master
        ? relationsBySlaves_
        : relationsByMasters_;

    TObjectIdSet result;
    for (auto objectId : objectIds) {
        auto relationsIt = relationsByObjects.find(objectId);
        if (relationsIt == relationsByObjects.end()) {
            continue;
        }

        for (auto relationId : relationsIt->second) {
            for (auto commitId : relationCommitIds_.at(relationId)) {
                const auto& rel = relationRevision(relationId, commitId);
                result.insert(type == RelationType::Master
                    ? rel.masterId()
                    : rel.slaveId());
            }
        }
    }

    return result;
}

TObjectIdSet
Snapshot::Impl::slaves(TObjectId objectId, TCommitId commitId) const
{
    REQUIRE(objectsWithLoadedSlaves_.count(objectId),
        "Slaves not loaded for object " << objectId << ", commit id " << commitId);

    TObjectIdSet slaveIds;
    for (auto relationPtr : relations(RelationType::Slave, objectId, commitId)) {
        slaveIds.insert(relationPtr->slaveId());
    }
    return slaveIds;
}

TObjectIdSet
Snapshot::Impl::masters(TObjectId objectId, TCommitId commitId) const
{
    REQUIRE(objectsWithLoadedMasters_.count(objectId),
        "Masters not loaded for object " << objectId << ", commit id " << commitId);

    TObjectIdSet masterIds;
    for (auto relationPtr : relations(RelationType::Master, objectId, commitId)) {
        masterIds.insert(relationPtr->masterId());
    }
    return masterIds;
}

TObjectIdSet
Snapshot::Impl::addObjects(const ObjectPtrList& objects)
{
    TObjectIdSet ids;
    for (const auto& object : objects) {
        ids.insert(object->objectId());
        addObject(object);
    }
    return ids;
}

void
Snapshot::Impl::addRelations(const RelationPtrList& relations)
{
    loadRelatedObjects(relations);

    for (auto rel : relations) {
        addRelation(rel);
    }
}

void
Snapshot::Impl::loadRelatedObjects(const RelationPtrList& relations)
{
    ObjectIdsByCategoryMap relatedObjects;
    for (const RelationPtr& rel : relations) {
        if (categories().isDefined(rel->masterCategoryId()) &&
            objectShouldBeLoaded(rel->masterId()))
        {
            relatedObjects[rel->masterCategoryId()].insert(rel->masterId());
        }
        if (categories().isDefined(rel->slaveCategoryId()) &&
            objectShouldBeLoaded(rel->slaveId()))
        {
            relatedObjects[rel->slaveCategoryId()].insert(rel->slaveId());
        }
    }

    if (relatedObjects.empty()) {
        return;
    }

    addObjects(loadObjects(relatedObjects));

    for (const auto& ids : relatedObjects) {
        loadedObjectIds_.insert(ids.second.begin(), ids.second.end());
    }
}

ObjectPtrList
Snapshot::Impl::loadObjects(const ObjectIdsByCategoryMap& objectIdsByCategoryMap)
{
    std::map<CategoryType, TObjectIdSet> objectIdsByCategoryType;
    for (const auto& ids : objectIdsByCategoryMap) {
        const TCategoryId& categoryId = ids.first;
        if (!ids.second.empty()) {
            objectIdsByCategoryType[categories().type(categoryId)].insert(
                ids.second.begin(), ids.second.end());
        }
    }

    ObjectPtrList result;

    for (const auto& catObjectIds : objectIdsByCategoryType) {
        auto catType = catObjectIds.first;
        const auto& objectIds = catObjectIds.second;
        auto revisionIds = loader_->loadObjectIds(catType, objectIds, headCommitId_);
        auto revisions = loader_->loadObjects(catType, revisionIds);
        for (const auto& rev : revisions) {
            result.push_back(std::make_shared<Object>(rev));
        }
    }

    return result;
}

TObjectIdSet
Snapshot::Impl::loadObjects(const TObjectIdSet& objectIds)
{
    TObjectIdSet toLoad;
    TObjectIdSet alreadyLoaded;
    for (auto id : objectIds) {
        (objectShouldBeLoaded(id) ? toLoad : alreadyLoaded).insert(id);
    }

    ObjectPtrList loaded;
    auto revisionIds = loader_->loadObjectIds(toLoad, headCommitId_);
    auto revisions = loader_->loadObjects(revisionIds);
    for (const auto& rev : revisions) {
        loaded.push_back(std::make_shared<Object>(rev));
    }
    auto loadedIds = addObjects(loaded);

    loadedIds.insert(alreadyLoaded.begin(), alreadyLoaded.end());

    loadedObjectIds_.insert(toLoad.begin(), toLoad.end());

    return loadedIds;
}

void
Snapshot::Impl::loadRelativesRecursive(RelationType type, const TObjectIdSet& objectIds)
{
    if (objectIds.empty()) {
        return;
    }

    loadRelatives(type, objectIds);
    TObjectIdSet ids = allRelativeIds(type, objectIds);
    while (!ids.empty()) {
        loadRelatives(type, ids);
        auto nextIds = allRelativeIds(type, ids);
        ids.swap(nextIds);
    }
}

void
Snapshot::Impl::loadRelatives(RelationType type, const TObjectIdSet& objectIds)
{
    if (objectIds.empty()) {
        return;
    }

    auto& objectsWithLoadedRelatives = type == RelationType::Master
        ? objectsWithLoadedMasters_
        : objectsWithLoadedSlaves_;

    ObjectIdsByCategoryMap objectsByCategories;
    for (auto objectId : objectIds) {
        if (objectsWithLoadedRelatives.count(objectId)) {
            continue;
        }
        auto maxObjectCommitId = this->maxObjectCommitId(objectId);
        if (!maxObjectCommitId) {
            continue;
        }
        const auto& categoryId = objectRevision(objectId, *maxObjectCommitId).categoryId();
        if (type == RelationType::Master
            ? categories().hasMasters(categoryId)
            : categories().hasSlaves(categoryId))
        {
            objectsByCategories[categoryId].insert(objectId);
        }
    }
    if (!objectsByCategories.empty()) {
        auto relations = loadRelations(type, objectsByCategories);
        addRelations(relations);
    }

    objectsWithLoadedRelatives.insert(objectIds.begin(), objectIds.end());
}

RelationPtrList
Snapshot::Impl::loadRelations(
    RelationType type,
    const ObjectIdsByCategoryMap& objectIdsByCategoryMap)
{
    RelationPtrList result;
    for (const auto& ids : objectIdsByCategoryMap) {
        auto revisions =
            loader_->loadRelations(type, ids.first, ids.second, headCommitId_);
        for (const auto& rev : revisions) {
            result.push_back(std::make_shared<Relation>(rev));
        }
    }
    return result;
}

bool
Snapshot::Impl::objectShouldBeLoaded(TObjectId objectId) const
{
    return !loadedObjectIds_.count(objectId);
}

} // namespace revision_meta
} // namespace contours
} // namespace wiki
} // namespace maps
