#pragma once

#include "categories.h"
#include "object.h"
#include "relation.h"
#include "revisions_loader.h"
#include "snapshot.h"

#include <boost/optional.hpp>
#include <memory>

namespace maps {
namespace wiki {
namespace contours {
namespace revision_meta {

typedef std::map<TCategoryId, TObjectIdSet> ObjectIdsByCategoryMap;

typedef std::shared_ptr<Object> ObjectPtr;
typedef std::list<ObjectPtr> ObjectPtrList;

typedef std::shared_ptr<Relation> RelationPtr;
typedef std::list<RelationPtr> RelationPtrList;

typedef std::map<TRevisionId, ObjectPtr> ObjectsByRevisionMap;
typedef std::map<TRevisionId, RelationPtr> RelationsByRevisionMap;

typedef std::map<TObjectId, TCommitIdSet> CommitsByObjectsMap;
typedef std::map<TObjectId, TObjectIdSet> RelationsByObjectsMap;

class Snapshot::Impl {
public:
    Impl(
        TCommitId headCommitId,
        Transaction& txn,
        TBranchId branchId,
        const Categories& categories);

    TCommitId headCommitId() const { return headCommitId_; }

    const Object*
    tryGetObjectRevision(TObjectId objectId, TCommitId maxCommitId) const;

    const Object&
    objectRevision(TObjectId objectId, TCommitId maxCommitId) const;

    const Object&
    existingObjectRevision(TObjectId objectId, TCommitId maxCommitId) const;

    TObjectIdSet slaves(TObjectId objectId, TCommitId commitId) const;
    TObjectIdSet masters(TObjectId objectId, TCommitId commitId) const;

    TObjectIdSet loadObjects(const TObjectIdSet& objectIds);

    void loadRelatives(RelationType type, const TObjectIdSet& objectIds);
    void loadRelativesRecursive(RelationType type, const TObjectIdSet& objectIds);

    /**
     * Adding data
     */

    /// Returns added object ids
    TObjectIdSet addObjects(const ObjectPtrList& objects);
    /// Loads and saves related objects
    void addRelations(const RelationPtrList& relations);

private:
    const Categories& categories() const { return loader_->categories(); }

    boost::optional<TCommitId>
    headObjectCommitId(TObjectId objectId, TCommitId commitId) const;

    boost::optional<TCommitId>
    headRelationCommitId(TObjectId relationId, TCommitId commitId) const;

    boost::optional<TCommitId>
    maxObjectCommitId(TObjectId objectId) const;

    const Relation*
    tryGetRelationRevision(TObjectId relId, TCommitId maxCommitId) const;

    const Relation&
    relationRevision(TObjectId relId, TCommitId maxCommitId) const;

    struct CompareRelations {
        bool operator () (const Relation* r1, const Relation* r2) const
        {
            return r1->relationId() < r2->relationId();
        }
    };

    typedef std::set<const Relation*, CompareRelations> RelationSet;

    /// Returns all relations cached for the object
    /// No checking whether all relations were loaded for this object
    RelationSet relations(RelationType type, TObjectId objectId, TCommitId commitId) const;

    TObjectIdSet allRelativeIds(
        RelationType type, const TObjectIdSet& objectIds) const;

    void addObject(const ObjectPtr& object);
    void addRelation(const RelationPtr& relation);

    void loadRelatedObjects(const RelationPtrList& relations);
    ObjectPtrList
    loadObjects(const ObjectIdsByCategoryMap& objectIdsByCategoryMap);

    RelationPtrList
    loadRelations(
        RelationType type,
        const ObjectIdsByCategoryMap& objectIdsByCategoryMap);

    bool objectShouldBeLoaded(TObjectId objectId) const;

    ObjectsByRevisionMap objects_;
    CommitsByObjectsMap objectCommitIds_;

    RelationsByRevisionMap relations_;
    CommitsByObjectsMap relationCommitIds_;

    // slave id -> relation id
    RelationsByObjectsMap relationsBySlaves_;
    // master id -> relation id
    RelationsByObjectsMap relationsByMasters_;

    TObjectIdSet loadedObjectIds_;
    TObjectIdSet objectsWithLoadedMasters_;
    TObjectIdSet objectsWithLoadedSlaves_;

    TCommitId headCommitId_;
    std::unique_ptr<RevisionsLoader> loader_;
};

} // namespace revision_meta
} // namespace contours
} // namespace wiki
} // namespace maps
