#include "../contour_object.h"
#include "../generalization.h"
#include "../pubsub.h"
#include "../tools.h"

#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <maps/libs/common/include/file_utils.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/unittest/config.h>
#include <yandex/maps/wiki/unittest/localdb.h>

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/serialization.h>

#include <library/cpp/testing/unittest/env.h>

#include <boost/algorithm/string/replace.hpp>
#include <boost/filesystem.hpp>
#include <boost/test/unit_test.hpp>

#include <csignal>

using namespace maps;
using namespace maps::wiki;
using namespace maps::wiki::contours;
using namespace maps::wiki::revision;
namespace fs = boost::filesystem;

namespace {

const std::string SERVICES_BASE_TEMPLATE = "/maps/wikimap/mapspro/cfg/services/services-base-template.xml";
const std::string CATEGORIES_DIR_PATH = "/maps/wikimap/mapspro/cfg/editor";

std::string
createTempServicesBaseXml()
{
    try {
        auto servicesBaseTemplate = maps::common::readFileToString(ArcadiaSourceRoot() + SERVICES_BASE_TEMPLATE);
        boost::replace_all(servicesBaseTemplate, "#CATEGORIES_DIR_PATH#", ArcadiaSourceRoot() + CATEGORIES_DIR_PATH);
        auto filepath = fs::temp_directory_path() / fs::unique_path();
        std::ofstream file(filepath.string());
        file << servicesBaseTemplate;
        return filepath.string();
    } catch (const std::exception& ex) {
        BOOST_REQUIRE_MESSAGE(false, ex.what());
    }
    return {};
}

class SetLogLevelFixture
{
public:
    SetLogLevelFixture()
    {
        log8::setLevel(log8::Level::FATAL);
    }
};

class ContoursDenormalizerFixture : public SetLogLevelFixture, public unittest::MapsproDbFixture
{
public:
    ContoursDenormalizerFixture()
        : unittest::MapsproDbFixture()
        , servicesBaseXmlPath_(createTempServicesBaseXml())
    {
    }
    const std::string& servicesBaseXmlPath() const { return servicesBaseXmlPath_; }

private:
    std::string servicesBaseXmlPath_;
};

const revision::Attributes REGION_AD_ASSIGNED_REL_ATTRS = {
    {"rel:master", "region"},
    {"rel:slave", "ad"},
    {"rel:role", "assigned"}};

const revision::Attributes REGION_AD_NEUTRAL_ASSIGNED_REL_ATTRS = {
    {"rel:master", "region"},
    {"rel:slave", "ad_neutral"},
    {"rel:role", "assigned_ad_neutral"}};

const revision::Attributes AD_FC_EL_PART_REL_ATTRS = {
    {"rel:master", "ad_fc"},
    {"rel:slave", "ad_el"},
    {"rel:role", "part"}};

const revision::Attributes AD_AD_FC_PART_REL_ATTRS = {
    {"rel:master", "ad"},
    {"rel:slave", "ad_fc"},
    {"rel:role", "part"}};

const revision::Attributes AD_NEUTRAL_FC_EL_PART_REL_ATTRS = {
    {"rel:master", "ad_neutral_fc"},
    {"rel:slave", "ad_neutral_el"},
    {"rel:role", "part"}};

const revision::Attributes AD_NEUTRAL_AD_NEUTRAL_FC_PART_REL_ATTRS = {
    {"rel:master", "ad_neutral"},
    {"rel:slave", "ad_neutral_fc"},
    {"rel:role", "part"}};



const revision::Attributes COMMIT_ATTRS = {{"description", "test"}};
const revision::UserID TEST_UID = 13;

template <typename GeomType = geolib3::Polyline2>
std::string wkt2wkb(const std::string& wkt)
{
    auto polyline = geolib3::WKT::read<GeomType>(wkt);
    std::stringstream wkb;
    geolib3::WKB::write(polyline, wkb);
    return wkb.str();
}


struct Data : revision::RevisionsGateway::NewRevisionData {
    Data(revision::DBID newObjectId, const revision::Attributes& attrs)
        : revision::RevisionsGateway::NewRevisionData
            ( revision::RevisionID::createNewID(newObjectId)
            , revision::ObjectData(attrs, std::nullopt, std::nullopt, std::nullopt)
            )
    {}

    Data
        ( revision::DBID newObjectId
        , const revision::Attributes& attrs
        , const revision::Wkb& geom
        )
        : revision::RevisionsGateway::NewRevisionData
            ( revision::RevisionID::createNewID(newObjectId)
            , revision::ObjectData(attrs, std::nullopt, geom, std::nullopt)
            )
    {}

    Data
        ( revision::DBID newObjectId
        , const revision::Attributes& attrs
        , const revision::RelationData& relationData
        )
        : revision::RevisionsGateway::NewRevisionData
            ( revision::RevisionID::createNewID(newObjectId)
            , revision::ObjectData(attrs, std::nullopt, std::nullopt, relationData)
            )
    {}

    Data(revision::RevisionID prevId, const revision::Wkb& geom)
        : revision::RevisionsGateway::NewRevisionData
            ( prevId
            , revision::ObjectData(std::nullopt, std::nullopt, geom, std::nullopt)
            )
    {}

    Data(revision::RevisionID prevId) // delete
        : revision::RevisionsGateway::NewRevisionData
            ( prevId
            , revision::ObjectData(std::nullopt, std::nullopt, std::nullopt, std::nullopt, true)
            )
    {}
};

typedef std::vector<Data> DataVector;

// add 2 contour objects with common rings/elements
const DataVector commit_1 = {
    {101, {{"cat:ad", "1"}}},
    {102, {{"cat:ad_fc", "1"}}},
    {103, {{"cat:ad_fc", "1"}}},
    {104, {{"cat:ad_el", "1"}}, wkt2wkb("LINESTRING(0 0, 0 10, 10 10)")},
    {105, {{"cat:ad_el", "1"}}, wkt2wkb("LINESTRING(10 10, 10 0, 0 0)")},
    {106, {{"cat:ad_el", "1"}}, wkt2wkb("LINESTRING(20 0, 20 10, 30 10)")},
    {107, {{"cat:ad_el", "1"}}, wkt2wkb("LINESTRING(30 10, 30 0, 20 0)")},
    {108, AD_AD_FC_PART_REL_ATTRS, RelationData{101, 102}},
    {109, AD_AD_FC_PART_REL_ATTRS, RelationData{101, 103}},
    {110, AD_FC_EL_PART_REL_ATTRS, RelationData{102, 104}},
    {111, AD_FC_EL_PART_REL_ATTRS, RelationData{102, 105}},
    {112, AD_FC_EL_PART_REL_ATTRS, RelationData{103, 106}},
    {113, AD_FC_EL_PART_REL_ATTRS, RelationData{103, 107}},

    {201, {{"cat:ad", "1"}}},
    {202, {{"cat:ad_fc", "1"}}},
    {203, {{"cat:ad_el", "1"}}, wkt2wkb("LINESTRING(40 0, 40 10, 50 10)")},
    {204, {{"cat:ad_el", "1"}}, wkt2wkb("LINESTRING(50 10, 50 0, 40 0)")},
    {205, AD_AD_FC_PART_REL_ATTRS, RelationData{201, 103}},
    {206, AD_AD_FC_PART_REL_ATTRS, RelationData{201, 202}},
    {207, AD_FC_EL_PART_REL_ATTRS, RelationData{202, 203}},
    {208, AD_FC_EL_PART_REL_ATTRS, RelationData{202, 204}},
};

// edit common element
const DataVector commit_2 = {
    {{106, 1}, wkt2wkb("LINESTRING(20 0, 30 10)")},
};

// add contour object with common ring
const DataVector commit_3 = {
    {301, {{"cat:ad", "1"}}},
    {302, {{"cat:ad_fc", "1"}}},
    {303, {{"cat:ad_el", "1"}}, wkt2wkb("LINESTRING(80 0, 80 10, 90 10)")},
    {304, {{"cat:ad_el", "1"}}, wkt2wkb("LINESTRING(90 10, 90 0, 80 0)")},
    {305, AD_AD_FC_PART_REL_ATTRS, RelationData{301, 202}},
    {306, AD_AD_FC_PART_REL_ATTRS, RelationData{301, 302}},
    {307, AD_FC_EL_PART_REL_ATTRS, RelationData{302, 303}},
    {308, AD_FC_EL_PART_REL_ATTRS, RelationData{302, 304}},
};

// remove contour object
const DataVector commit_4 = {
    {{101, 1}},
    {{102, 1}},
    {{108, 1}},
    {{109, 1}},
};

struct Test {
   DataVector objects;
   size_t expectedPolygons;
};

const Test TESTS[] =
    { {commit_1, 4}
    , {commit_2, 4}
    , {commit_3, 6}
    , {commit_4, 4}
    };

// add one object with two contours having areas 300 and 400
const DataVector areaCommit = {
    {401, {{"cat:ad", "1"}}},
    {402, {{"cat:ad_fc", "1"}}},
    {403, {{"cat:ad_fc", "1"}}},
    {404, {{"cat:ad_el", "1"}}, wkt2wkb("LINESTRING(0 0, 0 20, 10 20, 10 10, 20 10, 20 0, 0 0)")},
    {405, {{"cat:ad_el", "1"}}, wkt2wkb("LINESTRING(30 0, 30 20, 20 20, 20 30, 40 30, 40 0, 30 0)")},
    {406, AD_AD_FC_PART_REL_ATTRS, RelationData{401, 402}},
    {407, AD_AD_FC_PART_REL_ATTRS, RelationData{401, 403}},
    {408, AD_FC_EL_PART_REL_ATTRS, RelationData{402, 404}},
    {409, AD_FC_EL_PART_REL_ATTRS, RelationData{403, 405}},
};

const DataVector regionCommit = {
    {501, {{"cat:region", "1"}}, wkt2wkb<geolib3::Point2>("POINT(50 10)")},
    {502, REGION_AD_ASSIGNED_REL_ATTRS, RelationData{501, 201}},

    {601, {{"cat:ad_neutral", "1"}}},
    {602, {{"cat:ad_neutral_fc", "1"}}},
    {603, {{"cat:ad_neutral_el", "1"}}, wkt2wkb("LINESTRING(30 0, 30 20, 20 20, 20 30, 40 30, 40 0, 30 0)")},
    {604, AD_NEUTRAL_AD_NEUTRAL_FC_PART_REL_ATTRS, RelationData{601, 602}},
    {605, AD_NEUTRAL_FC_EL_PART_REL_ATTRS, RelationData{602, 603}},
    {606, REGION_AD_NEUTRAL_ASSIGNED_REL_ATTRS, RelationData{501, 601}},
};

const std::vector<std::pair<std::string, std::string>> serviceAttrsResult = {
    {"201", "(srv:is_part_of_region,1)"},
    {"201", "(srv:is_part_of_region_501,1)"},
    {"601", "(srv:is_part_of_region,1)"},
    {"601", "(srv:is_part_of_region_501,1)"},
};

const double COUNTOUR_AREAS[] = { 300, 400 };

void createCommit(pgpool3::Pool& pool, const DataVector& objects)
{
    auto txn = pool.masterWriteableTransaction();
    auto branch = revision::BranchManager(*txn).load(revision::TRUNK_BRANCH_ID);
    revision::RevisionsGateway gateway(*txn, branch);
    gateway.createCommit(objects, TEST_UID, COMMIT_ATTRS);
    txn->commit();
}

size_t queueSize(pgpool3::Pool& pool)
{
    auto txn = pool.slaveTransaction();
    return ::queueSize(*txn, *txn);
}

size_t selectCount(pgpool3::Pool& pool)
{
    return pool.slaveTransaction()->exec(sqlSelectCount())[0][0].as<size_t>();
}

size_t txidSnapshotXMin(pgpool3::Pool& pool)
{
    auto txn = pool.slaveTransaction();
    auto res = txn->exec("SELECT txid_snapshot_xmin(txid_current_snapshot())");
    return res[0][0].as<size_t>();
}

size_t txidSnapshotXMax(pgpool3::Pool& pool)
{
    auto txn = pool.slaveTransaction();
    auto res = txn->exec("SELECT txid_snapshot_xmax(txid_current_snapshot())");
    return res[0][0].as<size_t>();
}

void pubsubBarrier(pgpool3::Pool& pool, size_t transactionId)
{
    while (txidSnapshotXMin(pool) < transactionId) {
        sleep(1);
    }
}

} // namespace

BOOST_FIXTURE_TEST_SUITE(contours_tests, ContoursDenormalizerFixture)

BOOST_AUTO_TEST_CASE(test_gabarits_generalization)
{
    const std::string CATEGORY = "urban_areal";
    auto range = GeneralizationLoader::zoomRange(CATEGORY, 1000.);
    BOOST_CHECK(range.zmin);
    BOOST_CHECK_LE(range.zmin, range.zmax);
}

BOOST_AUTO_TEST_CASE(test_make_pool_holder)
{
    unittest::ConfigFileHolder configFileHolder(host(), port(), dbname(), user(), password(), "", servicesBaseXmlPath());
    wiki::common::ExtendedXmlDoc doc{configFileHolder.filepath()};
    makePoolHolder(doc, "dump-db", "dump-pool");
}

BOOST_AUTO_TEST_CASE(test_import)
{
    unittest::ConfigFileHolder configFileHolder(host(), port(), dbname(), user(), password(), "", servicesBaseXmlPath());
    wiki::common::ExtendedXmlDoc doc{configFileHolder.filepath()};
    auto holder = makePoolHolder(doc, DB_CORE, POOL_CORE);

    auto& pool = this->pool();
    Params params;
    for (auto& test: TESTS) {
        BOOST_CHECK_EQUAL(queueSize(pool), 0);
        createCommit(pool, test.objects);
        BOOST_CHECK_EQUAL(queueSize(pool), 1);

        pubsubBarrier(pool, txidSnapshotXMax(pool)); // todo:

        handleNextCommits(*holder, *holder, params);
        BOOST_CHECK_EQUAL(selectCount(pool), test.expectedPolygons);
    }
    BOOST_CHECK_EQUAL(queueSize(pool), 0);

    removeDifference(*holder, {});
    BOOST_CHECK_EQUAL(selectCount(pool), 0);
}

BOOST_AUTO_TEST_CASE(test_area)
{
    unittest::ConfigFileHolder configFileHolder(host(), port(), dbname(), user(), password(), "", servicesBaseXmlPath());
    wiki::common::ExtendedXmlDoc doc{configFileHolder.filepath()};
    auto holder = makePoolHolder(doc, DB_CORE, POOL_CORE);

    auto& pool = this->pool();
    Params params;
    params.minVerticesCount = 4; // to split 2 contours to 4

    createCommit(pool, areaCommit);
    pubsubBarrier(pool, txidSnapshotXMax(pool));
    handleNextCommits(*holder, *holder, params);

    BOOST_CHECK_EQUAL(selectCount(pool), 4); // 4 contours result
    for (const auto& row: pool.slaveTransaction()->exec(sqlSelectArea())) {
        const auto area = row[0].as<double>();
        BOOST_CHECK_NE
            ( std::find(std::begin(COUNTOUR_AREAS), std::end(COUNTOUR_AREAS), area)
            , std::end(COUNTOUR_AREAS)
            );
    }
}

auto selectServiceAttrs(pgpool3::Pool& pool)
{
    return pool.slaveTransaction()->exec(
        "SELECT DISTINCT object_id, each(service_attrs)"
        " FROM vrevisions_trunk.contour_objects_geom"
        " WHERE service_attrs IS NOT NULL");
}

BOOST_AUTO_TEST_CASE(test_region)
{
    unittest::ConfigFileHolder configFileHolder(host(), port(), dbname(), user(), password(), "", servicesBaseXmlPath());
    wiki::common::ExtendedXmlDoc doc{configFileHolder.filepath()};
    auto holder = makePoolHolder(doc, DB_CORE, POOL_CORE);

    auto& pool = this->pool();
    Params params;
    params.minVerticesCount = 4;
    createCommit(pool, commit_1);
    pubsubBarrier(pool, txidSnapshotXMax(pool));
    handleNextCommits(*holder, *holder, params);
    createCommit(pool, regionCommit);
    pubsubBarrier(pool, txidSnapshotXMax(pool));
    handleNextCommits(*holder, *holder, params);
    auto rows = selectServiceAttrs(pool);
    BOOST_REQUIRE_EQUAL(rows.size(), 4);
    for (const auto& row: selectServiceAttrs(pool)) {
        BOOST_CHECK_NE(
            std::find(serviceAttrsResult.begin(), serviceAttrsResult.end(),
                std::make_pair(row[0].as<std::string>(), row[1].as<std::string>())),
            serviceAttrsResult.end());
    }
}

BOOST_AUTO_TEST_SUITE_END()
