#include "helpers.h"

#include <boost/test/test_tools.hpp>

#include <fstream>
#include <regex>

namespace maps {
namespace wiki {
namespace contours {
namespace revision_meta {
namespace test {

namespace {

Categories makeCategories()
{
    Categories categories;

    categories.addCategory("ad", CategoryType::Complex);
    categories.addCategory("ad_fc", CategoryType::Complex);
    categories.addCategory("ad_el", CategoryType::Geom);
    categories.addRelation("ad_fc", "ad_el", "part");
    categories.addRelation("ad", "ad_fc", "part");

    categories.addCategory("rd", CategoryType::Complex);
    categories.addCategory("rd_el", CategoryType::Geom);
    categories.addRelation("rd", "rd_el", "part");

    categories.addCategory("hydro", CategoryType::Complex);
    categories.addCategory("hydro_fc", CategoryType::Complex);
    categories.addCategory("hydro_fc_el", CategoryType::Geom);
    categories.addCategory("hydro_ln_el", CategoryType::Geom);
    categories.addRelation("hydro", "hydro_fc", "fc_part");
    categories.addRelation("hydro", "hydro_ln_el", "ln_part");
    categories.addRelation("hydro_fc", "hydro_fc_el", "part");

    return categories;
}

} // namespace

const Categories& getCategories()
{
    static auto categories = makeCategories();
    return categories;
}

ObjectData::ObjectData(
        revision::DBID newObjectId,
        const revision::Attributes& attrs)
    : data_(
        revision::RevisionID::createNewID(newObjectId),
        revision::ObjectData(attrs, std::nullopt, std::nullopt, std::nullopt))
{}

ObjectData::ObjectData(
        revision::RevisionID prevId,
        const revision::Attributes& attrs)
    : data_(
        prevId,
        revision::ObjectData(attrs, std::nullopt, std::nullopt, std::nullopt))
{
    REQUIRE(prevId.valid(), "Revision id " << prevId << " is invalid");
}

ObjectData::ObjectData(
        revision::DBID newObjectId,
        const revision::Attributes& attrs,
        const revision::Wkb& geom)
    : data_(
        revision::RevisionID::createNewID(newObjectId),
        revision::ObjectData(attrs, std::nullopt, geom, std::nullopt))
{}

ObjectData::ObjectData(
        revision::RevisionID prevId,
        const revision::Wkb& geom)
    : data_(
        prevId,
        revision::ObjectData(std::nullopt, std::nullopt, geom, std::nullopt))
{
    REQUIRE(prevId.valid(), "Revision id " << prevId << " is invalid");
}

ObjectData::ObjectData(
        revision::DBID newObjectId,
        const revision::Attributes& attrs,
        const revision::RelationData& relationData)
    : data_(
        revision::RevisionID::createNewID(newObjectId),
        revision::ObjectData(attrs, std::nullopt, std::nullopt, relationData))
{}

ObjectData::ObjectData(
        revision::RevisionID prevId,
        const revision::RelationData& relationData)
    : data_(
        prevId,
        revision::ObjectData(std::nullopt, std::nullopt, std::nullopt, relationData))
{
    REQUIRE(prevId.valid(), "Revision id " << prevId << " is invalid");
}

ObjectData::ObjectData(
        revision::RevisionID prevId,
        bool deleted)
    : data_(
        prevId,
        revision::ObjectData(std::nullopt, std::nullopt, std::nullopt, std::nullopt, deleted))
{
    REQUIRE(prevId.valid(), "Revision id " << prevId << " is invalid");
}

namespace {

revision::Commit
createCommit(
    revision::RevisionsGateway& gateway,
    revision::UserID uid,
    const revision::Attributes& commitAttrs,
    const std::vector<ObjectData> objectsData)
{
    std::vector<revision::RevisionsGateway::NewRevisionData> newData;
    newData.reserve(objectsData.size());
    for (const auto& objectData : objectsData) {
        newData.push_back(objectData.data());
    }

    return gateway.createCommit(newData, uid, commitAttrs);
}

} // namespace

revision::Commit
createCommit(const CommitData& data, Transaction& txn)
{
    auto branch = revision::BranchManager(txn).load(data.branchId);
    revision::RevisionsGateway gateway(txn, branch);

    auto commit = createCommit(gateway, data.uid, data.attrs, data.objectsData);

    txn.commit();

    return commit;
}

void
createCommits(const CommitsData& data, Transaction& txn)
{
    auto branch = revision::BranchManager(txn).load(data.branchId);
    revision::RevisionsGateway gateway(txn, branch);

    for (const auto& commitData : data.commitsData) {
        createCommit(gateway, data.uid, commitData.attrs, commitData.objectsData);
    }

    txn.commit();
}

void
fillDatabase(
    pgpool3::Pool& pool,
    TBranchId branchId,
    revision::UserID uid,
    const revision::Attributes& commitAttrs,
    const std::vector<ObjectDataVector>& commitsData)
{
    CommitsData data {branchId, uid, {}};

    data.commitsData.reserve(commitsData.size());

    for (const auto& objects : commitsData) {
        data.commitsData.push_back({commitAttrs, objects});
    }

    auto txn = pool.masterWriteableTransaction();
    createCommits(data, *txn);
}

namespace {

template <class Set>
std::string
toString(const Set& ids)
{
    std::string res;
    for (auto id : ids) {
        res += (res.empty() ? "" : " ") + std::to_string(id);
    }
    return res;
}

template <class Set>
Set
idsDiff(const Set& ids1, const Set& ids2)
{
    Set res;
    std::set_difference(
        ids1.begin(), ids1.end(), ids2.begin(), ids2.end(),
        std::inserter(res, res.end()));
    return res;
}

template <class Set>
void
checkIds(
    const std::string& idsType,
    TRevisionId revisionId,
    const Set& receivedIds, const Set& expectedIds)
{
    auto unexpected = idsDiff<Set>(receivedIds, expectedIds);
    BOOST_CHECK_MESSAGE(
        unexpected.empty(),
        "Revision " << revisionId << " " << idsType << " unexpected: " << toString<Set>(unexpected));

    auto missing = idsDiff<Set>(expectedIds, receivedIds);
    BOOST_CHECK_MESSAGE(
        missing.empty(),
        "Revision " << revisionId << " " << idsType << " missing: " << toString<Set>(missing));
}

} // namespace

void
checkObjectSlaves(
    revision_meta::Snapshot& snapshot,
    TObjectId objectId, const TObjectIdSet& expectedSlaveIds)
{
    const auto& slaveIds = snapshot.slaves(objectId);
    checkIds(
        "slaves",
        TRevisionId(objectId, snapshot.headCommitId()),
        slaveIds,
        expectedSlaveIds);
}

void
checkObjectMasters(
    revision_meta::Snapshot& snapshot,
    TObjectId objectId, const TObjectIdSet& expectedMasterIds)
{
    const auto& masterIds = snapshot.masters(objectId);
    checkIds(
        "masters",
        TRevisionId(objectId, snapshot.headCommitId()),
        masterIds,
        expectedMasterIds);
}

} // namespace test
} // namespace revision_meta
} // namespace contours
} // namespace wiki
} // namespace maps
