#include <maps/wikimap/mapspro/services/tasks_realtime/src/edits_logger/lib/helpers.h>

#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <yandex/maps/wiki/pubsub/commit_consumer.h>
#include <yandex/maps/wiki/social/gateway.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/tasks/include/runner.h>

namespace common = maps::wiki::common;

namespace maps::wiki::edits_logger {

namespace {

const std::string CORE_DB_ID = "long-read";
const std::string CORE_POOL_ID = "long-read";
const std::string SOCIAL_DB_ID = "social";
const std::string SOCIAL_POOL_ID = "grinder";

const std::chrono::seconds WAIT_INTERVAL{5};
const size_t BATCH_SIZE_LIMIT = 1000;

} // namespace

void doWork(
    pgpool3::Pool& corePool,
    pgpool3::Pool& socialPool,
    const std::string& logFilename)
{
    DEBUG() << "check for new commits";

    try {
        auto socialTxn = socialPool.masterWriteableTransaction();
        social::Gateway socialGateway(*socialTxn);

        DEBUG() << "try to lock pubsub queue";
        pubsub::CommitConsumer consumer(
            *socialTxn, "EditsLogger", revision::TRUNK_BRANCH_ID);
        DEBUG() << "pubsub queue locked";

        consumer.setBatchSizeLimit(BATCH_SIZE_LIMIT);
        auto batch = [&] {
            auto coreTxn = corePool.slaveTransaction();
            return consumer.consumeBatch(*coreTxn);
        }();
        if (batch.empty()) {
            return;
        }

        DEBUG() << "logging " << batch.size() << " commits";
        auto logSize = logEditCommits(batch, socialGateway, logFilename);
        socialTxn->commit();
        INFO() << "wrote " << logSize << " lines";
    } catch (const pubsub::AlreadyLockedException&) {
        DEBUG() << "Pubsub queue already locked";
    } catch (const maps::Exception& ex) {
        ERROR() << "maps::Exception: " << ex;
    } catch (const std::exception& ex) {
        ERROR() << "std::exception: " << ex.what();
    }
}

void run(const common::ExtendedXmlDoc& config)
{
    auto logFileName = config.get<std::string>("/config/services/social/edits-log");

    common::PoolHolder corePoolHolder(config, CORE_DB_ID, CORE_POOL_ID);
    common::PoolHolder socialPoolHolder(config, SOCIAL_DB_ID, SOCIAL_POOL_ID);

    INFO() << "started, writing to " << logFileName;

    maps::wiki::tasks::Runner().run(
        [&] {
            doWork(corePoolHolder.pool(), socialPoolHolder.pool(), logFileName);
        },
        WAIT_INTERVAL
    );

    INFO() << "stopped";
}

} // namespace maps::wiki::edits_logger

int main(int argc, char** argv)
try {
    maps::cmdline::Parser parser;

    auto configPath = parser.file("config").help(
        "path to services configuration");
    auto syslogTag = parser.string("syslog-tag").help(
        "redirect log output to syslog with given tag");

    parser.parse(argc, argv);

    if (syslogTag.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslogTag));
    }

    auto configDocPtr = configPath.defined()
        ? std::make_unique<common::ExtendedXmlDoc>(configPath)
        : common::loadDefaultConfig();

    maps::wiki::edits_logger::run(*configDocPtr);

    return EXIT_SUCCESS;
} catch (const maps::Exception& ex) {
    FATAL() << "Worker failed: " << ex;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    FATAL() << "Worker failed: " << ex.what();
    return EXIT_FAILURE;
}
