#include <maps/wikimap/mapspro/services/tasks_realtime/src/edits_logger/lib/helpers.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/serialization.h>

#include <maps/wikimap/mapspro/libs/gdpr/include/user.h>
#include <yandex/maps/wiki/common/commit_properties.h>
#include <yandex/maps/wiki/common/date_time.h>
#include <yandex/maps/wiki/social/event.h>

#include <algorithm>
#include <chrono>
#include <fstream>
#include <sstream>
#include <utility>

namespace maps::wiki::edits_logger {

namespace {

const size_t PRECISION = 12;

const std::vector<std::string> LOGGED_ACTIONS = {
    common::COMMIT_PROPVAL_OBJECT_CREATED,
    common::COMMIT_PROPVAL_OBJECT_MODIFIED,
    common::COMMIT_PROPVAL_OBJECT_DELETED,
    common::COMMIT_PROPVAL_COMMIT_REVERTED,
};

const size_t CANONICAL_TS_LENGTH = std::string("0000-00-00 00:00:00").size();
const size_t CANONICAL_TZ_LENGTH = std::string("+03:00").size();

bool isLoggedAction(const std::string& action)
{
    return std::count(
        std::begin(LOGGED_ACTIONS), std::end(LOGGED_ACTIONS),
        action);
}

std::pair<std::string, std::string> statboxTsTzPair(
        const std::string& datetimeStr)
{
    auto canonical = common::canonicalDateTimeString(
        datetimeStr, common::WithTimeZone::Yes);

    ASSERT(canonical.size() == CANONICAL_TS_LENGTH + CANONICAL_TZ_LENGTH);

    std::pair<std::string, std::string> tsTzPair{
        canonical.substr(0, CANONICAL_TS_LENGTH),
        canonical.substr(CANONICAL_TS_LENGTH)};

    std::replace(
        std::begin(tsTzPair.first), std::end(tsTzPair.first),
        'T', ' ');

    auto delimPos = tsTzPair.second.find(':');
    if (delimPos != std::string::npos) {
        tsTzPair.second.erase(delimPos, 1);
    }

    return tsTzPair;
}

std::ostream& operator<<(
        std::ostream& os,
        const std::optional<geolib3::BoundingBox>& bbox)
{
    std::ostream::sentry s(os);
    if (bbox && s) {
        auto lowerCorner = geolib3::mercator2GeoPoint(bbox->lowerCorner());
        auto upperCorner = geolib3::mercator2GeoPoint(bbox->upperCorner());
        os << '['
            << '[' << lowerCorner.x() << ',' << lowerCorner.y() << ']' << ','
            << '[' << upperCorner.x() << ',' << upperCorner.y() << ']' << ']';
    }
    return os;
}

std::string toTskvValues(
    const std::optional<social::EventExtraData>& extraData)
{
    if (!extraData) {
        return "";
    }

    std::ostringstream result;

    if (extraData->ftTypeId) {
        result << "\tft_type_id=" << *extraData->ftTypeId;
    }
    if (extraData->businessRubricId) {
        result << "\tbusiness_rubric_id=" << *extraData->businessRubricId;
    }

    return result.str();
}

} // namespace

size_t logEditCommits(
    const std::vector<revision::DBID>& commitIds,
    social::Gateway& socialGateway,
    const std::string& logFilename)
{
    std::ofstream os;
    os.open(logFilename, std::ios_base::app);
    REQUIRE(os, "Could not open " << logFilename);

    os.precision(PRECISION);

    auto events = socialGateway.loadEditEventsWithExtraDataByCommitIds(
        {std::begin(commitIds), std::end(commitIds)});

    size_t linesWritten = 0;
    for (const auto& event : events) {
        if (!event.primaryObjectData() ||
            !event.commitData() ||
            !isLoggedAction(event.commitData()->action()))
        {
            continue;
        }

        auto tsTzPair = statboxTsTzPair(event.createdAt());

        os << "tskv"
           << "\ttskv_format=nmaps-edits-log"
           << "\ttimestamp=" << tsTzPair.first
           << "\ttimezone=" << tsTzPair.second
           << "\tpuid=" << gdpr::User(event.createdBy()).realUid()
           << "\tcommit_id=" << event.commitData()->commitId()
           << "\tbranch_id=" << event.commitData()->branchId()
           << "\taction=" << event.commitData()->action()
           << "\tobject_id=" << event.primaryObjectData()->id()
           << "\tobject_category=" << event.primaryObjectData()->categoryId()
           << "\tgeom=" << event.commitData()->bbox()
           << toTskvValues(event.extraData())
           << "\n";

        ++linesWritten;
    }
    os.flush();

    return linesWritten;
}

} // namespace maps::wiki::edits_logger
