#include <maps/wikimap/mapspro/services/tasks_realtime/src/edits_logger/lib/helpers.h>

#include <yandex/maps/wiki/common/batch.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/revision/commit.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <algorithm>


namespace maps::wiki::edits_logger::tool {

time_t unixTimeFromDate(pqxx::transaction_base& txn, const std::string& date)
{
    auto r = txn.exec(
        "SELECT"
        " to_timestamp(" + txn.quote(std::string(date)) + "::date::text, 'YYYY-MM-DD')");

    const auto& row = r[0];
    return chrono::convertToUnixTime(chrono::parseSqlDateTime(row[0].as<std::string>()));
}

void run(int argc, char** argv)
{
    cmdline::Parser parser;
    auto configFilename = parser.file("config")
        .help("services config file")
        .defaultValue("/etc/yandex/maps/wiki/services/services.xml");
    auto startDate = parser.string("start-date")
        .help("start date (including)");
    auto endDate = parser.string("end-date")
        .help("end date (excluding)");
    auto resultFilename = parser.string("file")
        .help("result tskv file");

    parser.parse(argc, argv);

    if (!resultFilename.defined()) {
        parser.printUsageAndExit(EXIT_FAILURE);
    }

    auto configPtr = configFilename.defined()
        ? std::make_unique<common::ExtendedXmlDoc>(configFilename)
        : common::loadDefaultConfig();

    common::PoolHolder corePoolHolder(*configPtr, "core", "core");
    common::PoolHolder socialPoolHolder(*configPtr, "social", "social");


    auto loadCommitIds = [&] {
        auto coreTxn = corePoolHolder.pool().masterReadOnlyTransaction();

        auto startTime = unixTimeFromDate(*coreTxn, startDate);
        auto endTime = unixTimeFromDate(*coreTxn, endDate);

        auto filter =
            revision::filters::CommitAttr::isTrunk() &&
            revision::filters::CommitCreationTime() >= startTime &&
            revision::filters::CommitCreationTime() < endTime;

        auto ids = revision::Commit::loadIds(*coreTxn, filter);
        std::sort(ids.begin(), ids.end());
        return ids;
    };

    INFO() << "Fetching commit ids";
    auto commitIds = loadCommitIds();
    INFO() << "Fetched " << commitIds.size() << " commit ids";

    auto socialTxn = socialPoolHolder.pool().masterReadOnlyTransaction();
    social::Gateway socialGateway(*socialTxn);

    constexpr size_t BATCH_SIZE = 1000;
    auto size = 0;
    common::applyBatchOp<decltype(commitIds)>(
        commitIds,
        BATCH_SIZE,
        [&](const auto& ids) {
            size += logEditCommits(ids, socialGateway, resultFilename);
        }
    );
    INFO() << "Saved " << size << " into file " << resultFilename;
}

} // namespace maps::wiki::edits_logger::tool

int main(int argc, char** argv) {
    try {
        maps::wiki::edits_logger::tool::run(argc, argv);
        return EXIT_SUCCESS;
    } catch (const maps::Exception& ex) {
        ERROR() << "maps::Exception: " << ex;
    } catch (const std::exception& ex) {
        ERROR() << "std::exception: " << ex.what();
    }
    return EXIT_FAILURE;
}
