#include <maps/wikimap/mapspro/services/tasks_realtime/src/export_cleaner/lib/cleaner.h>

#include <maps/wikimap/mapspro/services/tasks_export/src/export_worker/lib/config.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pg_advisory_lock_ids.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <array>
#include <memory>

namespace common = maps::wiki::common;
namespace ds = maps::wiki::mds_dataset;
namespace mlog = maps::log8;

namespace {
const size_t MAX_AGE_DAYS = 183;
const size_t NUM_TOTAL_TO_KEEP = 20;
const size_t NUM_TESTED_TO_KEEP = 10;
const size_t NUM_INCOMPLETE_TO_KEEP = 20;
const size_t NUM_DELETED_TO_KEEP = 0;

const std::array SUBSETS = {
    ds::Subset::Ymapsdf,
    ds::Subset::Masstransit,
    ds::Subset::Poi,
};
} // namespace

namespace maps::wiki::exporter::cleaner {

void cleanSubset(
    pgpool3::Pool& pool,
    const common::ExtendedXmlDoc& configXml,
    ds::Subset subset)
{
    INFO() << "Cleaning subset " << subset << " started";

    for (const auto& region: loadAllRegions(pool, subset)) {
        auto mdsConfig = makeMdsConfig(configXml);
        updateMdsPath(mdsConfig, region);
        mds::Mds mds(mdsConfig);

        deleteRedundantDatasets(pool, mds, ds::DatasetStatus::Available, subset, region, NUM_TOTAL_TO_KEEP, NUM_TESTED_TO_KEEP);
        deleteRedundantDatasets(pool, mds, ds::DatasetStatus::Incomplete, subset, region, NUM_INCOMPLETE_TO_KEEP);
        deleteRedundantDatasets(pool, mds, ds::DatasetStatus::Deleting, subset, region, NUM_DELETED_TO_KEEP);
    }

    auto mdsConfig = makeMdsConfig(configXml);
    mds::Mds mds(mdsConfig);
    deleteObsoleteDatasets(pool, mds, subset, MAX_AGE_DAYS);

    INFO() << "Cleaning subset " << subset << " finished";
}

} // maps::wiki::exporter::cleaner

namespace cleaner = maps::wiki::exporter::cleaner;

int main(int argc, char* argv[]) try
{
    maps::cmdline::Parser parser;
    auto workerConfig = parser.string("config")
        .help("path to worker configuration");
    auto syslogTag = parser.string("syslog-tag")
        .help("redirect log output to syslog with given tag");

    parser.parse(argc, argv);

    if (syslogTag.defined()) {
        mlog::setBackend(mlog::toSyslog(syslogTag));
    }

    std::unique_ptr<common::ExtendedXmlDoc> configXmlPtr;
    if (workerConfig.defined()) {
        configXmlPtr.reset(new common::ExtendedXmlDoc(workerConfig));
    } else {
        configXmlPtr = common::loadDefaultConfig();
    }

    common::PoolHolder dbHolder(*configXmlPtr, "core", "grinder");
    auto& pool = dbHolder.pool();

    maps::pgp3utils::PgAdvisoryXactMutex guard(
        pool,
        static_cast<int64_t>(common::AdvisoryLockIds::EXPORT_CLEANER));
    if (!guard.try_lock()) {
        INFO() << "Export Cleaner is already running. (DB is locked)";
        return EXIT_SUCCESS;
    }

    for (auto subset : SUBSETS) {
        cleaner::cleanSubset(pool, *configXmlPtr, subset);
    }
    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    ERROR() << "Export cleaner failed: " << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    ERROR() << "Export cleaner failed: " << e.what();
    return EXIT_FAILURE;
}

