#include "maps/wikimap/mapspro/services/tasks_realtime/src/export_cleaner/lib/cleaner.h"

#include <yandex/maps/mds/mds.h>
#include <yandex/maps/mrc/unittest/local_server.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/mds_dataset/dataset_gateway.h>
#include <yandex/maps/wiki/mds_dataset/export_metadata.h>
#include <yandex/maps/wiki/unittest/localdb.h>

#include <boost/test/unit_test.hpp>

#include <chrono>
#include <filesystem>
#include <fstream>
#include <sstream>

namespace ds = maps::wiki::mds_dataset;
namespace fs = std::filesystem;

namespace maps {
namespace wiki {
namespace exporter {
namespace test {

using Reader = ds::DatasetReader<ds::ExportMetadata>;
using Writer = ds::DatasetWriter<ds::ExportMetadata>;
using ds::DatasetStatus;
using ds::IsTested;
using ds::Subset;
using ds::Region;
using ds::NO_REGION;

const Region CIS1_REGION = "cis1";
const Region CIS2_REGION = "cis2";
const Region AAO_REGION = "aao";


class Fixture: public unittest::MapsproDbFixture {
public:
    Fixture()
        : mdsServer_()
        , mds_(makeMdsConfig(mdsServer_.getMdsPort()))
    {}

    mds::Mds& mds()
    {
        return mds_;
    }

private:
    const std::string MDS_HOST = "127.0.0.1";
    const std::string NAMESPACE = "mpro-dataset";
    const std::string AUTH_HEADER = "Basic bXByby1kYXRhc2V0OjhmZTU5ZGNjMzUzMzc4ODdkNzIxOWE4M2IwNWI4ZWRk";

    mrc::unittest::MdsStubFixture mdsServer_;
    mds::Mds mds_;

    mds::Configuration makeMdsConfig(std::uint16_t mdsPort)
    {
        return
        mds::Configuration(MDS_HOST, NAMESPACE, AUTH_HEADER)
            .setTimeout(std::chrono::milliseconds(5000))
            .setWritePort(mdsPort)
            .setReadPort(mdsPort);
    }
};


struct TestMeta {
    Region region;
    Subset subset;
    DatasetStatus status;
    IsTested isTested;
};


void createFile(const std::string& filename, const std::string& text)
{
    std::ofstream file(filename);
    file << text;
}


/// Creates datasets, so that the first dataset is the most recent.
///
/// @note It is easier to compare (visually) lists sorted in the same
/// direction. Moreover, the DatasetReader returns datasets sorted by time from
/// the most recent to the eldest one. Therefore, this function creates the
/// first dataset as the most recent and the last as the eldest.
void createDatasets(
    mds::Mds& mds,
    pgpool3::Pool& pool,
    const std::vector<TestMeta>& meta)
{
    const std::string filename = "file.txt";
    createFile(filename, "text");

    Writer writer(mds, pool);
    const auto now = ds::Timestamp::clock::now();
    size_t metaIx{0};

    for (const auto& info: meta) {
        writer.createDataset(
            {
                ds::BasicMetadata(
                    std::to_string(metaIx),
                    info.region,
                    info.status,
                    now - std::chrono::seconds(metaIx)
                ),
                info.subset,
                info.isTested
            },
            { filename }
        );
        ++metaIx;
    }
}


std::optional<std::string> checkDatasets(
    pgpool3::Pool& pool,
    const std::vector<DatasetStatus>& statuses)
{
    const auto datasets = Reader::datasets(*pool.slaveTransaction());

    if (datasets.size() != statuses.size()) {
        return
            "The datasets size " + std::to_string(datasets.size()) + " (got) != " +
            std::to_string(statuses.size()) + " (expected).";
    }

    std::ostringstream oss;
    for (size_t statusIx{0}; statusIx < statuses.size(); ++statusIx) {
        if (datasets[statusIx].metadata().basic().status() != statuses[statusIx]) {
            oss << "The dataset status "
                << datasets[statusIx].metadata().basic().status() << " (got) != "
                << statuses[statusIx] << " (expected), "
                << "in the dataset " << statusIx << ".\n";
        }
    }
    if (!oss.str().empty()) {
        return oss.str();
    }

    return std::nullopt;
}


#define CHECK_DATASETS(pool, ...)                       \
    do {                                                \
        auto result = checkDatasets(pool, __VA_ARGS__); \
        if (result) {                                   \
            BOOST_ERROR(*result);                    \
        }                                               \
    } while (0)


BOOST_FIXTURE_TEST_SUITE(main, Fixture)

BOOST_AUTO_TEST_CASE(test_cleaner_1)
{
    createDatasets(
        mds(), pool(),
        {
            {NO_REGION, Subset::Service, DatasetStatus::Incomplete, IsTested::No},
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Incomplete, IsTested::No},
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Incomplete, IsTested::No},
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No}
        });

    // Delete all but one Available Ymapsdf dataset
    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, NO_REGION, 1);
    CHECK_DATASETS(
        pool(),
        {
            DatasetStatus::Incomplete,
            DatasetStatus::Incomplete,
            DatasetStatus::Incomplete,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted
        });

    // Delete all Incomplete Ymapsdf datasets
    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Incomplete, Subset::Ymapsdf, NO_REGION, 0);
    CHECK_DATASETS(
        pool(),
        {
            DatasetStatus::Incomplete,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted
        });
}


BOOST_AUTO_TEST_CASE(shouldLeaveTestedAndNew_withGap) {
    createDatasets(
        mds(), pool(),
        {
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // new
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // new
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // gap
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}, // tested
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // old
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}  // old
        });

    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, NO_REGION, 2, 1);
    CHECK_DATASETS(
        pool(),
        {
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted
        });
}


BOOST_AUTO_TEST_CASE(shouldLeaveTestedAndNew_adjacent) {
    createDatasets(
        mds(), pool(),
        {
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // new
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // new
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}, // tested
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // old
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}  // old
        });

    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, NO_REGION, 2, 1);
    CHECK_DATASETS(
        pool(),
        {
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted
        });
}


BOOST_AUTO_TEST_CASE(shouldLeaveTestedAndNew_overlaped) {
    createDatasets(
        mds(), pool(),
        {
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // new
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}, // new, tested
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}, // tested
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // old
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}  // old
        });

    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, NO_REGION, 2, 1);
    CHECK_DATASETS(
        pool(),
        {
            DatasetStatus::Available,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted
        });
}


BOOST_AUTO_TEST_CASE(shouldLeaveTestedAndNew_equal) {
    createDatasets(
        mds(), pool(),
        {
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}, // new, tested
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}, // new, tested
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // old
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}  // old
        });

    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, NO_REGION, 2, 2);
    CHECK_DATASETS(
        pool(),
        {
            DatasetStatus::Available,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted
        });
}


BOOST_AUTO_TEST_CASE(shouldLeaveTestedAndNew_testedInsideNew) {
    createDatasets(
        mds(), pool(),
        {
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // new
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}, // new, tested
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // new
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // old
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}  // old
        });

    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, NO_REGION, 3, 1);
    CHECK_DATASETS(
        pool(),
        {
            DatasetStatus::Available,
            DatasetStatus::Available,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted
        });
}


BOOST_AUTO_TEST_CASE(shouldLeaveTestedAndNew_noTested) {
    createDatasets(
        mds(), pool(),
        {
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No}, // new
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No}, // new
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No}  // old
        });

    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, NO_REGION, 2, 1);
    CHECK_DATASETS(
        pool(),
        {
            DatasetStatus::Available,
            DatasetStatus::Available,
            DatasetStatus::Deleted
        });
}


BOOST_AUTO_TEST_CASE(shouldThrow_ifKeepTestedBiggerThanKeepTotal) {
    createDatasets(
        mds(), pool(),
        {{NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No}});

    BOOST_CHECK_THROW(
        cleaner::deleteRedundantDatasets(
            pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, NO_REGION, 0, 2),
        LogicError
    );
}

BOOST_AUTO_TEST_CASE(test_cleaner_multiregion)
{
    createDatasets(
        mds(), pool(),
        {
            {NO_REGION, Subset::Service, DatasetStatus::Incomplete, IsTested::No},
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes},
            {CIS1_REGION, Subset::Ymapsdf, DatasetStatus::Incomplete, IsTested::No},
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes},
            {CIS1_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},
            {CIS2_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},
            {NO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes},   // 6, delete
            {CIS2_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // 7, delete
            {CIS2_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // 8, delete
            {CIS2_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes},
            {CIS2_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},  // 10, delete
            {CIS2_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::Yes}, // 11, delete
            {AAO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},
            {CIS2_REGION, Subset::Ymapsdf, DatasetStatus::Incomplete, IsTested::No},
            {AAO_REGION, Subset::Ymapsdf, DatasetStatus::Available, IsTested::No},
        });

    // Delete all but two Available and one Tested Ymapsdf dataset
    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, NO_REGION, 2, 1);
    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, CIS1_REGION, 2, 1);
    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, CIS2_REGION, 2, 1);
    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Available, Subset::Ymapsdf, AAO_REGION, 2, 1);
    CHECK_DATASETS(
        pool(),
        {
            DatasetStatus::Incomplete,
            DatasetStatus::Available,
            DatasetStatus::Incomplete,
            DatasetStatus::Available,
            DatasetStatus::Available,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted,
            DatasetStatus::Available,
            DatasetStatus::Incomplete,
            DatasetStatus::Available,
        });

    // Delete all Incomplete Ymapsdf datasets
    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Incomplete, Subset::Ymapsdf, NO_REGION, 0);
    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Incomplete, Subset::Ymapsdf, CIS1_REGION, 0);
    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Incomplete, Subset::Ymapsdf, CIS2_REGION, 0);
    cleaner::deleteRedundantDatasets(
        pool(), mds(), DatasetStatus::Incomplete, Subset::Ymapsdf, AAO_REGION, 0);
    CHECK_DATASETS(
        pool(),
        {
            DatasetStatus::Incomplete,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Available,
            DatasetStatus::Available,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Deleted,
            DatasetStatus::Available,
            DatasetStatus::Deleted,
            DatasetStatus::Available,
        });
}


BOOST_AUTO_TEST_SUITE_END()

} // test
} // exporter
} // wiki
} // maps
