#include <maps/wikimap/mapspro/services/tasks_realtime/src/mrc_pedestrian_regions_logger/lib/common.h>
#include <maps/wikimap/mapspro/services/tasks_realtime/src/mrc_pedestrian_regions_logger/lib/log_mrc_pedestrian_regions.h>
#include <maps/wikimap/mapspro/services/tasks_realtime/src/mrc_pedestrian_regions_logger/lib/yt_table_writer.h>

#include <maps/libs/chrono/include/format.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/default_config.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/pgpool3_helpers.h>
#include <maps/wikimap/mapspro/libs/pubsub/include/yandex/maps/wiki/pubsub/commit_consumer.h>

#include <library/cpp/getopt/last_getopt.h>
#include <mapreduce/yt/interface/client.h>

#include <chrono>
#include <exception>
#include <string>
#include <thread>


namespace {
const std::chrono::seconds SLEEP_PERIOD_SEC{30};
} // namespace


using namespace maps;
using namespace maps::wiki;
using namespace maps::wiki::tasks::log_mrc_pedestrian_regions;


struct Options {
    std::string syslogTag;
    TString ytServer;
    TString ytLogPath;
    size_t batchSize;

    Options(int argc, char* argv[])
    {
        auto opts = NLastGetopt::TOpts::Default();
        opts.SetTitle(
            "Pubsub process that writes MRC pedestrian regions changes into an YT table.\n\n"
            "See more at https://docs.yandex-team.ru/nmaps/services/tasks_realtime/mrc_pedestrian_regions_logger"
        );
        opts.AddLongOption("syslog-tag", "Redirects log output to syslog with the given tag.")
            .StoreResult(&syslogTag).RequiredArgument();
        opts.AddLongOption("yt-server", "YT server.")
            .StoreResult(&ytServer).RequiredArgument().DefaultValue("hahn");
        opts.AddLongOption("yt-log-path", "MRC pedestrian regions changes log.")
            .StoreResult(&ytLogPath).RequiredArgument();
        opts.AddLongOption("batch-size", "Pubsub consumer batch size.")
            .StoreResult(&batchSize).DefaultValue(1000);
        NLastGetopt::TOptsParseResult(&opts, argc, argv);
    }
};


auto init(const Options& options)
{
    if (!options.syslogTag.empty()) {
        maps::log8::setBackend(maps::log8::toSyslog(options.syslogTag));
    }

    const auto configPtr = wiki::common::loadDefaultConfig();
    wiki::common::PoolHolder corePoolHolder{*configPtr, LONG_READ_DB_ID, LONG_READ_POOL_ID};

    NYT::JoblessInitialize();
    auto ytClient = NYT::CreateClient(options.ytServer);

    return std::make_tuple(
        std::move(corePoolHolder),
        std::move(ytClient)
    );
}


void
consumeCommits(
    wiki::common::PoolHolder& corePoolHolder,
    NYT::IClient& ytClient,
    const TString& ytLogPath,
    size_t batchSize)
try {
    auto coreWriteTxn = corePoolHolder.pool().masterWriteableTransaction();
    pubsub::CommitConsumer consumer{*coreWriteTxn, PUBSUB_CONSUMER_ID, revision::TRUNK_BRANCH_ID};
    consumer.setBatchSizeLimit(batchSize);

    auto coreReadTxn = corePoolHolder.pool().slaveTransaction();
    const auto commitIds = consumer.consumeBatch(*coreReadTxn);
    if (!commitIds.empty()) {
        auto ytTxn = ytClient.StartTransaction();
        YtTableWriter ytTableWriter{ytTxn, ytLogPath};

        INFO() << "Process " << commitIds.size() << " commits.";
        logMrcPedestrianRegions(*coreReadTxn, ytTableWriter, commitIds);
        INFO() << "Commits processed.";

        ytTableWriter.finish();
        ytTxn->Commit();
    }

    coreWriteTxn->commit();

    if (commitIds.size() < batchSize) {
        INFO() << "All commits processed sleep for " << chrono::format(SLEEP_PERIOD_SEC) << ".";
        std::this_thread::sleep_for(SLEEP_PERIOD_SEC);
    }
} catch (const pubsub::AlreadyLockedException& ex) {
    INFO() << "Commits queue already locked.";
    std::this_thread::sleep_for(SLEEP_PERIOD_SEC);
}


int main(int argc, char* argv[])
try {
    const Options options{argc, argv};
    auto [corePoolHolder, ytClient] = init(options);

    do {
        consumeCommits(corePoolHolder, *ytClient, options.ytLogPath, options.batchSize);
    } while (true);

    return EXIT_SUCCESS;
} catch (const Exception& ex) {
    FATAL() << "Worker failed: " << ex;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    FATAL() << "Worker failed: " << ex.what();
    return EXIT_FAILURE;
}
