#include <maps/wikimap/mapspro/services/tasks_realtime/src/mrc_pedestrian_regions_logger/lib/log_mrc_pedestrian_regions.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/exception.h>
#include <maps/wikimap/mapspro/libs/revision/include/yandex/maps/wiki/revision/filters.h>
#include <maps/wikimap/mapspro/libs/revision/include/yandex/maps/wiki/revision/objectrevision.h>
#include <maps/wikimap/mapspro/libs/revision/include/yandex/maps/wiki/revision/revisionsgateway.h>

#include <mapreduce/yt/interface/io.h>

#include <set>
#include <string>


namespace maps::wiki::tasks::log_mrc_pedestrian_regions {

namespace {

using CommitIdObjectId = std::tuple<revision::DBID, revision::DBID>;

namespace attr {
const std::string CAT_MRC_PEDESTRIAN_REGION = "cat:mrc_pedestrian_region";
const std::string COMPANY_ID                = "mrc_pedestrian_region:company_id";
const std::string IS_INDOOR                 = "mrc_pedestrian_region:is_indoor";
const std::string IS_PANORAMIC              = "mrc_pedestrian_region:is_panoramic";
const std::string IS_TESTING                = "mrc_pedestrian_region:is_testing";
const std::string NAME                      = "mrc_pedestrian_region:name";
const std::string PEDESTRIAN_LOGIN          = "mrc_pedestrian_region:pedestrian_login";
const std::string PEDESTRIAN_LOGIN_UID      = "mrc_pedestrian_region:pedestrian_login_uid";
const std::string STATUS                    = "mrc_pedestrian_region:status";
}

const std::string DEFAULT_STATUS = "draft";

namespace col {
const TString AD_ID            = "ad_id";
const TString COMPANY_ID       = "company_id";
const TString IS_INDOOR        = "is_indoor";
const TString IS_PANORAMIC     = "is_panoramic";
const TString IS_TESTING       = "is_testing";
const TString NAME             = "name";
const TString OBJECT_ID        = "object_id";
const TString PEDESTRIAN_LOGIN = "pedestrian_login";
const TString PEDESTRIAN_UID   = "pedestrian_uid";
const TString STATUS           = "status";
const TString TIMESTAMP        = "timestamp";
}


namespace filter {

auto mrcPedestrianRegionAssociatedWithRelations()
{
    return
        revision::filters::ObjRevAttr::isRelation() &&
        revision::filters::Attr("rel:role").equals("mrc_pedestrian_region_associated_with") &&
        !revision::filters::Geom::defined();
}

auto mrcPedestrianRegionObjects()
{
    return
        revision::filters::ObjRevAttr::isNotRelation() &&
        revision::filters::Attr("cat:mrc_pedestrian_region").defined() &&
        revision::filters::Geom::defined();
}

}


revision::Revisions
readMrcPedestrianRegionRevisions(
    const revision::Reader& reader,
    const CommitIdsVec& commitIds)
{
    return reader.loadRevisions(
        revision::filters::CommitAttr::id().in(commitIds) &&
        (
            filter::mrcPedestrianRegionObjects() ||
            filter::mrcPedestrianRegionAssociatedWithRelations()
        )
    );
}


std::set<CommitIdObjectId>
getCommitIdMrcRegionObjectIdPairs(const revision::Revisions& revisions)
{
    std::set<CommitIdObjectId> result;

    for (const auto& revision: revisions) {
        switch (revision.type()) {
        case revision::RevisionType::RegularObject:
            result.emplace(revision.id().commitId(), revision.id().objectId());
            break;
        case revision::RevisionType::Relation:
            result.emplace(revision.id().commitId(), revision.data().relationData->slaveObjectId());
            break;
        }
    }

    return result;
}


// See https://en.cppreference.com/w/cpp/language/if
template<typename>
[[maybe_unused]] inline constexpr bool dependent_false = false;

template<typename T>
NYT::TNode
getAttr(
    const revision::Attributes& attributes,
    const std::string& attrName,
    const std::optional<T> defaultValue = {})
{
    const auto attrIt = attributes.find(attrName);
    if (attrIt == attributes.cend()) {
        if (defaultValue) {
            return NYT::TNode(*defaultValue);
        }
        return NYT::TNode::CreateEntity();
    }

    const auto value = attrIt->second;
    if constexpr (std::is_same<T, std::string>::value) {
        return NYT::TNode(value);
    } else if constexpr (std::is_same<T, bool>::value) {
        return NYT::TNode(value == "1");
    } else if constexpr (std::is_same<T, long>::value) {
        return NYT::TNode(std::atol(value.c_str()));
    } else {
        static_assert(dependent_false<T>, "Unsupported type");
    }
}


revision::Attributes
getRegionAttrs(
    const revision::Snapshot& snapshot,
    revision::DBID regionObjectId)
{
    const auto regionRevision = snapshot.objectRevision(regionObjectId);
    REQUIRE(regionRevision, "Can't find revision for commit " << snapshot.maxCommitId() << " and objectId " << regionObjectId << ".");
    REQUIRE(regionRevision->data().attributes, "Revision " << regionRevision->id() << " has no attributes.");

    return *regionRevision->data().attributes;
}


NYT::TNode
getAdId(
    const revision::Snapshot& snapshot,
    revision::DBID regionObjectId)
{
    const auto adRelations = snapshot.loadMasterRelations(
        {regionObjectId},
        filter::mrcPedestrianRegionAssociatedWithRelations() && revision::filters::ObjRevAttr::isNotDeleted()
    );
    REQUIRE(adRelations.size() <= 1, "MRC pedestrian region " << regionObjectId << " is connected with more than one (" << adRelations.size() << ") AD.");

    if (adRelations.size() == 1) {
        const auto& relationData = adRelations.cbegin()->data().relationData;
        REQUIRE(relationData, "Relation " << adRelations.cbegin()->id() << "has no data.");
        return NYT::TNode(relationData->masterObjectId());
    }

    return NYT::TNode::CreateEntity();
}


NYT::TNode
getCreatedAt(
    pqxx::transaction_base &coreReadTxn,
    revision::DBID commitId)
{
    const auto commit = revision::Commit::load(coreReadTxn, commitId);
    const auto createdAt = chrono::parseSqlDateTime(commit.createdAt());
    return NYT::TNode(
        std::chrono::duration_cast<std::chrono::microseconds>(createdAt.time_since_epoch()).count()
    );
}


void
logStatus(
    const revision::RevisionsGateway& revisionGw,
    IYtTableWriter& ytTableWriter,
    const revision::Revisions& revisions)
{
    for (const auto& [commitId, regionObjectId]: getCommitIdMrcRegionObjectIdPairs(revisions)) {
        auto snapshot = revisionGw.snapshot(commitId);
        const auto regionAttrs = getRegionAttrs(snapshot, regionObjectId);

        ytTableWriter.addRow(
            NYT::TNode()
            (col::TIMESTAMP,        getCreatedAt(revisionGw.work(), commitId))
            (col::OBJECT_ID,        NYT::TNode(regionObjectId))
            (col::NAME,             getAttr<std::string>(regionAttrs, attr::NAME))
            (col::COMPANY_ID,       getAttr<std::string>(regionAttrs, attr::COMPANY_ID))
            (col::STATUS,           getAttr<std::string>(regionAttrs, attr::STATUS, DEFAULT_STATUS))
            (col::IS_INDOOR,        getAttr<bool>(regionAttrs, attr::IS_INDOOR))
            (col::IS_PANORAMIC,     getAttr<bool>(regionAttrs, attr::IS_PANORAMIC))
            (col::IS_TESTING,       getAttr<bool>(regionAttrs, attr::IS_TESTING))
            (col::PEDESTRIAN_UID,   getAttr<revision::UserID>(regionAttrs, attr::PEDESTRIAN_LOGIN_UID))
            (col::PEDESTRIAN_LOGIN, getAttr<std::string>(regionAttrs, attr::PEDESTRIAN_LOGIN))
            (col::AD_ID,            getAdId(snapshot, regionObjectId))
        );
    }
}

} // namespace


void
logMrcPedestrianRegions(
    pqxx::transaction_base &coreReadTxn,
    IYtTableWriter& ytTableWriter,
    const CommitIdsVec& commitIds)
{
    if (commitIds.empty()) {
        return;
    }

    const auto revisionGw = revision::RevisionsGateway(coreReadTxn);
    const auto regionsRevisions = readMrcPedestrianRegionRevisions(revisionGw.reader(), commitIds);
    if (regionsRevisions.empty()) {
        return;
    }

    logStatus(revisionGw, ytTableWriter, regionsRevisions);
}

} // namespace maps::wiki::tasks::log_mrc_pedestrian_regions
