#include "tasks_generator.h"
#include "strings.h"

#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/revision/diff.h>
#include <yandex/maps/wiki/revision/filters.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/pubsub/commit_consumer.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/streetview/backoffice/tools/mrc_targets/include/generate_targets.h>

#include <algorithm>
#include <map>
#include <set>
#include <string>

namespace rev = maps::wiki::revision;
namespace rf = maps::wiki::revision::filters;
using IdSet = std::set<rev::DBID>;

namespace maps::wiki::tasks::mrc_pedestrian {

namespace {

constexpr rev::DBID PUBSUB_BRANCH_ID = rev::TRUNK_BRANCH_ID;
const std::string PUBSUB_CONSUMER_ID = "MrcPedestrianTasksGenerator";
const std::string ATTR_CAT_MRC_PEDESTRIAN_REGION = "cat:mrc_pedestrian_region";

const std::string UGC_TASK_PREFIX = "mrc_pedestrian:";
constexpr double MAX_AREA_FOR_TARGET_GENERATION = 1000000.0; // square meters

bool isRegionCreated(const rev::ObjectRevisionDiff& revDiff)
{
    return !revDiff.oldId().valid();
}

bool isRegionDeleted(const rev::ObjectRevisionDiff& revDiff)
{
    return revDiff.data().deleted && revDiff.data().deleted->after;
}

bool shouldShowTaskInStatus(const std::string& status)
{
    return status == STATUS_CAN_START;
}

std::string makeUgcTaskId(rev::DBID objectId)
{
    return UGC_TASK_PREFIX + std::to_string(objectId);
}

auto getKeys(const auto& map)
{
    std::vector<typename std::decay_t<decltype(map)>::key_type> keys;
    keys.reserve(map.size());
    for (const auto& [key, _] : map) {
        keys.push_back(key);
    }
    return keys;
}

} // anonymous namespace

TasksGenerator::TasksGenerator(
        pgpool3::Pool& pool,
        UgcBackofficeClient ugcBackofficeClient)
    : pool_(pool)
    , ugcBackofficeClient_(std::move(ugcBackofficeClient))
{
}

size_t TasksGenerator::consumeNewCommits()
try {
    INFO() << "Start consuming changes in mrc pedestrian regions";

    auto writeTxn = pool_.masterWriteableTransaction();
    auto readTxn = pool_.slaveTransaction();

    pubsub::CommitConsumer consumer(*writeTxn, PUBSUB_CONSUMER_ID, PUBSUB_BRANCH_ID);
    consumer.setBatchSizeLimit(PUBSUB_CONSUMER_BATCH_SIZE);
    auto commitIds = consumer.consumeBatch(*readTxn);
    INFO() << "Loaded " << commitIds.size() << " new commits to consume";

    auto commitIdToObjectIds = loadCommitIdToObjectIdsMap(*readTxn, commitIds);

    if (commitIdToObjectIds.empty()) {
        INFO() << "No commits with mrc pedestrian region changes to consume";
        writeTxn->commit();
        return commitIds.size();
    }

    auto commitIdToCommit = loadCommitIdToCommitMap(*readTxn, getKeys(commitIdToObjectIds));

    for (const auto& [commitId, objectIds] : commitIdToObjectIds) {
        auto commitIt = commitIdToCommit.find(commitId);
        ASSERT(commitIt != commitIdToCommit.end());

        consumeCommit(*readTxn, commitIt->second, objectIds);
    }
    writeTxn->commit();
    return commitIds.size();
} catch (const pubsub::AlreadyLockedException&) {
    INFO() << "Another instance has already locked pubsub queue";
    return 0;
}

std::map<rev::DBID, IdSet>
TasksGenerator::loadCommitIdToObjectIdsMap(
    pqxx::transaction_base& txn,
    const std::vector<rev::DBID>& commitIds)
{
    if (commitIds.empty()) {
        return {};
    }

    rev::RevisionsGateway revGateway(txn);
    auto commitIdsMinMax = std::minmax_element(
        commitIds.begin(),
        commitIds.end());
    auto histSnapshot = revGateway.historicalSnapshot(
        *commitIdsMinMax.first,
        *commitIdsMinMax.second);
    auto revIds = histSnapshot.revisionIdsByFilter(
        rf::ObjRevAttr::isNotRelation() &&
        rf::Geom::defined() &&
        rf::Attr(ATTR_CAT_MRC_PEDESTRIAN_REGION).defined() &&
        rf::CommitAttr::id().in(commitIds));

    std::map<rev::DBID, IdSet> commitIdToObjectIds;
    for (const auto& revId : revIds) {
        commitIdToObjectIds[revId.commitId()].insert(revId.objectId());
    }
    return commitIdToObjectIds;
}

std::map<rev::DBID, rev::Commit>
TasksGenerator::loadCommitIdToCommitMap(
    pqxx::transaction_base& txn,
    const std::vector<rev::DBID>& commitIds)
{
    std::map<rev::DBID, rev::Commit> commitIdToCommit;
    auto commits = rev::Commit::load(txn, rf::CommitAttr::id().in(commitIds));
    for (auto&& commit : commits) {
        auto commitId = commit.id();
        commitIdToCommit.emplace(commitId, std::move(commit));
    }
    return commitIdToCommit;
}


void TasksGenerator::consumeCommit(
    pqxx::transaction_base& txn,
    const rev::Commit& commit,
    const IdSet& objectIds)
{
    const auto& commitDiff = rev::commitDiff(txn, commit.id());

    auto objectIdToRegion = loadRegions(txn, commit.id(), objectIds);

    for (const auto& objectId : objectIds) {
        ASSERT(commitDiff.find(objectId) != commitDiff.end());

        auto& region = objectIdToRegion.find(objectId)->second;

        const auto& revDiff = commitDiff.at(objectId);
        const auto& revId = revDiff.newId();
        const auto& revDiffData = revDiff.data();

        if (isRegionCreated(revDiff)) {
            onRegionCreated(
                region,
                revId.objectId(),
                commit,
                *revDiffData.attributes);
            continue;
        }
        if (isRegionDeleted(revDiff)) {
            onRegionDeleted(region, revId.objectId(), commit);
            continue;
        }
        if (revDiffData.attributes) {
            onRegionAttrsChanged(
                region,
                revId.objectId(),
                commit,
                *revDiffData.attributes);
        }
    }
}

void TasksGenerator::onRegionCreated(
    const MrcPedestrianRegion& region,
    rev::DBID objectId,
    const rev::Commit& commit,
    const rev::AttributesDiff& attrsDiff)
{
    INFO() <<
        "Commit id=" << commit.id() <<
        ", mrc pedestrian region id=" << objectId << ": created";

    bool doCreate = false;

    for (const auto& [attr, value] : attrsDiff) {
        INFO() <<
            "Commit id=" << commit.id() <<
            ", mrc pedestrian region id=" << objectId <<
            ": attribute '" << attr << "' set to '" << value.after << "'";

        if (attr == ATTR_STATUS && shouldShowTaskInStatus(value.after)) {
            doCreate = true;
        }
    }

    if (doCreate) {
        createTaskFor(region);
    }
}

void TasksGenerator::onRegionDeleted(
    const MrcPedestrianRegion& region,
    rev::DBID objectId,
    const rev::Commit& commit)
{
    INFO() <<
        "Commit id=" << commit.id() <<
        ", mrc pedestrian region id=" << objectId << ": deleted";

    deleteTaskFor(region.objectId(), region.pedestrianUid());
}

void TasksGenerator::onRegionAttrsChanged(
    MrcPedestrianRegion& region,
    rev::DBID objectId,
    const rev::Commit& commit,
    const rev::AttributesDiff& attrsDiff)
{
    bool isUpdated = false;

    for (const auto& [attr, value] : attrsDiff) {
        INFO() <<
            "Commit id=" << commit.id() <<
            ", mrc pedestrian region id=" << objectId <<
            ": attribute '" << attr << "' changed from '" <<
            value.before << "' to '" << value.after << "'";

        if (attr == ATTR_STATUS) {
            const bool wasVisible = shouldShowTaskInStatus(value.before);
            const bool shouldBeVisible = shouldShowTaskInStatus(value.after);

            if (wasVisible && !shouldBeVisible) {
                deleteTaskFor(region.objectId(), region.pedestrianUid());
                return;
            } else if (!wasVisible && shouldBeVisible) {
                isUpdated = true;
            }
        }

        if (attr == ATTR_NAME || attr == ATTR_IS_INDOOR || attr == ATTR_IS_PANORAMIC) {
            isUpdated = true;
        }
        if (attr == ATTR_PEDESTRIAN_LOGIN_UID) {
            deleteTaskFor(region.objectId(), value.before);
            isUpdated = true;
        }
    }

    if (isUpdated && shouldShowTaskInStatus(region.status())) {
        createTaskFor(region);
    }
}

void TasksGenerator::createTaskFor(const MrcPedestrianRegion& region)
{
    const auto taskId = makeUgcTaskId(region.objectId());

    if (region.pedestrianUid().empty()) {
        INFO() << "Don't create task for empty uid";
        return;
    }
    std::vector<streetview::MrcTarget> panoramaTargets;
    if (region.isPanoramic()) {
        const auto geodeticPolygon = geolib3::convertMercatorToGeodetic(region.mercatorPolygon());
        const auto mrcRegionArea = geodeticPolygon.fastGeoArea();
        INFO() << "Mrc region area is: " << mrcRegionArea;
        if (mrcRegionArea <= MAX_AREA_FOR_TARGET_GENERATION) {
            panoramaTargets = streetview::generateTargets(pool_, region.mercatorPolygon());
        } else {
            WARN() << "Mrc region area is too large, skipping panorama targets generation";
        }
    }

    ugcBackofficeClient_.createAssignment(
        taskId,
        region.name(),
        region.pedestrianUid(),
        region.mercatorPolygon(),
        region.isIndoor(),
        region.isPanoramic(),
        panoramaTargets);
}

void TasksGenerator::deleteTaskFor(
    rev::DBID objectId,
    const std::string& uid)
{
    if (uid.empty()) {
        INFO() << "Don't delete task for empty uid";
        return;
    }

    const auto taskId = makeUgcTaskId(objectId);
    ugcBackofficeClient_.deleteAssignment(taskId, uid);
}


std::unordered_map<rev::DBID, MrcPedestrianRegion>
TasksGenerator::loadRegions(
    pqxx::transaction_base& txn,
    rev::DBID commitId,
    const IdSet& objectIds)
{
    std::unordered_map<rev::DBID, MrcPedestrianRegion> objectIdToRegion;

    rev::RevisionsGateway revGateway(txn);
    auto snapshot = revGateway.snapshot(commitId);

    INFO() <<
        "Load revision info for region ids " <<
        common::join(objectIds, ", ");

    auto objectRevs = snapshot.objectRevisions(objectIds);
    for (const auto& objectId : objectIds) {
        REQUIRE(objectRevs.find(objectId) != objectRevs.end(),
            "Failed to revision info for region " << objectId);
    }

    for (const auto& [objectId, rev] : objectRevs) {
        ASSERT(rev.data().attributes);
        ASSERT(rev.data().geometry);
        objectIdToRegion.emplace(objectId, MrcPedestrianRegion(rev));
    }
    return objectIdToRegion;
}

} // namespace maps::wiki::tasks::mrc_pedestrian
