#include "outsource_region_handler.h"

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/libs/gdpr/include/user.h>
#include <yandex/maps/wiki/common/date_time.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/outsource/task_result.h>
#include <yandex/maps/wiki/pubsub/commit_consumer.h>
#include <yandex/maps/wiki/revision/filters.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/log8/include/log8.h>

#include <algorithm>
#include <iomanip>
#include <string>

namespace rev = maps::wiki::revision;
namespace rf = maps::wiki::revision::filters;

namespace maps {
namespace wiki {
namespace tasks {
namespace outsource_regions {

namespace {

const std::string ST_QUEUE_KEY = "OUTKART";

const std::string PUBSUB_CONSUMER_ID = "OutsourceRegions";
const rev::DBID PUBSUB_BRANCH_ID = rev::TRUNK_BRANCH_ID;

const std::string ATTR_CAT_OUTSOURCE_REGION = "cat:outsource_region";

const std::string ATTR_NAME = "outsource_region:name";
const std::string ATTR_COMPANY_ID = "outsource_region:company_id";
const std::string ATTR_QUALITY = "outsource_region:quality";
const std::string ATTR_STATUS = "outsource_region:status";
const std::string ATTR_PARENT_ST_KEY = "outsource_region:parent_st_key";

const std::string STATUS_CAN_START = "can_start";
const std::string STATUS_AWAITING_BILLING = "awaiting_billing";

const std::string UNDEFINED_USER = "UNDEFINED USER";

const std::string STR_STAT_HEADERS =
    "|| id ЗА" \
    " | Имя ЗА" \
    " | Компания" \
    " | Аутсорсер" \
    " | Цена за единицу работ" \
    " | Коэффициент сложности" \
    " | Объём работ" \
    " | Плата" \
    " | Начало работ" \
    " | Окончание работ" \
    " ||\n";

const std::string STR_TAX_INCLUDED = "Все налоги включены";
const std::string STR_TAX_NOT_INCLUDED = "Все суммы посчитаны без учёта налога";

const std::string ACTIONS_LOG_FILE = "/var/lib/yandex/maps/wiki/stat/outsource-regions-actions-log.tskv";
const std::string PAYMENTS_LOG_FILE = "/var/lib/yandex/maps/wiki/stat/outsource-regions-payments-log.tskv";

const std::string PAYMENTS_LOG_FORMAT = "nmaps-outsource-regions-payments-log";

bool
isRegionCreated(const rev::ObjectRevisionDiff& revDiff)
{
    return !revDiff.oldId().valid();
}

bool
isRegionDeleted(const rev::ObjectRevisionDiff& revDiff)
{
    return revDiff.data().deleted && revDiff.data().deleted->after;
}

std::string
getStatComment(const outsource::TaskResult& taskResult)
{
    std::stringstream text;
    text << "#|\n"
         << STR_STAT_HEADERS
         << "|| " << taskResult.regionId
         << " | " << taskResult.regionName
         << " | " << taskResult.companyName
         << " | " << taskResult.outsourcerLogin
         << " | " << taskResult.rate
         << " | " << taskResult.complexityRate
         << " | " << taskResult.workAmount
         << std::fixed << std::setprecision(1)
         << " | " << taskResult.moneyAmount
         << " | " << common::canonicalDateTimeString(taskResult.workStart, common::WithTimeZone::Yes)
         << " | " << common::canonicalDateTimeString(taskResult.workEnd, common::WithTimeZone::Yes)
         << " ||\n|#";
     if (!taskResult.companyName.empty()) {
         text << (taskResult.taxIncluded ? STR_TAX_INCLUDED : STR_TAX_NOT_INCLUDED);
     }
    return text.str();
}

common::TskvMessage
getPaymentsLogMessage(const outsource::TaskResult& taskResult, const std::string& createdAt)
{
    common::TskvMessage message(PAYMENTS_LOG_FORMAT);
    message.setParam("region_id", taskResult.regionId);
    message.setParam("region_name", taskResult.regionName);
    message.setParam("task_type", taskResult.taskType);
    message.setParam("company_name", taskResult.companyName);
    message.setParam("outsourcer_login", taskResult.outsourcerLogin);
    message.setParam("quality", taskResult.quality);
    message.setParam("rate", taskResult.rate);
    message.setParam("complexity_rate", taskResult.complexityRate);
    message.setParam("work_amount", taskResult.workAmount);

    std::stringstream strMoneyAmount;
    strMoneyAmount << std::fixed << std::setprecision(1) << taskResult.moneyAmount;
    message.setParam("money_amount", strMoneyAmount.str());

    message.setParam("work_start",
        common::canonicalDateTimeString(taskResult.workStart, common::WithTimeZone::Yes));
    message.setParam("work_end",
        common::canonicalDateTimeString(taskResult.workEnd, common::WithTimeZone::Yes));

    message.setParam(
        "unixtime",
        chrono::convertToUnixTime(chrono::parseSqlDateTime(createdAt))
    );

    return message;
}

} // anonymous namespace

OutsourceRegionHandler::OutsourceRegionHandler(
        pgpool3::Pool& pool,
        const st::Configuration& stConfig,
        diffalert::EditorConfig editorConfig,
        outsource::Config outsourceConfig,
        std::string nproHost,
        Translator translator,
        YtEnabled ytEnabled)
    : pool_(pool)
    , stGateway_(stConfig)
    , editorConfig_(std::move(editorConfig))
    , outsourceConfig_(std::move(outsourceConfig))
    , nproHost_(std::move(nproHost))
    , translator_(std::move(translator))
    , ytEnabled_(ytEnabled)
    , actionsLogger_(ACTIONS_LOG_FILE)
    , paymentsLogger_(PAYMENTS_LOG_FILE)
{
}

size_t
OutsourceRegionHandler::dispatchNewCommits()
try {
    INFO() << "Starting routine for dispatching changes in outsource regions";

    auto writeTxn = pool_.masterWriteableTransaction();
    auto readTxn = pool_.slaveTransaction();

    pubsub::CommitConsumer consumer(*writeTxn, PUBSUB_CONSUMER_ID, PUBSUB_BRANCH_ID);
    consumer.setBatchSizeLimit(PUBSUB_CONSUMER_BATCH_SIZE);
    auto commitIds = consumer.consumeBatch(*readTxn);

    if (commitIds.empty()) {
        INFO() << "No new commits to consume";
        writeTxn->commit();
        return commitIds.size();
    }
    INFO() << "Consumed batch of " << commitIds.size() << " commits";

    rev::RevisionsGateway revGateway(*readTxn);
    auto commitIdsMinMax = std::minmax_element(
        commitIds.begin(),
        commitIds.end());
    auto histSnapshot = revGateway.historicalSnapshot(
        *commitIdsMinMax.first,
        *commitIdsMinMax.second);
    auto revIds = histSnapshot.revisionIdsByFilter(
        rf::ObjRevAttr::isNotRelation() &&
        rf::Geom::defined() &&
        rf::Attr(ATTR_CAT_OUTSOURCE_REGION).defined() &&
        rf::CommitAttr::id().in(commitIds));

    std::set<rev::DBID> commitIdsToLoad;
    std::map<rev::DBID, std::set<rev::DBID>> commitIdToObjectIds;
    for (const auto& revId : revIds) {
        commitIdsToLoad.insert(revId.commitId());
        commitIdToObjectIds[revId.commitId()].insert(revId.objectId());
    }

    if (commitIdsToLoad.empty()) {
        INFO() << "No commits with outsource region changes";
        writeTxn->commit();
        return commitIds.size();
    }

    actionsLogger_.onLogrotate();
    paymentsLogger_.onLogrotate();

    std::set<rev::UserID> userIds;
    std::map<rev::DBID, rev::UserID> commitIdToUserId;
    std::map<rev::DBID, rev::Commit> commitIdToCommit;
    auto commits = rev::Commit::load(*readTxn, rf::CommitAttr::id().in(commitIdsToLoad));
    for (auto&& commit : commits) {
        auto commitId = commit.id();
        auto uid = gdpr::User(commit.createdBy()).realUid();
        userIds.insert(uid);
        commitIdToUserId.emplace(commitId, uid);
        commitIdToCommit.emplace(commitId, std::move(commit));
    }
    loadNewUsers(*readTxn, userIds);

    for (const auto& kvp : commitIdToObjectIds) {
        auto commitId = kvp.first;
        const auto& commitDiff = rev::commitDiff(*readTxn, commitId);

        auto userIdIt = commitIdToUserId.find(commitId);
        ASSERT(userIdIt != commitIdToUserId.end());
        auto userId = userIdIt->second;

        auto commitIt = commitIdToCommit.find(commitId);
        ASSERT(commitIt != commitIdToCommit.end());
        const auto& commit = commitIt->second;

        const auto& objectIds = kvp.second;
        auto objectIdToRegion = loadRegions(*readTxn, commitId, objectIds);

        for (const auto& objectId : objectIds) {
            ASSERT(commitDiff.find(objectId) != commitDiff.end());

            auto& region = objectIdToRegion.find(objectId)->second;

            const auto& revDiff = commitDiff.at(objectId);
            const auto& revId = revDiff.newId();
            const auto& revDiffData = revDiff.data();

            if (isRegionCreated(revDiff)) {
                onRegionCreated(region, revId.objectId(), commit, *revDiffData.attributes, userId);
                continue;
            }
            if (isRegionDeleted(revDiff)) {
                onRegionDeleted(region, revId.objectId(), commit, userId);
                continue;
            }
            if (revDiffData.geometry) {
                onRegionGeometryChanged(region, revId.objectId(), commit, *revDiffData.geometry);
            }
            if (revDiffData.attributes) {
                onRegionAttrsChanged(region, revId.objectId(), commit, *revDiffData.attributes, userId);
            }
        }
    }
    writeTxn->commit();
    return commitIds.size();
} catch (const pubsub::AlreadyLockedException&) {
    INFO() << "Another instance has already locked pubsub queue";
    return 0;
}

void
OutsourceRegionHandler::onRegionCreated(
    OutsourceRegion& region,
    rev::DBID objectId,
    const rev::Commit& commit,
    const rev::AttributesDiff& attrsDiff,
    rev::UserID userId)
{
    INFO() <<
        "Commit id=" << commit.id() <<
        ", outsource region id=" << objectId << ": created";

    bool doCreate = false;

    for (const auto& kvp : attrsDiff) {
        const auto& attr = kvp.first;
        const auto& value = kvp.second.after;

        INFO() <<
            "Commit id=" << commit.id() <<
            ", outsource region id=" << objectId <<
            ": attribute '" << attr << "' set to '" << value << "'";

        if (attr == ATTR_STATUS && value == STATUS_CAN_START) {
            doCreate = true;
        }
    }

    if (doCreate) {
        createIssue(region, commit.id(), userId);
    }

    if (ytEnabled_ == YtEnabled::Yes) {
        actionsLogger_.log(region.logMessage(commit, userId));
    }
}

void
OutsourceRegionHandler::onRegionDeleted(
    OutsourceRegion& region,
    rev::DBID objectId,
    const rev::Commit& commit,
    rev::UserID userId)
{
    INFO() <<
        "Commit id=" << commit.id() <<
        ", outsource region id=" << objectId << ": deleted";

    if (region.stKey().empty()) {
        WARN() << "Issue is not found for region " << objectId;
        return;
    }

    auto text = region.commentAboutDeletion(commit.id(), getLogin(userId));
    auto comment = stGateway_.createComment(region.stKey(), text);
    INFO() << "Created comment " << comment.self();

    if (ytEnabled_ == YtEnabled::Yes) {
        actionsLogger_.log(region.logMessage(commit, userId));
    }
}

void
OutsourceRegionHandler::onRegionGeometryChanged(
    OutsourceRegion& region,
    rev::DBID objectId,
    const rev::Commit& commit,
    const rev::GeometryDiff& geomDiff)
{
    INFO() <<
        "Commit id=" << commit.id() <<
        ", outsource region id=" << objectId << ": geometry changed";

    region.updateGeometry(geomDiff);
}

void
OutsourceRegionHandler::onRegionAttrsChanged(
    OutsourceRegion& region,
    rev::DBID objectId,
    const rev::Commit& commit,
    const rev::AttributesDiff& attrsDiff,
    rev::UserID userId)
{
    bool doCreate = false;
    bool updateName = false;
    bool updateCompany = false;
    bool updateParent = false;
    bool addStat = false;

    for (const auto& kvp : attrsDiff) {
        const auto& attr = kvp.first;
        const auto& valueBefore = kvp.second.before;
        const auto& valueAfter = kvp.second.after;

        INFO() <<
            "Commit id=" << commit.id() <<
            ", outsource region id=" << objectId <<
            ": attribute '" << attr << "' changed from '" <<
            valueBefore << "' to '" << valueAfter << "'";

        if (attr == ATTR_STATUS && valueAfter == STATUS_CAN_START) {
            doCreate = true;
        }
        if (attr == ATTR_STATUS && valueAfter == STATUS_AWAITING_BILLING) {
            addStat = true;
        }
        if (attr == ATTR_NAME) {
            updateName = true;
        }
        if (attr == ATTR_COMPANY_ID) {
            updateCompany = true;
        }
        if (attr == ATTR_PARENT_ST_KEY) {
            updateParent = true;
        }
    }

    region.updateAttributes(attrsDiff);

    if (doCreate) {
        createIssue(region, commit.id(), userId);
    } else {
        if (!region.stKey().empty()) {
            auto text = region.comment(commit.id(), attrsDiff, getLogin(userId), translator_);
            auto comment = stGateway_.createComment(region.stKey(), text);
            INFO() << "Created comment " << comment.self();

            if (updateName || updateCompany || updateParent) {
                st::IssuePatch issuePatch;
                if (updateName) {
                    issuePatch.summary().set(region.name());
                }
                if (updateCompany) {
                    const auto& managerLogin =
                        outsourceConfig_.companyInfo(region.companyId()).managerLogin;
                    if (!managerLogin.empty()) {
                        INFO() << "Updated manager login to " << managerLogin;
                        issuePatch.assignee().set(managerLogin);
                    }
                }
                if (updateParent) {
                    if (region.parentStKey().empty()) {
                        issuePatch.parent().clear();
                    } else {
                        issuePatch.parent().set(region.parentStKey());
                    }
                }
                stGateway_.patchIssue(region.stKey(), issuePatch);
            }
            if (addStat) {
                auto taskResults = outsource::calcTaskResults(
                    {region.objectId()}, outsource::RegionCalcPolicy::CurrentTask,
                    rev::TRUNK_BRANCH_ID, pool_, editorConfig_, outsourceConfig_);
                REQUIRE(taskResults.size() == 1 && taskResults.begin()->second.size() == 1,
                    "Failed to calculate stat for region " << region.objectId());

                const auto& taskResult = taskResults.begin()->second.front();
                auto statText = getStatComment(taskResult);
                auto comment = stGateway_.createComment(region.stKey(), statText);

                if (ytEnabled_ == YtEnabled::Yes) {
                    paymentsLogger_.log(getPaymentsLogMessage(taskResult, commit.createdAt()));
                }

                INFO() <<
                    "Commit id=" << commit.id() <<
                    ", outsource region id=" << objectId <<
                    ": stat created";
            }
        } else {
            WARN() << "Issue is not found for region " << region.objectId();
        }
    }

    if (ytEnabled_ == YtEnabled::Yes) {
        actionsLogger_.log(region.logMessage(commit, userId));
    }
}

void OutsourceRegionHandler::createIssue(
    OutsourceRegion& region,
    revision::DBID commitId,
    revision::UserID userId)
{
    try {
        st::IssuePatch issuePatch;
        issuePatch.summary().set(region.name());
        issuePatch.description().set(region.description(commitId, getLogin(userId), translator_));
        if (!region.parentStKey().empty()) {
            issuePatch.parent().set(region.parentStKey());
        }

        const auto& managerLogin =
            outsourceConfig_.companyInfo(region.companyId()).managerLogin;
        if (!managerLogin.empty()) {
            issuePatch.assignee().set(managerLogin);
        }

        issuePatch.tags().set({
            "outsource_region_" + std::to_string(region.objectId()),
            "start_commit_id_" + std::to_string(commitId)
        });

        auto unique = std::to_string(region.objectId()) + ":" + std::to_string(commitId);
        auto issue = stGateway_.createIssue(ST_QUEUE_KEY, issuePatch, unique);
        region.setStKey(issue.key());
        INFO() << "Issue created " << issue.key();
    } catch (const st::ConflictError& e) {
        INFO() << "Conflict error. Skip. " << e;
    }
}

std::unordered_map<revision::DBID, OutsourceRegion>
OutsourceRegionHandler::loadRegions(
    pqxx::transaction_base& txn,
    revision::DBID commitId,
    const std::set<rev::DBID>& objectIds)
{
    std::unordered_map<revision::DBID, OutsourceRegion> objectIdToRegion;

    rev::RevisionsGateway revGateway(txn);
    auto snapshot = revGateway.snapshot(commitId);

    INFO() << "Load info for region ids " << common::join(objectIds, ", ");

    auto objectRevs = snapshot.objectRevisions(objectIds);
    for (const auto& objectId : objectIds) {
        REQUIRE(objectRevs.find(objectId) != objectRevs.end(),
            "Failed to found info for region " << objectId);
    }

    for (const auto& pair : objectRevs) {
        auto objectId = pair.first;
        const auto& rev = pair.second;

        ASSERT(rev.data().attributes);
        ASSERT(rev.data().geometry);

        OutsourceRegion region(rev);

        auto stKey = findIssueKeyForRegion(objectId);
        if (!stKey.empty()) {
            region.setStKey(stKey);
        }
        region.setNproHost(nproHost_);

        objectIdToRegion.emplace(objectId, std::move(region));
    }

    return objectIdToRegion;
}

std::string OutsourceRegionHandler::findIssueKeyForRegion(revision::DBID objectId)
{
    auto query = "Tags: outsource_region_" + std::to_string(objectId);

    boost::optional<st::Issue> lastIssue;
    for (const auto& issue : stGateway_.loadIssues(query)) {
        if (issue.createdAt()
            && (!lastIssue || issue.createdAt() > lastIssue->createdAt())) {
                lastIssue = issue;
        }
    }
    if (lastIssue) {
        return lastIssue->key();
    }
    return {};
}

void OutsourceRegionHandler::loadNewUsers(pqxx::transaction_base& txn, const std::set<rev::UserID>& userIds)
{
    std::vector<rev::UserID> userIdsToLoad;
    userIdsToLoad.reserve(userIds.size());

    for (auto userId : userIds) {
        if (!userIdToLogin_.count(userId)) {
            userIdsToLoad.push_back(userId);
        }
    }

    if (userIdsToLoad.empty()) {
        return;
    }

    INFO() << "Preload user logins for users uids " << common::join(userIdsToLoad, ", ");

    acl::ACLGateway aclGateway(txn);
    for (const auto& user : aclGateway.users(userIdsToLoad)) {
        userIdToLogin_.emplace(user.uid(), user.login());
    }
}

const std::string& OutsourceRegionHandler::getLogin(rev::UserID userId) const
{
    auto it = userIdToLogin_.find(userId);
    if (it != userIdToLogin_.end()) {
        return it->second;
    }
    WARN() << "User " << userId << " is not found in user cache";
    return UNDEFINED_USER;
}

} // namespace outsource_regions
} // namespace tasks
} // namespace wiki
} // namespace maps
