#include "outsource_region_handler.h"

#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/common/secrets.h>
#include <yandex/maps/wiki/diffalert/revision/editor_config.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <library/cpp/resource/resource.h>

#include <chrono>
#include <exception>
#include <memory>
#include <string>
#include <thread>

namespace osr = maps::wiki::tasks::outsource_regions;

namespace {

const std::string DB_ID = "long-read";
const std::string POOL_ID = "long-read";

const std::chrono::seconds WORKER_SLEEP_PERIOD{30};

} // anonymous namespace

int main(int argc, char** argv)
try {
    maps::cmdline::Parser parser;

    auto servicesConfig = parser.file("config").help(
        "path to services configuration file");
    auto syslogTag = parser.string("syslog-tag").help(
        "redirect log output to syslog with given tag");

    parser.parse(argc, argv);

    if (syslogTag.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslogTag));
    }

    std::unique_ptr<maps::wiki::common::ExtendedXmlDoc> configPtr;
    if (servicesConfig.defined()) {
        configPtr.reset(new maps::wiki::common::ExtendedXmlDoc(servicesConfig));
    } else {
        configPtr = maps::wiki::common::loadDefaultConfig();
    }

    maps::wiki::common::PoolHolder poolHolder(*configPtr, DB_ID, POOL_ID);

    maps::st::Configuration stConfig(
        configPtr->getAttr<std::string>("/config/common/st", "api-base-url"),
        maps::wiki::common::secrets::tokenByKey(maps::wiki::common::secrets::Key::RobotWikimapStToken));
    maps::wiki::diffalert::EditorConfig editorConfig(
        configPtr->get<std::string>("/config/services/editor/config"));
    maps::wiki::outsource::Config outsourceConfig(
        NResource::Find(maps::wiki::outsource::OUTSOURCE_CONFIG_RESOURCE_ID));

    auto nproHost = configPtr->get<std::string>("/config/common/npro/host", {});

    osr::Translator translator(configPtr->get<std::string>("/config/common/tanker/base-url"));

    auto ytEnabled = configPtr->getAttr<bool>("/config/services/pubsub/outsource-regions", "yt-enabled");

    osr::OutsourceRegionHandler handler(
        poolHolder.pool(),
        stConfig,
        std::move(editorConfig),
        std::move(outsourceConfig),
        std::move(nproHost),
        std::move(translator),
        ytEnabled ? osr::YtEnabled::Yes : osr::YtEnabled::No);

    while (true) {
        size_t commitsDispatched = 0;
        try {
            commitsDispatched = handler.dispatchNewCommits();
        } catch (const maps::Exception& ex) {
            ERROR() << "Consume failed: " << ex;
        } catch (const std::exception& ex) {
            ERROR() << "Consume failed: " << ex.what();
        }

        // Sleeping only if consumed batch is lesser in size than requested.
        // This is a probably-good approximation of no-new-commits-pending state
        if (commitsDispatched < osr::PUBSUB_CONSUMER_BATCH_SIZE) {
            INFO() << "Worker is going to sleep for "
                << WORKER_SLEEP_PERIOD.count() << " seconds";
            std::this_thread::sleep_for(WORKER_SLEEP_PERIOD);
        }
    }
    return EXIT_SUCCESS;
} catch (const maps::Exception& ex) {
    FATAL() << "Worker failed: " << ex;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    FATAL() << "Worker failed: " << ex.what();
    return EXIT_FAILURE;
}
