#include "handler.h"

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/libs/gdpr/include/user.h>
#include <yandex/maps/wiki/common/batch.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/pubsub/commit_consumer.h>
#include <yandex/maps/wiki/revision/filters.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/threadutils/executor.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/geolib/include/spatial_relation.h>

#include <maps/libs/pgpool/include/pool_configuration.h>
#include <maps/libs/log8/include/log8.h>

#include <algorithm>
#include <string>
#include <unordered_set>

namespace rev = maps::wiki::revision;
namespace rf = maps::wiki::revision::filters;

namespace maps {
namespace wiki {
namespace tasks {
namespace outsourcer_edits {

namespace {

struct Range
{
    size_t from;
    size_t to;
};

const std::string PUBSUB_CONSUMER_ID = "OutsourcerEdits";
const rev::DBID PUBSUB_BRANCH_ID = rev::TRUNK_BRANCH_ID;

const std::string ATTR_CAT_OUTSOURCE_REGION = "cat:outsource_region";

const std::string OUTSOURCE_ROLE = "outsource-role";

const std::string STAT_OUTPUT_FILE = "/var/lib/yandex/maps/wiki/stat/outsourcer-edits-log.tskv";

const std::string STAT_LOG_FORMAT = "nmaps-outsourcer-edits-log";

const size_t COMMIT_BATCH_COUNT = 10000;
const size_t USER_BATCH_COUNT = 100;

// Copy-pasted from social servant
const std::unordered_set<std::string> LOGGED_ACTIONS = {
    "object-created",
    "object-modified",
    "object-deleted",
    "commit-reverted"
};

bool isLoggedAction(const std::string& action)
{
    return LOGGED_ACTIONS.count(action);
}

class AclHelper
{
public:
    AclHelper(pqxx::transaction_base& work, const social::Events& events)
    {
        std::vector<acl::UID> uids;
        for (const auto& event : events) {
            uids.push_back(gdpr::User(event.createdBy()).realUid());
        }

        acl::ACLGateway aclGateway(work);

        common::applyBatchOp(
            uids,
            USER_BATCH_COUNT,
            [&](const std::vector<acl::UID>& batchUids) {
                auto batchUsers = aclGateway.users(batchUids);
                for (const auto& user : batchUsers) {
                    uidToId_.emplace(user.uid(), user.id());
                }
                auto batchIdToRole = aclGateway.firstApplicableRoles(batchUsers, {OUTSOURCE_ROLE}, {});
                idToRole_.insert(batchIdToRole.begin(), batchIdToRole.end());
            });
    }

    bool isOutsourcer(acl::UID uid) const
    {
        auto idIt = uidToId_.find(uid);
        if (idIt == uidToId_.end()) {
            return false;
        }
        auto roleIt = idToRole_.find(idIt->second);
        if (roleIt == idToRole_.end() || roleIt->second.empty()) {
            return false;
        }
        return true;
    }

private:
    std::unordered_map<acl::UID, acl::ID> uidToId_;
    std::map<acl::ID, std::string> idToRole_;
};

std::set<revision::DBID> computeRegionIds(pqxx::transaction_base& coreTxn, const social::CommitData& commitData)
{
    ASSERT(commitData.bbox());

    rev::RevisionsGateway revGateway(coreTxn);
    auto snapshot = revGateway.snapshot(commitData.commitId());

    const auto& bbox = *commitData.bbox();

    auto filter =
        rf::Attr(ATTR_CAT_OUTSOURCE_REGION).defined() &&
        rf::Geom::defined() &&
        rf::GeomFilterExpr(
            rf::GeomFilterExpr::Operation::IntersectsPolygons,
            bbox.minX(), bbox.minY(), bbox.maxX(), bbox.maxY()) &&
        rf::ObjRevAttr::isNotDeleted() &&
        rf::ObjRevAttr::isNotRelation();

    std::set<revision::DBID> regionIds;
    for (const auto& rev : snapshot.objectRevisionsByFilter(filter)) {
        ASSERT(rev.data().geometry);
        auto geom = geolib3::WKB::read<geolib3::Polygon2>(*rev.data().geometry);
        if (geolib3::spatialRelation(bbox, geom, geolib3::SpatialRelation::Intersects)) {
            regionIds.insert(rev.id().objectId());
        }
    }

    return regionIds;
}

std::vector<Range> getRanges(Range mainRange, size_t size)
{
    std::vector<Range> ranges;

    auto from = mainRange.from;
    while (from <= mainRange.to) {
        auto to = std::min(from + size, mainRange.to);
        ranges.emplace_back(Range{from, to});
        from = to + 1;
    }

    return ranges;
}

social::Events loadEvents(size_t fromCommit, size_t toCommit, pgpool3::Pool& socialPool)
{
    auto socialTxn = socialPool.slaveTransaction();
    social::Gateway socialGateway(*socialTxn);
    return socialGateway.loadEditEventsByCommitRange(fromCommit, toCommit);
}

} // anonymous namespace

Handler::Handler(pgpool3::Pool& corePool, pgpool3::Pool& socialPool)
    : corePool_(corePool)
    , socialPool_(socialPool)
    , statLogger_(STAT_OUTPUT_FILE)
{
}

size_t
Handler::dispatchNewCommits(size_t batchSize)
try {
    INFO() << "Starting routine for logging outsourcer edits";

    auto socialWriteTxn = socialPool_.masterWriteableTransaction();
    auto coreReadTxn = corePool_.slaveTransaction();

    pubsub::CommitConsumer consumer(*socialWriteTxn, PUBSUB_CONSUMER_ID, PUBSUB_BRANCH_ID);
    consumer.setBatchSizeLimit(batchSize);
    auto commitIds = consumer.consumeBatch(*coreReadTxn);

    if (commitIds.empty()) {
        INFO() << "No new commits to consume";
        socialWriteTxn->commit();
        return 0;
    }
    INFO() << "Consumed batch of " << commitIds.size() << " commits";

    statLogger_.onLogrotate();

    social::Gateway socialGateway(*socialWriteTxn);
    auto events = socialGateway.loadEditEventsByCommitIds(
        {std::begin(commitIds), std::end(commitIds)});

    processEvents(events, coreReadTxn, statLogger_);

    socialWriteTxn->commit();
    return commitIds.size();
} catch (const pubsub::AlreadyLockedException&) {
    INFO() << "Another instance has already locked pubsub queue";
    return 0;
}

void Handler::processRange(size_t fromCommit, size_t toCommit, const std::string& logPath)
{
    ASSERT(fromCommit <= toCommit);

    common::TskvLogger logger(logPath);

    auto ranges = getRanges(Range{fromCommit, toCommit}, COMMIT_BATCH_COUNT);

    Executor executor;
    for (const auto& range : ranges) {
        executor.addTask([range, this, &logger]() {
            INFO() << "Load events from commit range [" << range.from << "; " << range.to << "]";
            auto events = loadEvents(range.from, range.to, socialPool_);
            auto coreReadTxn = corePool_.slaveTransaction();
            processEvents(events, coreReadTxn, logger);
        });
    }

    auto corePoolState = corePool_.state();
    auto coreSlaveCount = corePoolState.configuration.slaves().size();
    auto socialPoolState = socialPool_.state();
    auto socialSlaveCount = socialPoolState.configuration.slaves().size();

    size_t upperConnectionsLimit = std::min(
        coreSlaveCount * corePoolState.constants.slaveMaxSize,
        socialSlaveCount * socialPoolState.constants.slaveMaxSize);

    size_t threadCount = std::min(
        std::min(
            ranges.size(),
            corePoolState.constants.slaveMaxSize),
        upperConnectionsLimit);

    INFO() << "Thread count " << threadCount;

    ThreadPool threadPool(threadCount);
    executor.executeAllInThreads(threadPool);
}

void Handler::processEvents(
    const social::Events& events,
    pgpool3::TransactionHandle& coreTxn,
    common::TskvLogger& logger)
{
    INFO() << "Initial event count " << events.size();

    AclHelper aclHelper(*coreTxn, events);

    size_t counter = 0;

    for (const auto& event : events) {
        ASSERT(event.commitData());
        const auto& commitData = event.commitData().value();

        if (!event.primaryObjectData() || !isLoggedAction(commitData.action())) {
            continue;
        }

        auto uid = gdpr::User(event.createdBy()).realUid();
        if (!aclHelper.isOutsourcer(uid)) {
            continue;
        }

        auto commitId = commitData.commitId();

        INFO() << "Outsourcer commit " << commitId << " created by " << uid;

        common::TskvMessage message(STAT_LOG_FORMAT);
        message.setParam(
            "unixtime",
            chrono::convertToUnixTime(chrono::parseSqlDateTime(event.createdAt()))
        );
        message.setParam("commit_id", commitId);
        message.setParam("puid", uid);
        message.setParam("region_id",
            commitData.bbox()
                ? common::join(computeRegionIds(*coreTxn, commitData), ",")
                : std::string{});

        logger.log(std::move(message));

        counter++;
    }

    INFO() << "Outsourcer event count " << counter;
}

} // namespace outsourcer_edits
} // namespace tasks
} // namespace wiki
} // namespace maps
