#include "handler.h"

#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <chrono>
#include <exception>
#include <memory>
#include <string>
#include <thread>

namespace osr = maps::wiki::tasks::outsourcer_edits;

namespace {

const std::string CORE_DB_ID = "long-read";
const std::string CORE_POOL_ID = "long-read";
const std::string SOCIAL_DB_ID = "social";
const std::string SOCIAL_POOL_ID = "grinder";

const size_t PUBSUB_CONSUMER_BATCH_SIZE = 50;

const std::chrono::seconds WORKER_SLEEP_PERIOD{30};

} // anonymous namespace

int main(int argc, char** argv)
try {
    maps::cmdline::Parser parser;

    auto syslogTag = parser.string("syslog-tag").help(
        "redirect log output to syslog with given tag");
    auto workerConfig = parser.file("config").help(
        "path to services configuration");
    auto fromCommit = parser.size_t("from").help(
        "the first commit of the range");
    auto toCommit = parser.size_t("to").help(
        "the last commit of the range");
    auto logPath = parser.file("log-path").help(
        "path to write log to").defaultValue("log.tskv");

    parser.parse(argc, argv);

    if (syslogTag.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslogTag));
    }

    std::unique_ptr<maps::wiki::common::ExtendedXmlDoc> configDocPtr;
    if (workerConfig.defined()) {
        configDocPtr.reset(new maps::wiki::common::ExtendedXmlDoc(workerConfig));
    } else {
        configDocPtr = maps::wiki::common::loadDefaultConfig();
    }

    maps::wiki::common::PoolHolder corePoolHolder(*configDocPtr, CORE_DB_ID, CORE_POOL_ID);
    maps::wiki::common::PoolHolder socialPoolHolder(*configDocPtr, SOCIAL_DB_ID, SOCIAL_POOL_ID);

    osr::Handler handler(corePoolHolder.pool(), socialPoolHolder.pool());

    if (fromCommit.defined() || toCommit.defined()) {
        REQUIRE(fromCommit.defined() && toCommit.defined(),
            "'from' and 'to' arguments must be specified simultaneously");

        handler.processRange(fromCommit, toCommit, logPath);

        return EXIT_SUCCESS;
    }

    while (true) {
        size_t commitsDispatched = 0;
        try {
            commitsDispatched = handler.dispatchNewCommits(PUBSUB_CONSUMER_BATCH_SIZE);
        } catch (const maps::Exception& ex) {
            ERROR() << "Consume failed: " << ex;
        } catch (const std::exception& ex) {
            ERROR() << "Consume failed: " << ex.what();
        }

        // Sleeping only if consumed batch is lesser in size than requested.
        // This is a probably-good approximation of no-new-commits-pending state
        if (commitsDispatched < PUBSUB_CONSUMER_BATCH_SIZE) {
            INFO() << "Worker is going to sleep for "
                << WORKER_SLEEP_PERIOD.count() << " seconds";
            std::this_thread::sleep_for(WORKER_SLEEP_PERIOD);
        }
    }

    return EXIT_SUCCESS;
} catch (const maps::Exception& ex) {
    FATAL() << "Worker failed: " << ex;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    FATAL() << "Worker failed: " << ex.what();
    return EXIT_FAILURE;
}
