#include "garden.h"

#include <fmt/format.h>

#include <boost/algorithm/string/split.hpp>
#include <maps/libs/json/include/value.h>

#include <optional>
#include <utility>


namespace maps::wiki::garden {

namespace {

using RegionToBranch = std::unordered_map<std::string, BranchId>;

const std::string& getModuleName(const std::string& event)
{
    static const std::unordered_map<std::string, std::string> eventToModuleName = {
        { EVENT_DEPLOYED_RENDERER, "renderer_deployment" },
        { EVENT_DEPLOYED_CAMS, "export_cams" },
        { EVENT_DEPLOYED_CARPARKS, "carparks_activation" },
        { EVENT_DEPLOYED_GEOCODER, "geocoder_deployment" },
        { EVENT_DEPLOYED_MTR_EXPORT, "masstransit_deployment" },
        { EVENT_DEPLOYED_GRAPH, "road_graph_deployment" },
        { EVENT_DEPLOYED_BICYCLE_GRAPH, "bicycle_graph_deployment" },
        // Pedestrian graph is deployed only with masstransit data.
        // So receive event of pedestrian graph
        // deployment from masstransit_deployment build
        { EVENT_DEPLOYED_PEDESTRIAN_GRAPH, "masstransit_deployment" },
    };

    auto it = eventToModuleName.find(event);
    REQUIRE(it != eventToModuleName.end(), "Failed to find module name for event '" << event << "'");
    return it->second;
}

RegionToBranch parseAncestorBuilds(const json::Value& ancestorBuilds, IHttpClient& httpClient)
{
    static auto cache = std::map<std::pair<std::string, int64_t>, json::Value>();
    RegionToBranch regionToBranch;

    for (const auto& ancestor: ancestorBuilds) {
        if (!ancestor.hasField("name") || ancestor["name"].as<std::string>("") != MODULE_YMAPSDF_SRC) {
            continue;
        }
        auto build_id = ancestor["build_id"].as<int64_t>();
        auto cacheKey = std::pair{MODULE_YMAPSDF_SRC, build_id};
        auto res = cache.find(cacheKey);
        if (res == cache.end()) {
            auto url = fmt::format(
                    GARDEN_BUILD_INFO,
                    MODULE_YMAPSDF_SRC,
                    build_id
            );

            res = cache.emplace(cacheKey, httpClient.get(url)).first;
        }

        const auto& properties = res->second["properties"];
        const auto& region = properties["region"].as<std::string>();
        //TODO: check for the trunk and approved branches
        BranchId branchId = std::stoll(properties["nmaps_branch_id"].as<std::string>());
        if (branchId < 100) {
            continue; // Hack to prevent accounting AND branches
        }

        auto ret = regionToBranch.emplace(region, branchId);
        REQUIRE(ret.second, "Duplicate region '" << region << "'");
    }
    return regionToBranch;
}

BranchId getMtrBranch(const json::Value& deployment)
{
    const auto& properties = deployment["properties"];
    REQUIRE(
            properties.hasField("shipping_date"),
            "Masstransit deployment has no shipping date!");
    const std::string shipmentId = properties["shipping_date"].as<std::string>();
    std::vector<std::string> parts;
    boost::split(parts, shipmentId, [](char c){ return c == '_'; });
    REQUIRE(parts.size() == 4 || parts.size() == 5, "bad shipment id: `" << shipmentId << "'");
    return std::stoul(parts[2]);
}

std::optional<chrono::TimePoint> getFinishTime(const json::Value& build)
{
    if (build.hasField("completed_at")) {
        return chrono::parseIsoDateTime(build["completed_at"].as<std::string>(""));
    }

    return std::nullopt;
}

} // namespace

HttpClient::HttpClient()
{
}

json::Value HttpClient::get(const std::string& url)
{
    using namespace std::chrono_literals;

    auto retryPolicy = common::RetryPolicy()
        .setTryNumber(10)
        .setInitialCooldown(1s)
        .setCooldownBackoff(2.0);

    auto [responseBody, status] = httpClient_.get(url, retryPolicy);
    return json::Value::fromString(responseBody);
}

RegionToBranch getRegionToBranch(
    const std::string& event,
    IHttpClient& httpClient,
    const json::Value& deployment)
{
    if (event == EVENT_DEPLOYED_MTR_EXPORT) {
        return {{REGION_CIS1, getMtrBranch(deployment)}};
    }
    return parseAncestorBuilds(deployment["tracked_ancestor_builds"], httpClient);
}

EventsHolder loadBranchDeployedTimes(
    IHttpClient& httpClient,
    const std::string& event,
    const chrono::TimePoint fromTime)
{
    EventsHolder eventsHolder;

    auto url = fmt::format(
        GARDEN_MODULES_STATISTICS,
        getModuleName(event),
        chrono::formatIsoDateTime(fromTime));

    auto deploymentsJson = httpClient.get(url);
    for (const auto& deployment: deploymentsJson) {
        auto deployStep = deployment["properties"]["deploy_step"].as<std::string>(DEPLOY_STEP_STABLE);
        if (deployStep != DEPLOY_STEP_STABLE) {
            continue;
        }
        auto completedAt = getFinishTime(deployment);
        if (completedAt) {
            auto regionToBranch = getRegionToBranch(event, httpClient, deployment);
            for (const auto& [region, branchId] : regionToBranch) {
                eventsHolder.addEvent(event, region, branchId, completedAt.value());
            }
        }
    }
    return eventsHolder;
}

} // namespace maps::wiki::garden
