#include "auto_experts_pred.h"

#include <optional>

namespace maps::wiki::skills_updater {

namespace {

const auto ADDR_GROUP = "addr_group";


// Aggregates skills by categories and resolutions.
//
// @return Amount of tasks in provided categories with provided resolution.
//
// @param skills Skills to be aggregated.
// @param categoryIds Categories that must be reflected in the result. If the
//   set is empty then all categories are counted.
// @param withResolution Resolution that must be reflected in the result. If
//   absent then all resolutions are counted.
size_t
amountOfTasks(
    const social::Skills& skills,
    const std::set<std::string>& categoryIds,
    std::optional<social::ResolveResolution> withResolution)
{
    size_t result = 0;

    for (const auto& [categoryId, amountByResolution]: skills) {
        if (categoryIds.empty() || categoryIds.count(categoryId)) {
            for (const auto [resolution, amount]: amountByResolution) {
                if (!withResolution || withResolution == resolution) {
                    result += amount;
                }
            }
        }
    }

    return result;
}


Action
addrPredicate(
    const configs::editor::CategoryGroups& categoryGroups,
    const social::Skills& skills)
{
    const auto MIN_TOTAL_CHANGES_IN_GROUP = 500;
    const auto ACCEPTED_PART = 0.95;

    const auto addrGroupCategories = categoryGroups.categoryIdsByGroup(ADDR_GROUP);

    const auto totalAddrGroup = amountOfTasks(skills, addrGroupCategories, std::nullopt);
    if (totalAddrGroup < MIN_TOTAL_CHANGES_IN_GROUP) {
        return Action::None;
    }

    const auto acceptedAddrGroup =
        amountOfTasks(skills, addrGroupCategories, social::ResolveResolution::Accept);
    if (acceptedAddrGroup < ACCEPTED_PART * totalAddrGroup) {
        return Action::None;
    }

    return Action::Add;
}

} // namespace


const std::map<std::string, Predicate> PREDICATES_BY_CATEGORY_GROUPS {
    {ADDR_GROUP, addrPredicate}
};

} // maps::wiki::skills_updater
