#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <yandex/maps/wiki/social/profile_gateway.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pg_advisory_lock_ids.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <maps/wikimap/mapspro/services/tasks_realtime/src/stat_users_dump_worker/lib/common.h>
#include <maps/wikimap/mapspro/services/tasks_realtime/src/stat_users_dump_worker/lib/user.h>

#include <algorithm>
#include <fstream>
#include <memory>
#include <string>
#include <unordered_map>
#include <vector>

namespace acl = maps::wiki::acl;
namespace common = maps::wiki::common;
namespace social = maps::wiki::social;

namespace dump = maps::wiki::stat_users_dump_worker;

namespace {

const std::string ACL_DB_ID = "core";
const std::string ACL_POOL_ID = "core";
const std::string SOCIAL_DB_ID = "social";
const std::string SOCIAL_POOL_ID = "social";

const std::string USERS_OUTPUT_FILE =
    "/var/lib/yandex/maps/wiki/stat/users-dump.tskv";
const std::string OUTSOURCERS_OUTPUT_FILE =
    "/var/lib/yandex/maps/wiki/stat/outsourcers-dump.tskv";


void dumpUsers(
    acl::ACLGateway& aclGateway,
    social::ProfileGateway& socialProfileGateway)
{
    std::ofstream outStream(USERS_OUTPUT_FILE, std::ios_base::app);
    REQUIRE(outStream,
            "Error opening output file \"" << USERS_OUTPUT_FILE << '"');

    auto now = dump::generateUnixTimeStr();
    const auto users = dump::getUsersDumpData(aclGateway, socialProfileGateway);

    INFO() << "Writing " << users.size() << " user infos "
           << " with timestamp " << now
           << " to \"" << USERS_OUTPUT_FILE << '"';

    for (const auto& user : users) {
        outStream << "tskv"                               //
                  << "\ttskv_format=nmaps-users-dump-log" //
                  << "\tunixtime=" << now;                //

        user.writeUserData(outStream);

        outStream << '\n';
    }

    outStream.close();
    REQUIRE(outStream,
            "Error closing output file \"" << USERS_OUTPUT_FILE << '"');
}

void dumpOutsourcers(acl::ACLGateway& aclGateway)
{
    auto now = dump::generateUnixTimeStr();

    auto idToGroup = dump::getOutsourcerIdToGroup(aclGateway);

    std::vector<acl::ID> userIds;
    userIds.reserve(idToGroup.size());
    for (const auto& pair : idToGroup) {
        userIds.push_back(pair.first);
    }

    INFO() << "Writing " << userIds.size() << " outsourcer infos "
           << " with timestamp " << now
           << " to \"" << OUTSOURCERS_OUTPUT_FILE << '"';

    std::ofstream outStream(OUTSOURCERS_OUTPUT_FILE, std::ios_base::app);
    REQUIRE(outStream,
            "Error opening output file \"" << OUTSOURCERS_OUTPUT_FILE << '"');

    auto users = aclGateway.usersByIds(userIds);
    for (const auto& user : users) {
        outStream << "tskv"
            << "\tunixtime=" << now
            << "\ttskv_format=nmaps-outsourcers-dump-log"
            << "\tpuid=" << user.uid()
            << "\tcompany=" << idToGroup[user.id()]
            << '\n';
    }

    outStream.close();
    REQUIRE(outStream,
            "Error closing output file \"" << OUTSOURCERS_OUTPUT_FILE << '"');
}

} // namespace

int main(int argc, char** argv)
{
    try {
        maps::cmdline::Parser parser;

        auto syslogTag = parser.string("syslog-tag").help(
            "redirect log output to syslog with given tag");
        auto workerConfig = parser.file("config").help(
            "path to worker configuration");

        parser.parse(argc, argv);

        if (syslogTag.defined()) {
            maps::log8::setBackend(maps::log8::toSyslog(syslogTag));
        }

        std::unique_ptr<common::ExtendedXmlDoc> configDocPtr;
        if (workerConfig.defined()) {
            configDocPtr =
                std::make_unique<common::ExtendedXmlDoc>(workerConfig);
        } else {
            configDocPtr = common::loadDefaultConfig();
        }

        common::PoolHolder aclPoolHolder(
            *configDocPtr, ACL_DB_ID, ACL_POOL_ID);
        maps::pgp3utils::PgAdvisoryXactMutex aclLocker(
            aclPoolHolder.pool(),
            static_cast<int64_t>(common::AdvisoryLockIds::STAT_USERS_DUMP));
        if (!aclLocker.try_lock()) {
            INFO() << ACL_DB_ID
                   << " database is already locked. Task interrupted.";
            return EXIT_SUCCESS;
        }
        acl::ACLGateway aclGateway(aclLocker.writableTxn());

        INFO() << "Start dumping outsourcers statistics";
        dumpOutsourcers(aclGateway);

        common::PoolHolder socialPoolHolder(
            *configDocPtr, SOCIAL_DB_ID, SOCIAL_POOL_ID);
        auto txnHandler =
            socialPoolHolder.pool().masterReadOnlyTransaction();
        social::ProfileGateway socialProfileGateway(txnHandler.get());

        INFO() << "Start dumping users statistics";
        dumpUsers(aclGateway, socialProfileGateway);

        INFO() << "Done";
        return EXIT_SUCCESS;
    } catch (const maps::Exception& ex) {
        FATAL() << "Worker failed: " << ex;
        return EXIT_FAILURE;
    } catch (const std::exception& ex) {
        FATAL() << "Worker failed: " << ex.what();
        return EXIT_FAILURE;
    }
}
