#include "common.h"

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/log8/include/log8.h>

#include <yandex/maps/wiki/common/moderation.h>

#include <boost/algorithm/string/predicate.hpp>

#include <chrono>

namespace common = maps::wiki::common;

namespace {

// Ordered by descending significance
const std::vector<std::string> MODERATION_STATUSES = {
    common::MODERATION_STATUS_ROBOT,
    common::MODERATION_STATUS_CARTOGRAPHER,
    common::MODERATION_STATUS_YANDEX_MODERATOR,
    common::MODERATION_STATUS_MODERATOR,
    common::OUTSOURCE_ROLE,
    common::MODERATION_STATUS_EXPERT,
    common::MODERATION_STATUS_COMMON
};

const size_t ACL_BATCH_SIZE = 10000;

const std::string OUTSOURCE_GROUP_PREFIX = "outsource-group";

} // namespace

namespace maps::wiki::stat_users_dump_worker {

std::unordered_map<acl::ID, std::string> getModerationStatusesByAgentId(
    acl::ACLGateway& gateway, const std::vector<acl::User>& users)
{
    std::unordered_map<acl::ID, std::string> moderationStatusesByAgentId;

    auto batchBegin = std::begin(users);
    while (batchBegin != std::end(users)) {
        auto batchEnd = std::next(
            batchBegin,
            std::min<size_t>(
                ACL_BATCH_SIZE, std::distance(batchBegin, std::end(users))));

        std::vector<acl::User> batch(batchBegin, batchEnd);
        for (auto&& idStatusPair: gateway.firstApplicableRoles(
                 batch,
                 MODERATION_STATUSES,
                 common::MODERATION_STATUS_COMMON)) {
            moderationStatusesByAgentId.insert(std::move(idStatusPair));
        }

        batchBegin = batchEnd;
    }

    return moderationStatusesByAgentId;
}

std::unordered_map<acl::ID, std::string> getOutsourcerIdToGroup(
    acl::ACLGateway& aclGateway)
{
    std::unordered_map<acl::ID, std::string> result;

    for (const auto& group: aclGateway.groups()) {
        if (!boost::algorithm::starts_with(
                group.name(), OUTSOURCE_GROUP_PREFIX)) {
            continue;
        }

        INFO() << "Dump outsourcers group " << group.name();

        auto userIds = aclGateway.userIds(group.id(), {}, {});
        for (const auto id: userIds) {
            auto res = result.emplace(id, group.name());
            if (!res.second) {
                WARN() << "User with id " << id
                       << " has many outsourcer groups: " << res.first->second
                       << ", " << group.name();
            }
        }
    }

    const auto& outsourceRole = aclGateway.role(common::OUTSOURCE_ROLE);
    auto userIds = aclGateway.userIds({}, outsourceRole.id(), {});
    for (const auto id: userIds) {
        result.emplace(id, std::string {});
    }

    return result;
}

std::string generateUnixTimeStr()
{
    return std::to_string(maps::chrono::sinceEpoch<std::chrono::seconds>());
}

} // namespace maps::wiki::stat_users_dump_worker
