#include "user.h"
#include "profiles.h"
#include "common.h"

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/log8/include/log8.h>

#include <maps/wikimap/mapspro/libs/acl/include/user.h>
#include <yandex/maps/wiki/common/date_time.h>

namespace acl = maps::wiki::acl;
namespace social = maps::wiki::social;

namespace {

const std::string TRUE_STRING = "TRUE";
const std::string FALSE_STRING = "FALSE";
const std::string DATE_TIME_FORMAT = "%Y-%m-%d %H:%M:%S";

} // namespace


namespace maps::wiki::stat_users_dump_worker {

User::User(
    acl::User aclUser,
    std::string moderationStatus,
    const std::optional<social::Profile>& profile)
    : aclUser_(std::move(aclUser)),
      profile_(profile),
      moderationStatus_(std::move(moderationStatus))
{}

std::string User::registrationDateString() const
{
    auto registrationDate = common::canonicalDateTimeString(
        aclUser_.created(), common::WithTimeZone::No);
    // Date-time delimiter should be space
    std::replace(
        std::begin(registrationDate), std::end(registrationDate), 'T', ' ');

    return registrationDate;
}

acl::User::Status User::userStatus() const { return aclUser_.status(); }

std::string User::locale() const {
    if (profile_.has_value()) {
        return profile_->locale();
    }
    return {};
}

std::string User::newsSubscriptionString() const
{
    if (profile_.has_value() &&
        profile_->hasNewsSubscription()) {
        return TRUE_STRING;
    }
    return FALSE_STRING;
}

std::string User::newsSubscriptionModifiedAtString() const
{
    if (!profile_.has_value() ||
        !profile_->newsSubscriptionModifiedAt().has_value())
    {
        return {};
    }

    const auto t = profile_->newsSubscriptionModifiedAt().value();
    return maps::chrono::formatIntegralDateTime(t, DATE_TIME_FORMAT);
}

std::string User::email() const
{
    if (profile_.has_value()) {
        return profile_->email();
    }
    return {};
}

std::string_view User::deleteReason() const
{
    const auto reason = aclUser_.deleteReason();
    return aclUser_.status() == acl::User::Status::Deleted && reason
        ? acl::toString(*reason)
        : std::string_view();
}

void User::writeUserData(std::ostream& outStream) const
{
    outStream
        << "\tpuid=" << aclUser_.uid()
        << "\tum_login=" << aclUser_.login()
        << "\tregistration_date=" << registrationDateString()
        << "\tstatus=" << userStatus()
        << "\tmoderation_status=" << moderationStatus_
        << "\tcontact_email=" << email()
        << "\tlocale=" << locale()
        << "\thas_news_subscription=" << newsSubscriptionString()
        << "\tnews_subscription_modified_at=" << newsSubscriptionModifiedAtString()
        << "\tdelete_reason=" << deleteReason();
}

Users getUsersDumpData(
    acl::ACLGateway& aclGateway, social::ProfileGateway& socialProfileGateway)
{
    DEBUG() << "Fetching users.";
    auto users = aclGateway.users();
    DEBUG() << "Fetching moderation statuses.";
    auto moderationStatusesByAgentId =
        getModerationStatusesByAgentId(aclGateway, users);
    DEBUG() << "Fetching profiles.";
    const auto profiles = Profiles(socialProfileGateway.getAllUserProfiles());

    Users result;
    for (const auto& user: users) {
        const auto& moderationStatus = moderationStatusesByAgentId[user.id()];
        if (moderationStatus.empty()) {
            WARN() << "Unable to get moderation status for uid "
                   << user.uid();
        }
        result.emplace_back(user,
                            moderationStatus, profiles.getByUid(user.uid()));
    }
    return result;
}

} // namespace maps::wiki::stat_users_dump_worker
