#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/lib/commit_loader.h>
#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/lib/load_helpers.h>

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/libs/gdpr/include/user.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/batch.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/robot.h>
#include <maps/wikimap/mapspro/libs/revision/include/yandex/maps/wiki/revision/filters.h>
#include <maps/wikimap/mapspro/libs/revision/include/yandex/maps/wiki/revision/revisionsgateway.h>

#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/common/include/profiletimer.h>
#include <maps/libs/log8/include/log8.h>

namespace maps::wiki::user_edits_metrics {

namespace chr = std::chrono;
namespace gl = geolib3;
namespace rev = revision;
namespace rf = revision::filters;

using namespace std::string_literals;

namespace {

const size_t LOAD_BATCH_SIZE = 5000;

UserIds getYandexUids(pqxx::transaction_base& txnCore)
{
    UserIds yandexUids;
    acl::ACLGateway gw(txnCore);

    const auto NO_LIMIT = std::numeric_limits<int64_t>::max();
    for (const std::string& roleName: {"yandex-moderator", "cartographer"}) {
        auto role = gw.role(roleName);
        auto users = gw.users(
                /* groupId = */ 0,
                role.id(),
                /* aoiId = */ 0,
                /* status = */ std::nullopt,
                /* page = */ 1,
                /* perPage = */ NO_LIMIT);
        for (const auto& user: users.value()) {
            yandexUids.insert(user.uid());
        }
    }

    return yandexUids;
}

std::optional<CommitData>
createCommitData(
    const rev::Commit& commit,
    const UserIds& yandexUids,
    const std::unordered_map<rev::DBID, social::Task>& tasksByCommitId)
{
    auto uid = gdpr::User(commit.createdBy()).realUid();
    if (yandexUids.count(uid)) {
        return std::make_optional<CommitData>(commit, UserType::Yandex);
    }

    auto taskIt = tasksByCommitId.find(commit.id());
    if (taskIt == tasksByCommitId.end()) {
        // autoapproved commit
        return std::make_optional<CommitData>(commit, UserType::Trusted);
    }

    // moderated commit
    const auto& task = taskIt->second;
    if (task.isResolved() && (task.resolved().resolution() == social::ResolveResolution::Revert) ||
        task.isClosed()   && (task.closed().resolution()   == social::CloseResolution::Revert))
    {
        // reverted commit
        return std::nullopt;
    }

    if (task.isResolved()) {
        if (task.resolved().uid() == common::ROBOT_UID ||
            task.resolved().uid() == uid)
        {
            return std::make_optional<CommitData>(commit, task, UserType::Trusted);
        }
    }

    return std::make_optional<CommitData>(commit, task, UserType::Common);
}

} // namespace

CommitLoader::CommitLoader(
        pqxx::transaction_base& txnCore,
        pqxx::transaction_base& txnSocial,
        pqxx::transaction_base& txnViewTrunk,
        RegionsData regionsData,
        AoiRegionsData aoiRegionsData,
        TimePoint startTime,
        chr::seconds interval)
    : txnCore_(txnCore)
    , txnViewTrunk_(txnViewTrunk)
    , commitRegions_(txnCore)
    , socialGw_(txnSocial)
    , regionsData_(std::move(regionsData))
    , aoiRegionsData_(std::move(aoiRegionsData))
    , interval_(interval)
    , curMinTime_(startTime)
{
    branchesData_ = loadBranchInfos(txnCore_);
    yandexUids_ = getYandexUids(txnCore_);

    curMaxCommitId_ = rev::RevisionsGateway(txnCore_).headCommitId();
}

void CommitLoader::loadNextBatch()
{
    curBatch_.clear();
    curMaxTime_ = curMinTime_;
    curMinTime_ = curMaxTime_ - interval_;

    ProfileTimer pt;

    auto events = socialGw_.loadEditEventsByCreationInterval(social::DateTimeCondition(curMinTime_, curMaxTime_));

    common::applyBatchOp(
        events,
        LOAD_BATCH_SIZE,
        [&](const social::Events& batchEvents) {
            rev::DBIDSet commitIds;
            for (const auto& event : batchEvents) {
                ASSERT(event.commitData());
                commitIds.emplace(event.commitData()->commitId());
            }

            auto filter = rf::CommitAttr::id().in(commitIds);

            auto batchCommits = rev::Commit::load(txnCore_, filter);
            batchCommits.sort(
                [](const rev::Commit& a, const rev::Commit& b) {
                    return a.id() > b.id();
                });

            std::unordered_map<rev::DBID, social::Event> eventsByCommitId;
            for (const auto& event : batchEvents) {
                ASSERT(event.commitData());
                auto commitId = event.commitData()->commitId();
                eventsByCommitId.emplace(commitId, event);
            }

            std::unordered_map<rev::DBID, social::Task> tasksByCommitId;
            for (auto&& task : socialGw_.loadEditTasksByCommitIds(commitIds)) {
                auto commitId = task.commitId();
                tasksByCommitId.emplace(commitId, std::move(task));
            }

            auto regionsByCommitId = commitRegions_.getCommitRegions(commitIds);

            for (const auto& commit : batchCommits) {
                addCommitData(commit, eventsByCommitId, tasksByCommitId, regionsByCommitId);
            }
        });

    INFO() << "Loaded " << curBatch_.size() << " commits in " << pt.getElapsedTime();
}

void CommitLoader::addCommitData(
    const rev::Commit& commit,
    const std::unordered_map<rev::DBID, social::Event>& eventsByCommitId,
    const std::unordered_map<rev::DBID, social::Task>& tasksByCommitId,
    const std::unordered_map<rev::DBID, revision::DBIDSet>& regionsByCommitId)
{
    auto eventIt = eventsByCommitId.find(commit.id());
    ASSERT(eventIt != eventsByCommitId.end());
    ASSERT(eventIt->second.commitData());

    const auto& bbox = eventIt->second.commitData()->bbox();
    if (!bbox) {
        return;
    }

    auto commitData = createCommitData(commit, yandexUids_, tasksByCommitId);
    if (!commitData) {
        return;
    }

    commitData->addRegionEvents(
        txnViewTrunk_, commit.stableBranchId(), branchesData_, regionsByCommitId, regionsData_, *bbox);
    commitData->addAoiRegions(findAoiNames(*bbox));

    curBatch_.emplace_back(std::move(*commitData));
}

std::vector<std::string> CommitLoader::findAoiNames(const geolib3::BoundingBox& bbox) const
{
    std::vector<std::string> aoiNames;

    for (const auto& pair : aoiRegionsData_) {
        const auto& name = pair.first;
        const auto& geometry = pair.second;
        if (geolib3::spatialRelation(bbox, geometry, gl::Intersects)) {
            aoiNames.push_back(name);
        }
    }

    return aoiNames;
}

} // namespace maps::wiki::user_edits_metrics
