#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/lib/load_helpers.h>

#include <maps/libs/enum_io/include/enum_io.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/wikimap/mapspro/libs/revision/include/yandex/maps/wiki/revision/branch_manager.h>
#include <maps/wikimap/mapspro/libs/revision/include/yandex/maps/wiki/revision/revisionsgateway.h>

namespace maps::wiki::user_edits_metrics {

namespace {

using namespace std::string_literals;

const auto AOIS_XPATH = "/config/services/tasks/user_edits_metrics/aois"s;
const auto AOI_NODE = "aoi"s;
const auto ATTR_ID = "id"s;
const auto ATTR_NAME = "name"s;
const auto CAT_AOI = "cat:aoi"s;

const auto ATTRIBUTE_ID_SEP = ':';

const auto LOAD_REGIONS_DATA_QUERY =
    "SELECT object_id, lower_text_data"
    " FROM vrevisions_trunk.suggest_data"
    " WHERE categories ? 'cat:region' AND lang='en'"s;


std::optional<std::pair<Event, std::string>>
toEventRegionPair(const std::string& branchAttributeName)
{
    auto sepPos = branchAttributeName.find(ATTRIBUTE_ID_SEP);
    if (sepPos == std::string::npos) {
        return std::nullopt;
    }

    auto event = enum_io::tryFromString<Event>(branchAttributeName.substr(0, sepPos));
    if (!event) {
        return std::nullopt;
    }

    return std::make_pair(
        event.value(),
        branchAttributeName.substr(sepPos + 1)
    );
}

} //namespace

MetricsConfig loadMetricsConfig(const common::ExtendedXmlDoc& cfg)
{
    MetricsConfig metricsConfig;

    auto aoisNode = cfg.node(AOIS_XPATH, /*quiet =*/true);
    if (aoisNode.isNull()) {
        return metricsConfig;
    }

    auto aoiNodes = aoisNode.nodes(AOI_NODE, /*quiet =*/true);
    for (size_t i = 0; i < aoiNodes.size(); ++i) {
        auto id = aoiNodes[i].attr<revision::DBID>(ATTR_ID);
        auto name = aoiNodes[i].attr<std::string>(ATTR_NAME);
        REQUIRE(!name.empty(), "Name is empty for aoi " << id);
        metricsConfig.aoiIdToName.emplace(id, std::move(name));
    }

    return metricsConfig;
}

AoiRegionsData loadAoiRegions(
    pqxx::transaction_base& txnCore,
    const MetricsConfig& metricsConfig)
{
    if (metricsConfig.aoiIdToName.empty()) {
        return {};
    }

    std::vector<revision::DBID> aoiIds;
    for (const auto& [aoiId, _] : metricsConfig.aoiIdToName) {
        aoiIds.push_back(aoiId);
    }

    revision::RevisionsGateway gateway(txnCore);
    auto snapshot = gateway.snapshot(gateway.headCommitId());

    AoiRegionsData aoiData;
    for (const auto& [id, rev] : snapshot.objectRevisions(aoiIds)) {
        REQUIRE(rev.data().attributes, "Object " << rev.id() << " has no attributes");
        REQUIRE(rev.data().attributes->count(CAT_AOI), "Object " << rev.id() << " is not an aoi");
        REQUIRE(rev.data().geometry, "Object " << rev.id() << " has no geometry");

        auto it = metricsConfig.aoiIdToName.find(id);
        auto geometry = geolib3::WKB::read<geolib3::Polygon2>(*rev.data().geometry);

        aoiData.emplace(it->second, std::move(geometry));
    }

    return aoiData;
}

BranchesById loadBranchInfos(pqxx::transaction_base& txnCore)
{
    BranchesById result;
    revision::BranchManager branchManager(txnCore);
    auto branches = branchManager.load({
            {revision::BranchType::Stable, revision::BranchManager::UNLIMITED},
            {revision::BranchType::Archive, revision::BranchManager::UNLIMITED},
            {revision::BranchType::Deleted, revision::BranchManager::UNLIMITED},});
    for (const auto& branch : branches) {
        BranchInfo branchInfo;
        branchInfo.created = chrono::parseSqlDateTime(branch.createdAt());

        for (const auto& [attributeName, time]: branch.attributes()) {
            const auto eventRegionName = toEventRegionPair(attributeName);
            if (!eventRegionName) {
                continue;
            }

            const auto& [event, regionName] = *eventRegionName;

            if (!BRANCH_BASE_EVENTS.count(event)) {
                continue;
            }

            branchInfo
                .regionToEvents[regionName]
                .emplace(event, chrono::parseSqlDateTime(time));
        }

        result.emplace(branch.id(), std::move(branchInfo));
    }
    return result;
}

RegionsData loadRegionsData(pqxx::transaction_base& txnView)
{
    RegionsData result;
    for (const auto& row : txnView.exec(LOAD_REGIONS_DATA_QUERY)) {
        result.emplace(row[0].as<revision::DBID>(), row[1].as<std::string>());
    }
    return result;
}

} // namespace maps::wiki::user_edits_metrics
