#include "metrics_calculator.h"
#include "metrics_calculator_impl.h"

#include "time_computer.h"

#include <maps/libs/chrono/include/days.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <unordered_set>
#include <unordered_map>

namespace maps::wiki::user_edits_metrics {

using namespace chrono::literals;

namespace {

const std::map<bool, chrono::Days> IN_TRUNK_TO_SLA{
    {false, chrono::Days(1)},
    {true,  chrono::Days(5)}
};

struct UserTypeFilter {
    std::string userType;
    CommitFilter filter;
};

const std::array USER_TYPE_FILTERS = {
    UserTypeFilter{
      USER_TYPE_COMMON,
      [](const CommitData& c) { return c.userType() == UserType::Common; },
    },
    UserTypeFilter{
      USER_TYPE_TRUSTED,
      [](const CommitData& c) { return c.userType() == UserType::Trusted; },
    },
    UserTypeFilter{
      USER_TYPE_NON_YANDEX,
      [](const CommitData& c) { return c.userType() != UserType::Yandex; },
    },
    UserTypeFilter{
      USER_TYPE_YANDEX,
      [](const CommitData& c) { return c.userType() == UserType::Yandex; },
    }
};

enum class DeploymentType: unsigned char {
     BicycleGraph,
     Cams,
     Carparks,
     Geocoder,
     Graph,
     MtrExport,
     PedestrianGraph,
     Renderer,
     EnumBitsetFlagsEnd
};

size_t
maxNumberOfQuantileMetrics(
    const EventToRegions& eventToRegions,
    size_t quantilesNum)
{
    size_t result = 0;
    std::set<std::string> baseRegions;
    const size_t multiplier =
        USER_TYPE_FILTERS.size()
        * quantilesNum
        * 2; // inTrunk: true or false
    for (const auto& [_, regions] : eventToRegions) {
        result += regions.size() * multiplier;
        baseRegions.insert(regions.begin(), regions.end());
    }
    result += NON_BRANCH_BASE_EVENTS.size() * multiplier * baseRegions.size();
    return result;
}

} // namespace


MetricsCalculator::MetricsCalculator(
    std::chrono::seconds window,
    int singleMetricDepth,
    std::vector<double> quantiles,
    EventToRegions eventToRegions,
    EventToRegions eventToRegionsFromCommits,
    CommitLoader& commitLoader)
    : window_(window)
    , singleMetricDepth_(singleMetricDepth)
    , quantiles_(std::move(quantiles))
    , eventToRegions_(std::move(eventToRegions))
    , eventToRegionsFromCommits_(std::move(eventToRegionsFromCommits))
    , finishedMetrics_(0)
    , maxNumberOfQuantileMetrics_(maxNumberOfQuantileMetrics(
        eventToRegions_,
        quantiles_.size()
    ))
    , commitLoader_(commitLoader)
{}

MetricVec MetricsCalculator::calculate(int totalDepth) {
    MetricVec result;

    for (int i = 0; i < totalDepth && !isFinished(); ++i) {
        commitLoader_.loadNextBatch();

        auto minTime = commitLoader_.curMinTime();
        auto maxTime = commitLoader_.curMaxTime();

        INFO() << "Processing commits from: " << chrono::formatSqlDateTime(minTime)
               << " to: " << chrono::formatSqlDateTime(maxTime);
        calculate(&result, minTime);
    }

    return result;
}

void MetricsCalculator::calculate(MetricVec* metrics, chrono::TimePoint time)
{
    finishedMetrics_ = 0;

    for (const auto& event: COMMIT_EVENTS) {
        for (const auto& regionName : eventToRegions_.at(event)) {
            for (const auto& [userType, commitFilter]: USER_TYPE_FILTERS) {
                for (double quantile: quantiles_) {
                    for (const bool inTrunk: {false, true}) {
                        auto metric = quantileMetric(
                            commitLoader_.curBatch(),
                            regionName,
                            eventToRegionsFromCommits_.at(event),
                            commitFilter,
                            event,
                            quantile,
                            inTrunk,
                            userType
                        );
                        if (!metric) {
                            continue;
                        }
                        auto startTime = metricStartTimes_.emplace(metric->name(), time).first->second;
                        auto finishTime = startTime - window_ * singleMetricDepth_;
                        if (time > finishTime) {
                            metrics->emplace_back(*metric);
                        }
                        if (time - window_ <= finishTime) {
                            ++finishedMetrics_;
                        }
                    }
                }
            }
        }
    }
}

bool MetricsCalculator::isFinished() const
{
    return finishedMetrics_ >= maxNumberOfQuantileMetrics_;
}

namespace impl {

MetricVec getMetrics(
    const MetricVec& allMetrics,
    double quantile,
    bool inTrunk,
    const std::string& region,
    const std::string& userType,
    const EventSet& events)
{
    MetricVec result;
    for (const auto& metric: allMetrics) {
        if (std::abs(metric.quantile - quantile) <= std::numeric_limits<double>::epsilon() &&
            metric.inTrunk == inTrunk &&
            metric.region == region &&
            metric.userType == userType &&
            events.count(metric.event))
        {
            result.emplace_back(metric);
        }
    }
    return result;
}

MetricVec sortAndConsolidateMetricsByDays(
    MetricVec metrics,
    ConsolidationFunc consolidationFunc)
{
    if (metrics.empty()) {
        return {};
    }

    std::sort(
        metrics.begin(), metrics.end(),
        [](const Metric& lhs, const Metric& rhs) {
            return lhs.time < rhs.time;
        }
    );

    MetricVec result;

    for (auto metricsIt = metrics.cbegin(); metricsIt != metrics.cend();) {
        const auto dayBeginIt = metricsIt;
        const auto [dayBegin, dayEnd] = chrono::getComprisingTimeSpan<chrono::Days>(dayBeginIt->time);
        while (metricsIt != metrics.cend() && metricsIt->time < dayEnd) {
            ++metricsIt;
        }
        const auto consolidatedMetric = consolidationFunc(dayBegin, dayBeginIt, metricsIt);
        if (consolidatedMetric) {
            result.emplace_back(*consolidatedMetric);
        }
    }

    return result;
}

/// @note There must be at least one metric for each of needed events
/// bucket. Otherwise, the value can't be calculated and left empty.
/// @warning All input metrics must have the same quantile, region, inTrunk and
/// the userType.
MetricVec sortAndConsolidateDeployedMetricsByDays(
    const MetricVec& deployedMetrics,
    Event consolidatedMetricEvent,
    const EventSet& neededEvents)
{
    EventSet neededEventsCounter = neededEvents;
    return sortAndConsolidateMetricsByDays(
        deployedMetrics,
        [&](auto dayBegin, auto beginIt, auto endIt) -> std::optional<Metric> {
            std::chrono::seconds maxValue{0};
            std::chrono::seconds maxWorkdaysValue{0};
            for (auto metricIt = beginIt; metricIt != endIt; ++metricIt) {
                REQUIRE(
                    neededEvents.count(metricIt->event) > 0,
                    "Found wrong event " << metricIt->event);
                neededEventsCounter.erase(metricIt->event);
                maxValue = std::max(maxValue, metricIt->value);
                maxWorkdaysValue = std::max(maxWorkdaysValue, metricIt->workdaysValue);
            }

            if (!neededEventsCounter.empty()) {
                return std::nullopt;
            }

            return Metric{
                dayBegin,
                maxValue,
                maxWorkdaysValue,
                beginIt->quantile,
                beginIt->region,
                beginIt->userType,
                consolidatedMetricEvent,
                beginIt->inTrunk
            };
        }
    );
}

MetricVec addEstimatedMissingMetrics(
    MetricVec&& sortedMetrics,
    chrono::TimePoint calculationDate)
{
    if (sortedMetrics.empty()) {
        return std::move(sortedMetrics);
    }

    const auto QUANTILE  = sortedMetrics.front().quantile;
    const auto REGION    = sortedMetrics.front().region;
    const auto USER_TYPE = sortedMetrics.front().userType;
    const auto EVENT     = sortedMetrics.front().event;
    const auto IN_TRUNK  = sortedMetrics.front().inTrunk;

    auto [lastKnownMetricTime, _] = chrono::getComprisingTimeSpan<chrono::Days>(calculationDate + 1_days);
    auto lastKnownValue = std::chrono::seconds(-1_days);
    auto lastKnownWorkdaysValue = -computeDurationMinusWeekends(lastKnownMetricTime, lastKnownMetricTime + 1_days);;
    auto metricsIx = sortedMetrics.size() - 1;
    do {
        const auto& metric = sortedMetrics[metricsIx];

        if (metric.time + 1_days >= lastKnownMetricTime) {
            if (metricsIx == 0) {
                break;
            }
            lastKnownMetricTime    = metric.time;
            lastKnownValue         = metric.value;
            lastKnownWorkdaysValue = metric.workdaysValue;
            metricsIx--;
        } else {
            lastKnownMetricTime -= 1_days;
            lastKnownValue += 1_days;
            lastKnownWorkdaysValue += computeDurationMinusWeekends(lastKnownMetricTime, lastKnownMetricTime + 1_days);

            sortedMetrics.insert(
                sortedMetrics.begin() + metricsIx + 1,
                {
                    lastKnownMetricTime,
                    lastKnownValue,
                    lastKnownWorkdaysValue,
                    QUANTILE,
                    REGION,
                    USER_TYPE,
                    EVENT,
                    IN_TRUNK,
                    Metric::Estimated::Yes
                }
            );
        }
    } while (true);

    return std::move(sortedMetrics);
}

/// Removes estimations unsuitable for calculating metrics in sliding window as
/// those values can grow up during the time and became bigger than SLA
/// value. Then these metrics values will change and that is not desirable
/// behaviour.
///
/// @see NMAPS-9286
MetricVec dropEstimationsInSlaWindow(
    MetricVec&& sortedMetrics,
    chrono::TimePoint calculationDate,
    chrono::Days sla)
{
    auto metricsIt = sortedMetrics.crbegin();
    while (metricsIt != sortedMetrics.crend() &&
           metricsIt->estimated == Metric::Estimated::Yes &&
           metricsIt->time + sla > calculationDate)
    {
        ++metricsIt;
    }

    sortedMetrics.erase(metricsIt.base(), sortedMetrics.cend());
    return sortedMetrics;
}

MetricVec daysOutOfSlaInSlidingWindow(
    const MetricVec& metrics,
    Event event,
    size_t windowSize,
    double quantile,
    bool inTrunk,
    const std::string& region,
    const std::string& userType)
{
    if (windowSize > metrics.size()) {
        WARN() << "Insufficient information for '" << event << "' metric calculation. "
            "quantile: " << quantile << ", "
            "region: " << region << ", "
            "user type: " << userType << ", "
            "in trunk: " << inTrunk << ", "
            "available metrics size: " << metrics.size() << ", "
            "window size: " << windowSize << ".";
        return {};
    }

    std::chrono::seconds value{0};
    std::chrono::seconds workdaysValue{0};

    size_t beginWndIx = 0;
    size_t endWndIx = windowSize - 1;
    const chrono::Days sla = IN_TRUNK_TO_SLA.at(inTrunk);

    // Accumulating values in the window from the very beginning.
    for (auto metricIx = beginWndIx; metricIx <= endWndIx; ++metricIx) {
        if (metrics[metricIx].value > sla) {
            value += 1_days;
        }
        if (metrics[metricIx].workdaysValue > sla) {
            workdaysValue += 1_days;
        }
    }

    MetricVec result;
    result.reserve(metrics.size() - windowSize + 1);
    result.emplace_back(
        Metric{
            metrics[endWndIx].time,
            value,
            workdaysValue,
            quantile,
            region,
            userType,
            event,
            inTrunk
        }
    );

    // Move window and recalculate its value;
    while (endWndIx < metrics.size() - 1) {
        if (metrics[beginWndIx].value > sla) {
            value -= 1_days;
        }
        if (metrics[beginWndIx].workdaysValue > sla) {
            workdaysValue -= 1_days;
        }
        ++beginWndIx;

        ++endWndIx;
        if (metrics[endWndIx].value > sla) {
            value += 1_days;
        }
        if (metrics[endWndIx].workdaysValue > sla) {
            workdaysValue += 1_days;
        }

        result.emplace_back(
            Metric{
               metrics[endWndIx].time,
               value,
               workdaysValue,
               quantile,
               region,
               userType,
               event,
               inTrunk
            }
        );
    }

    return result;
}

MetricVec achievableSlaInSlidingWindow(
    const MetricVec& metrics,
    Event event,
    size_t windowSize,
    size_t brokenSlaBudget,
    double quantile,
    bool inTrunk,
    const std::string& region,
    const std::string& userType)
{
    if (windowSize > metrics.size()) {
        WARN() << "Insufficient information for '" << event << "' metric calculation. "
            "quantile: " << quantile << ", "
            "region: " << region << ", "
            "user type: " << userType << ", "
            "available metrics size: " << metrics.size() << ", "
            "window size: " << windowSize << ".";
        return {};
    }

    MetricVec result;
    result.reserve(metrics.size() - windowSize + 1);

    size_t beginWndIx = 0;
    size_t endWndIx = windowSize - 1;

    // Copy values from the window into separate variables.
    using ValuesWindow = std::vector<std::chrono::seconds>;
    ValuesWindow valuesWnd;
    ValuesWindow workdaysValuesWnd;
    valuesWnd.reserve(windowSize);
    workdaysValuesWnd.reserve(windowSize);

    for (auto metricIx = beginWndIx; metricIx <= endWndIx; ++metricIx) {
        valuesWnd.push_back(metrics[metricIx].value);
        workdaysValuesWnd.push_back(metrics[metricIx].workdaysValue);
    }

    // The current value is the minimum value that is fit in to the broken SLA
    // budget. It could be find by sorting values in the window and taking the
    // last fit value.
    while (true) {
        // Find current values.
        ValuesWindow::iterator valueIt = valuesWnd.end() - brokenSlaBudget - 1;
        ValuesWindow::iterator workdaysValueIt = workdaysValuesWnd.end() - brokenSlaBudget - 1;
        std::nth_element(valuesWnd.begin(), valueIt, valuesWnd.end());
        std::nth_element(workdaysValuesWnd.begin(), workdaysValueIt, workdaysValuesWnd.end());

        result.emplace_back(
            Metric{
               metrics[endWndIx].time,
               *valueIt,
               *workdaysValueIt,
               quantile,
               region,
               userType,
               event,
               inTrunk
            }
        );

        // Move the window by replacing values from the beginning by values just
        // past the end.
        valueIt = std::find(valuesWnd.begin(), valuesWnd.end(), metrics[beginWndIx].value);
        workdaysValueIt = std::find(workdaysValuesWnd.begin(), workdaysValuesWnd.end(), metrics[beginWndIx].workdaysValue);
        ++beginWndIx;
        ++endWndIx;
        if (endWndIx >= metrics.size()) {
            break;
        }

        *valueIt = metrics[endWndIx].value;
        *workdaysValueIt = metrics[endWndIx].workdaysValue;
    }

    return result;
}

MetricVec getConsolidatedByDaysMetricsWithEstimations(
    const MetricVec& allMetrics,
    double quantile,
    bool inTrunk,
    const std::string& region,
    const std::string& userType,
    const Event& event,
    const maps::chrono::TimePoint& calculationDate)
{
    const auto deployedMetrics = impl::getMetrics(allMetrics, quantile, inTrunk, region, userType, {event});
    auto consolidatedMetrics = impl::sortAndConsolidateMetricsByDays(
        deployedMetrics,
        [&](auto dayBegin, auto, auto endIt) -> std::optional<Metric> {
            Metric result = *(endIt - 1);
            result.time = dayBegin;
            return result;
        }
    );
    return impl::addEstimatedMissingMetrics(std::move(consolidatedMetrics), calculationDate);
}

} // namespace impl

RegionUserTypeToMetricVec getConsolidatedDeployedMetricsWithEstimations(
    const MetricVec& allMetrics,
    chrono::TimePoint calculationDate,
    double quantile,
    bool inTrunk,
    const RegionToEvents& regionToDeployedEvents)
{
    ASSERT(0 < quantile && quantile <= 1);

    RegionUserTypeToMetricVec result;
    for (const auto& [region, deployedEvents] : regionToDeployedEvents) {
        for (const auto& userType: ALL_USER_TYPES) {
            const auto deployedMetrics = impl::getMetrics(
                allMetrics, quantile, inTrunk, region, userType, deployedEvents);
            auto consolidatedMetrics = impl::sortAndConsolidateDeployedMetricsByDays(
                deployedMetrics, Event::DeployedTotal, deployedEvents);
            consolidatedMetrics = impl::addEstimatedMissingMetrics(std::move(consolidatedMetrics), calculationDate);
            result.emplace(RegionUserType{region, userType}, std::move(consolidatedMetrics));
        }
    }
    return result;
}

RegionUserTypeToMetricVec getConsolidatedMetricsWithEstimations(
    const MetricVec& allMetrics,
    chrono::TimePoint calculationDate,
    Event event,
    double quantile,
    bool inTrunk,
    const std::set<std::string>& regions)
{
    ASSERT(0 < quantile && quantile <= 1);

    RegionUserTypeToMetricVec result;
    for (const auto& region: regions) {
        for (const auto& userType: ALL_USER_TYPES) {
            auto consolidatedMetrics = impl::getConsolidatedByDaysMetricsWithEstimations(
                allMetrics,
                quantile,
                inTrunk,
                region,
                userType,
                event,
                calculationDate
            );
            result.emplace(RegionUserType{region, userType}, std::move(consolidatedMetrics));
        }
    }
    return result;
}

MetricVec daysOutOfSlaInSlidingWindow(
    const RegionUserTypeToMetricVec& regionUserTypeToMetricVec,
    Event event,
    size_t windowSize,
    chrono::TimePoint calculationDate,
    double quantile,
    bool inTrunk)
{
    ASSERT(windowSize > 0);
    ASSERT(0 < quantile && quantile <= 1);

    const chrono::Days sla = IN_TRUNK_TO_SLA.at(inTrunk);
    MetricVec result;
    for (auto [regionUserType, metrics]: regionUserTypeToMetricVec) {
        metrics = impl::dropEstimationsInSlaWindow(std::move(metrics), calculationDate, sla);
        const auto daysOutOfSlaMetrics = impl::daysOutOfSlaInSlidingWindow(
            metrics,
            EVENT_TO_BROKEN_SLA_EVENT.at(event),
            windowSize,
            quantile,
            inTrunk,
            regionUserType.first,
            regionUserType.second
        );
        result.insert(result.cend(), daysOutOfSlaMetrics.cbegin(), daysOutOfSlaMetrics.cend());
    }
    return result;
}

MetricVec achievableSlaInSlidingWindow(
    const RegionUserTypeToMetricVec& regionUserTypeToMetricVec,
    Event event,
    size_t windowSize,
    size_t brokenSlaBudget,
    double quantile,
    bool inTrunk)
{
    ASSERT(windowSize > 0);
    ASSERT(0 < quantile && quantile <= 1);
    ASSERT(0 <= brokenSlaBudget && brokenSlaBudget < windowSize);

    MetricVec result;
    for (const auto& [regionUserType, metrics]: regionUserTypeToMetricVec) {
        const auto achievableSlaMetrics = impl::achievableSlaInSlidingWindow(
            metrics,
            EVENT_TO_REAL_SLA_EVENT.at(event),
            windowSize,
            brokenSlaBudget,
            quantile,
            inTrunk,
            regionUserType.first,
            regionUserType.second
        );
        result.insert(result.cend(), achievableSlaMetrics.cbegin(), achievableSlaMetrics.cend());
    }
    return result;
}

} // namespace maps::wiki::user_edits_metrics
