#include "consolidated_deployment_time.h"

#include "stat_util.h"

namespace maps::wiki::user_edits_metrics {

ConsolidatedDeploymentTimeDimensions::ConsolidatedDeploymentTimeDimensions(
    chrono::TimePoint fielddate,
    Days days,
    std::string region,
    size_t userType,
    unsigned char percentile,
    bool inTrunk
)
    : fielddate_(std::chrono::round<chrono::Days>(fielddate))
    , days_(days)
    , region_(std::move(region))
    , userType_(userType)
    , percentile_(percentile)
    , inTrunk_(inTrunk)
{}


void ConsolidatedDeploymentTimeDimensions::printHeader(csv::OutputStream& os)
{
    os << "fielddate" << "days" << "region" << "user_type" << "percentile" << "in_trunk";
}


void ConsolidatedDeploymentTimeDimensions::print(csv::OutputStream& os) const
{
    os << chrono::formatIntegralDateTime(fielddate_, "%Y-%m-%d")
       << daysToStatDictValue(days_)
       << region_
       << userType_
       << static_cast<unsigned int>(percentile_)
       << inTrunk_;
}

std::map<std::string, std::string> ConsolidatedDeploymentTimeDimensions::keyValues(
    pqxx::transaction_base& txn) const
{
    std::map<std::string, std::string> result;
    result.emplace(
        "fielddate",
        txn.quote(chrono::formatIntegralDateTime(fielddate_, "%Y-%m-%d")));

    result.emplace("days__str", txn.quote(std::string(toString(days_))));
    result.emplace("days", std::to_string(daysToStatDictValue(days_)));
    result.emplace("region", txn.quote(region_));
    result.emplace("user_type__str", txn.quote(statDictValueToUserType(userType_)));
    result.emplace("user_type", std::to_string(userType_));
    result.emplace("percentile", std::to_string(static_cast<unsigned int>(percentile_)));
    result.emplace("in_trunk", inTrunk_ ? "1" : "0");
    result.emplace("in_trunk__str", inTrunk_ ? "'True'" : "'False'");

    return result;
}

void ConsolidatedDeploymentTimeMeasures::printHeader(csv::OutputStream& os)
{
    os << "total" << "total_estimated";
}


void ConsolidatedDeploymentTimeMeasures::print(csv::OutputStream& os) const
{
    os << toString(total) << toString(totalEstimated);
}

std::map<std::string, std::string> ConsolidatedDeploymentTimeMeasures::keyValues() const
{
    std::map<std::string, std::string> result;

    result.emplace("total", toString(total));
    result.emplace("total_estimated", toString(totalEstimated));

    return result;
}

ConsolidatedDeploymentTimeReport::ConsolidatedDeploymentTimeReport()
    : Report("mapspro.consolidated_deployment_time")
    , ReportDataUploader(Report::name)
{}

void ConsolidatedDeploymentTimeReport::upload(pqxx::transaction_base& txn) const
{
    ReportDataUploader::upload(txn, dimensionsToMeasures_);
}

void
ConsolidatedDeploymentTimeReport::add(const RegionUserTypeToMetricVec& regionUserToMetrics)
{
    for (const auto& [_, metrics]: regionUserToMetrics) {
        for (const auto& metric: metrics) {
            REQUIRE(metric.event == Event::DeployedTotal, "Unsupported event type: '" << metric.event << "'.");

            const unsigned char percentile = metric.quantile * 100;
            const auto userType = userTypeToStatDictValue(metric.userType);

            auto& alldaysMeasures = dimensionsToMeasures_[
                ConsolidatedDeploymentTimeDimensions(
                    metric.time, Days::AllDays, metric.region, userType, percentile, metric.inTrunk
                )
            ];
            auto& workdaysMeasures = dimensionsToMeasures_[
                ConsolidatedDeploymentTimeDimensions(
                    metric.time, Days::WorkDays, metric.region, userType, percentile, metric.inTrunk
                )
            ];

            switch (metric.estimated) {
                case Metric::Estimated::No:
                    alldaysMeasures.total = metric.value;
                    workdaysMeasures.total = metric.workdaysValue;
                    break;
                case Metric::Estimated::Yes:
                    alldaysMeasures.totalEstimated = metric.value;
                    workdaysMeasures.totalEstimated = metric.workdaysValue;
                    break;
            }
        }
    }
}

} // namespace maps::wiki::user_edits_metrics
