#include "deployment_sla.h"

#include <maps/libs/common/include/exception.h>

#include "stat_util.h"

namespace maps::wiki::user_edits_metrics {

DeploymentSlaDimensions::DeploymentSlaDimensions(
    chrono::TimePoint fielddate,
    Days days,
    std::string region,
    size_t userType,
    unsigned char percentile,
    size_t window,
    size_t budget,
    bool inTrunk
)
    : fielddate_(std::chrono::round<chrono::Days>(fielddate))
    , days_(days)
    , region_(std::move(region))
    , userType_(userType)
    , percentile_(percentile)
    , window_(window)
    , budget_(budget)
    , inTrunk_(inTrunk)
{}


void DeploymentSlaDimensions::printHeader(csv::OutputStream& os)
{
    os << "fielddate" << "days" << "region" << "user_type"
       << "percentile" << "window" << "budget" << "in_trunk";
}


void DeploymentSlaDimensions::print(csv::OutputStream& os) const
{
    os << chrono::formatIntegralDateTime(fielddate_, "%Y-%m-%d")
       << daysToStatDictValue(days_)
       << region_
       << userType_
       << static_cast<unsigned int>(percentile_)
       << window_
       << budget_
       << inTrunk_;
}

std::map<std::string, std::string> DeploymentSlaDimensions::keyValues(
    pqxx::transaction_base& txn) const
{
    std::map<std::string, std::string> result;
    result.emplace(
        "fielddate",
        txn.quote(chrono::formatIntegralDateTime(fielddate_, "%Y-%m-%d")));

    result.emplace("days__str", txn.quote(std::string(toString(days_))));
    result.emplace("days", std::to_string(daysToStatDictValue(days_)));
    result.emplace("region", txn.quote(region_));
    result.emplace("user_type__str", txn.quote(statDictValueToUserType(userType_)));
    result.emplace("user_type", std::to_string(userType_));
    result.emplace("percentile", std::to_string(static_cast<unsigned int>(percentile_)));
    result.emplace("\"window\"", std::to_string(window_));
    result.emplace("budget", std::to_string(budget_));
    result.emplace("in_trunk", inTrunk_ ? "1" : "0");
    result.emplace("in_trunk__str", inTrunk_ ? "'True'" : "'False'");

    return result;
}


void DeploymentSlaMeasures::printHeader(csv::OutputStream& os)
{
    os <<                  "broken_sla" <<                  "real_sla"
       <<    "bicycle_graph_broken_sla" <<    "bicycle_graph_real_sla"
       <<          "cameras_broken_sla" <<          "cameras_real_sla"
       <<         "carparks_broken_sla" <<         "carparks_real_sla"
       <<         "geocoder_broken_sla" <<         "geocoder_real_sla"
       <<            "graph_broken_sla" <<            "graph_real_sla"
       <<       "mtr_export_broken_sla" <<       "mtr_export_real_sla"
       << "pedestrian_graph_broken_sla" << "pedestrian_graph_real_sla"
       <<         "renderer_broken_sla" <<         "renderer_real_sla";
}


void DeploymentSlaMeasures::print(csv::OutputStream& os) const
{
    os << toString(               brokenSla) << toString(               realSla)
       << toString(   bicycleGraphBrokenSla) << toString(   bicycleGraphRealSla)
       << toString(        camerasBrokenSla) << toString(        camerasRealSla)
       << toString(       carparksBrokenSla) << toString(       carparksRealSla)
       << toString(       geocoderBrokenSla) << toString(       geocoderRealSla)
       << toString(          graphBrokenSla) << toString(          graphRealSla)
       << toString(      mtrExportBrokenSla) << toString(      mtrExportRealSla)
       << toString(pedestrianGraphBrokenSla) << toString(pedestrianGraphRealSla)
       << toString(       rendererBrokenSla) << toString(       rendererRealSla);
}

std::map<std::string, std::string> DeploymentSlaMeasures::keyValues() const
{
    std::map<std::string, std::string> result;

    result.emplace("broken_sla", toString(brokenSla));
    result.emplace("real_sla", toString(realSla));

    result.emplace("bicycle_graph_broken_sla", toString(bicycleGraphBrokenSla));
    result.emplace("bicycle_graph_real_sla", toString(bicycleGraphRealSla));

    result.emplace("cameras_broken_sla", toString(camerasBrokenSla));
    result.emplace("cameras_real_sla", toString(camerasRealSla));

    result.emplace("carparks_broken_sla", toString(carparksBrokenSla));
    result.emplace("carparks_real_sla", toString(carparksRealSla));

    result.emplace("geocoder_broken_sla", toString(geocoderBrokenSla));
    result.emplace("geocoder_real_sla", toString(geocoderRealSla));

    result.emplace("graph_broken_sla", toString(graphBrokenSla));
    result.emplace("graph_real_sla", toString(graphRealSla));

    result.emplace("mtr_export_broken_sla", toString(mtrExportBrokenSla));
    result.emplace("mtr_export_real_sla", toString(mtrExportRealSla));

    result.emplace("pedestrian_graph_broken_sla", toString(pedestrianGraphBrokenSla));
    result.emplace("pedestrian_graph_real_sla", toString(pedestrianGraphRealSla));

    result.emplace("renderer_broken_sla", toString(rendererBrokenSla));
    result.emplace("renderer_real_sla", toString(rendererRealSla));

    return result;
}


DeploymentSlaReport::DeploymentSlaReport()
    : Report("mapspro.deployment_sla")
    , ReportDataUploader(Report::name)
{}

void DeploymentSlaReport::upload(pqxx::transaction_base& txn) const
{
    ReportDataUploader::upload(txn, dimensionsToMeasures_);
}

void
DeploymentSlaReport::add(const MetricVec& metrics, size_t window, size_t budget)
{
    for (const auto& metric: metrics) {
        const unsigned char percentile = metric.quantile * 100;
        const auto userType = userTypeToStatDictValue(metric.userType);

        auto& alldaysMeasures = dimensionsToMeasures_[
            DeploymentSlaDimensions(metric.time, Days::AllDays, metric.region, userType, percentile, window, budget, metric.inTrunk)
        ];
        auto& workdaysMeasures = dimensionsToMeasures_[
            DeploymentSlaDimensions(metric.time, Days::WorkDays, metric.region, userType, percentile, window, budget, metric.inTrunk)
        ];

        switch (metric.event) {
            case Event::DeployedTotalBrokenSla:
                alldaysMeasures.brokenSla = metric.value;
                workdaysMeasures.brokenSla = metric.workdaysValue;
                break;
            case Event::DeployedTotalRealSla:
                alldaysMeasures.realSla = metric.value;
                workdaysMeasures.realSla = metric.workdaysValue;
                break;
            case Event::DeployedBicycleGraphBrokenSla:
                alldaysMeasures.bicycleGraphBrokenSla = metric.value;
                workdaysMeasures.bicycleGraphBrokenSla = metric.workdaysValue;
                break;
            case Event::DeployedBicycleGraphRealSla:
                alldaysMeasures.bicycleGraphRealSla = metric.value;
                workdaysMeasures.bicycleGraphRealSla = metric.workdaysValue;
                break;
            case Event::DeployedCamsBrokenSla:
                alldaysMeasures.camerasBrokenSla = metric.value;
                workdaysMeasures.camerasBrokenSla = metric.workdaysValue;
                break;
            case Event::DeployedCamsRealSla:
                alldaysMeasures.camerasRealSla = metric.value;
                workdaysMeasures.camerasRealSla = metric.workdaysValue;
                break;
            case Event::DeployedCarparksBrokenSla:
                alldaysMeasures.carparksBrokenSla = metric.value;
                workdaysMeasures.carparksBrokenSla = metric.workdaysValue;
                break;
            case Event::DeployedCarparksRealSla:
                alldaysMeasures.carparksRealSla = metric.value;
                workdaysMeasures.carparksRealSla = metric.workdaysValue;
                break;
            case Event::DeployedGeocoderBrokenSla:
                alldaysMeasures.geocoderBrokenSla = metric.value;
                workdaysMeasures.geocoderBrokenSla = metric.workdaysValue;
                break;
            case Event::DeployedGeocoderRealSla:
                alldaysMeasures.geocoderRealSla = metric.value;
                workdaysMeasures.geocoderRealSla = metric.workdaysValue;
                break;
            case Event::DeployedGraphBrokenSla:
                alldaysMeasures.graphBrokenSla = metric.value;
                workdaysMeasures.graphBrokenSla = metric.workdaysValue;
                break;
            case Event::DeployedGraphRealSla:
                alldaysMeasures.graphRealSla = metric.value;
                workdaysMeasures.graphRealSla = metric.workdaysValue;
                break;
            case Event::DeployedMtrExportBrokenSla:
                alldaysMeasures.mtrExportBrokenSla = metric.value;
                workdaysMeasures.mtrExportBrokenSla = metric.workdaysValue;
                break;
            case Event::DeployedMtrExportRealSla:
                alldaysMeasures.mtrExportRealSla = metric.value;
                workdaysMeasures.mtrExportRealSla = metric.workdaysValue;
                break;
            case Event::DeployedPedestrianGraphBrokenSla:
                alldaysMeasures.pedestrianGraphBrokenSla = metric.value;
                workdaysMeasures.pedestrianGraphBrokenSla = metric.workdaysValue;
                break;
            case Event::DeployedPedestrianGraphRealSla:
                alldaysMeasures.pedestrianGraphRealSla = metric.value;
                workdaysMeasures.pedestrianGraphRealSla = metric.workdaysValue;
                break;
            case Event::DeployedRendererBrokenSla:
                alldaysMeasures.rendererBrokenSla = metric.value;
                workdaysMeasures.rendererBrokenSla = metric.workdaysValue;
                break;
            case Event::DeployedRendererRealSla:
                alldaysMeasures.rendererRealSla = metric.value;
                workdaysMeasures.rendererRealSla = metric.workdaysValue;
                break;
            default:
                throw RuntimeError() << "Unsupported event type: '" << metric.event << "'.";
        }
    }
}


bool DeploymentSlaMeasures::realSlaFilled() const
{
    // Do not consider mtr_export: it's not necessary for all regions
    // Do not consider broken sla: usually we can't caclculate it for new dates
    // that is expected behaviour
    return realSla.has_value() &&
        bicycleGraphRealSla.has_value() &&
        camerasRealSla.has_value() &&
        carparksRealSla.has_value() &&
        geocoderRealSla.has_value() &&
        graphRealSla.has_value() &&
        pedestrianGraphRealSla.has_value() &&
        rendererRealSla.has_value();
}


void DeploymentSlaReport::removeIncompleteDimensions()
{
    // Stat clears all measures for specific dimension before upload
    // new data for this dimension. So if we can't calculate some measures for dimension
    // we will rewrite old data in report with nulls.
    // Clear incomplete dimensions from report to prevent it.
    for (auto it = dimensionsToMeasures_.begin(); it != dimensionsToMeasures_.end();) {
        const auto& measures = it->second;
        if (!measures.realSlaFilled()) {
            it = dimensionsToMeasures_.erase(it);
        } else {
            ++it;
        }
    }
}

} // namespace maps::wiki::user_edits_metrics
