#include "time_computer.h"

#include <library/cpp/resource/resource.h>

#include <boost/date_time/posix_time/posix_time.hpp>

#include <ctime>

namespace maps::wiki::user_edits_metrics {

namespace chr = std::chrono;
namespace bg = boost::gregorian;
using boost::posix_time::ptime;

namespace {

constexpr auto ONE_DAY = chr::hours(24);
constexpr auto MOSCOW_UTC_OFFSET = chr::hours(3);
const auto EPOCH = ptime(bg::date(1970,1,1));

bg::date convertToDate(chrono::TimePoint time)
{
    auto timeT = chrono::TimePoint::clock::to_time_t(
        std::chrono::time_point_cast<chrono::TimePoint::clock::duration>(time));

    struct tm parts;
    localtime_r(&timeT, &parts);

    return bg::date_from_tm(parts);
}

chrono::TimePoint convertToTimePoint(const bg::date& date)
{
    return chrono::TimePoint(
        chr::seconds((ptime(date) - EPOCH).total_seconds()) - MOSCOW_UTC_OFFSET);
}

bool isHoliday(bg::date date)
{
    static const auto holidays = parseDates(NResource::Find(HOLIDAYS_DATA_RESOURCE_KEY));
    static const auto workdays = parseDates(NResource::Find(WORKDAYS_DATA_RESOURCE_KEY));

    if (holidays.count(date)) {
        return true;
    } else if (workdays.count(date)) {
        return false;
    }

    return date.day_of_week().as_number() == 6 || //Saturday
        date.day_of_week().as_number() == 0; //Sunday
}

} // namespace

chr::seconds computeDurationMinusWeekends(chrono::TimePoint startTime, chrono::TimePoint endTime)
{
    if (startTime >= endTime) {
        return chr::seconds(0);
    }

    auto startDate = convertToDate(startTime);
    auto endDate = convertToDate(endTime);

    if (startDate == endDate) {
        return chr::duration_cast<chr::seconds>(endTime - startTime);
    }

    chr::seconds result(0);

    if (!isHoliday(startDate)) {
        auto startDateEnd = convertToTimePoint(startDate + bg::days(1));
        result += chr::duration_cast<chr::seconds>(startDateEnd - startTime);
    }

    bg::day_iterator itr(startDate);
    ++itr;
    while (itr < endDate) {
        if (!isHoliday(*itr)) {
            result += ONE_DAY;
        }
        ++itr;
    }

    if (!isHoliday(endDate)) {
        auto endDateBegin = convertToTimePoint(endDate);
        result += chr::duration_cast<chr::seconds>(endTime - endDateBegin);
    }

    return result;
}

std::set<boost::gregorian::date> parseDates(const std::string& data)
{
    std::set<boost::gregorian::date> result;

    std::string line;
    std::istringstream stream(data);
    while (!stream.eof()) {
        std::getline(stream, line);
        if (!line.empty()) {
            result.emplace(bg::from_simple_string(line));
        }
    }

    return result;
}

bool isHoliday(chrono::TimePoint date)
{
    return isHoliday(convertToDate(date));
}

} // namespace maps::wiki::user_edits_metrics
