#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/lib/commit_loader.h>
#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/lib/load_helpers.h>
#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/lib/util.h>
#include <maps/wikimap/mapspro/libs/revision_meta/include/commit_regions.h>

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/commit_manager.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/social/gateway.h>
#include <yandex/maps/wiki/unittest/arcadia.h>

#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/serialization.h>

#include <library/cpp/testing/unittest/registar.h>

namespace maps::wiki::user_edits_metrics::tests {

using namespace std::chrono_literals;
using namespace std::string_literals;

namespace {

const revision::UserID TEST_USER_ID = 1;

const revision::DBID CIS1_REGION_ID = 123;
const revision::DBID TR_REGION_ID = 456;

const auto CIS1_REGION = "cis1"s;
const auto TR_REGION = "tr"s;
const auto MOSCOW_AOI = "moscow"s;

const std::unordered_set<std::string> ALL_REGIONS = {
    CIS1_REGION,
    TR_REGION,
    MOSCOW_AOI,
    ALL_REGIONS_REGION
};

const auto MOSCOW_AOI_GEOMETRY = "POLYGON((50 50, 55 50, 55 55, 50 55, 50 50))"s;

const auto EXPORT_TR_TIMEPOINT = "2018-05-12 01:00:00+00:00"s;
const auto EXPORT_CIS1_TIMEPOINT = "2018-05-12 01:00:00+00:00"s;
const auto DEPLOYED_TR_TIMEPOINT = "2018-05-18 01:00:00+00:00"s;

const auto USER_CREATED_OR_UNBANNED_TIME = "2010-10-10 10:10:10"s;

const revision::Attributes COMMIT_ATTRIBUTES = { {"description", "commit"} };

void prepareAclData(pqxx::transaction_base& txn)
{
    acl::ACLGateway aclGateway(txn);
    aclGateway.createRole("yandex-moderator", "");
    aclGateway.createRole("cartographer", "");
}

std::string createGeoBoundsFromMercator(const geolib3::Point2& p1, const geolib3::Point2& p2)
{
    geolib3::BoundingBox mercatorBbox(p1, p2);

    auto geoBbox = geolib3::convertMercatorToGeodetic(mercatorBbox);

    return "[" + std::to_string(geoBbox.minX()) + ", "
        + std::to_string(geoBbox.minY()) + ", "
        + std::to_string(geoBbox.maxX()) + ", "
        + std::to_string(geoBbox.maxY()) + "]";
}

revision::DBID createCommitWithBboxInBranch(
    pqxx::transaction_base& txn,
    double xMin, double yMin, double xMax, double yMax,
    const std::set<revision::DBID>& regionIds)
{
    revision::RevisionsGateway revisionGateway(txn);
    std::list<revision::RevisionsGateway::NewRevisionData> data = {
        { revisionGateway.acquireObjectId(), {} }
    };
    auto commit = revisionGateway.createCommit(
        data.begin(), data.end(),
        TEST_USER_ID,
        COMMIT_ATTRIBUTES);

    social::Gateway socialGateway(txn);
    auto event = socialGateway.createCommitEvent(
        TEST_USER_ID,
        social::CommitData(
            revision::TRUNK_BRANCH_ID,
            commit.id(),
            "", //action
            createGeoBoundsFromMercator(geolib3::Point2(xMin, yMin), geolib3::Point2(xMax, yMax))),
        std::nullopt, //primary object id
        {});
    socialGateway.createAcceptedTask(event, USER_CREATED_OR_UNBANNED_TIME);

    revision_meta::CommitRegions commitRegions(txn);
    commitRegions.push(commit.id(), regionIds);

    revision::CommitManager commitManager(txn);
    commitManager.approve({ commit.id() });

    return commit.id();
}

void prepareBranchAndCommitData(pqxx::transaction_base& txn)
{
    revision::BranchManager branchManager(txn);
    branchManager.createApproved(TEST_USER_ID, {});

    auto branch2 = branchManager.createStable(TEST_USER_ID, {
        { "exported:tr", EXPORT_TR_TIMEPOINT },
        { "exported:cis1", EXPORT_CIS1_TIMEPOINT },
        { "deployed_renderer:tr", DEPLOYED_TR_TIMEPOINT },
        { "deployed_renderer_datatesting:tr", DEPLOYED_TR_TIMEPOINT },
        { "invalid_event:cis1", EXPORT_CIS1_TIMEPOINT}
    });
    branch2.setState(txn, revision::BranchState::Normal);

    //in cis1 in branch 2
    createCommitWithBboxInBranch(txn, 10, 10, 30, 30, { CIS1_REGION_ID });

    //in tr in branch 2
    createCommitWithBboxInBranch(txn, 110, 10, 130, 30, { TR_REGION_ID });

    //in cis1 and tr in branch 2
    createCommitWithBboxInBranch(txn, 80, 10, 130, 30, { CIS1_REGION_ID, TR_REGION_ID });

    //in cis1 and moscow in branch 2
    auto commit4Id = createCommitWithBboxInBranch(txn, 50, 50, 60, 60, { CIS1_REGION_ID });

    revision::CommitManager commitManager(txn);
    commitManager.mergeApprovedToStable(commit4Id);
    branch2.finish(txn, TEST_USER_ID);

    auto branch3 = branchManager.createStable(TEST_USER_ID, {
        { "other_attribute", "1" },
        { "in_stable:tr", "1" }
    });
    branch3.setState(txn, revision::BranchState::Normal);

    //in cis1 in branch 3
    auto commit5Id = createCommitWithBboxInBranch(txn, 10, 10, 30, 30, { CIS1_REGION_ID });

    commitManager.mergeApprovedToStable(commit5Id);

    //in cis1 in trunk
    createCommitWithBboxInBranch(txn, 10, 10, 30, 30, { CIS1_REGION_ID });
}

void checkCommitData(
    const CommitData& commitData,
    const std::unordered_map<std::string, EventSet>& expectedRegionToEventSet)
{
    UNIT_ASSERT(commitData.userType() == UserType::Trusted);

    for (const auto& regionName : ALL_REGIONS) {
        if (expectedRegionToEventSet.count(regionName)) {
            UNIT_ASSERT_C(
                commitData.insideRegion(regionName),
                "Commit " << commitData.id() << " must have region '" << regionName << "'");
        } else {
            UNIT_ASSERT_C(
                !commitData.insideRegion(regionName),
                "Commit " << commitData.id() << " must not have region '" << regionName << "'");
        }
    }

    for (const auto& [regionName, expectedEventSet]: expectedRegionToEventSet) {
        for (const auto& eventName : BRANCH_BASE_EVENTS) {
            if (expectedEventSet.count(eventName)) {
                UNIT_ASSERT_C(
                    commitData.findEvent(regionName, eventName),
                    "Commit " << commitData.id()
                        << " must have event '" << eventName
                        << "' in the region '" << regionName << "'");
            } else {
                UNIT_ASSERT_C(
                    !commitData.findEvent(regionName, eventName),
                    "Commit " << commitData.id()
                        << " must not have event '" << eventName
                        << "' in the region '" << regionName << "'");
            }
        }
    }
}

} // namespace

Y_UNIT_TEST_SUITE_F(loader, unittest::ArcadiaDbFixture)
{

Y_UNIT_TEST(test_branch_info)
{
    auto txn = pool().masterWriteableTransaction();

    prepareBranchAndCommitData(*txn);

    auto branchesById = loadBranchInfos(*txn);

    UNIT_ASSERT_VALUES_EQUAL(branchesById.size(), 2);

    auto it = branchesById.begin();
    const auto& actualBranch2Events = it->second.regionToEvents;
    const RegionToEventToTime expectedBranch2Events = {
        { TR_REGION, {
            { Event::Exported, chrono::parseIsoDateTime(EXPORT_TR_TIMEPOINT) },
            { Event::DeployedRenderer, chrono::parseIsoDateTime(DEPLOYED_TR_TIMEPOINT) } }},
        { CIS1_REGION, {
            { Event::Exported, chrono::parseIsoDateTime(EXPORT_CIS1_TIMEPOINT) } }},
    };
    UNIT_ASSERT_EQUAL(actualBranch2Events, expectedBranch2Events);

    ++it;
    const auto& actualBranch3Events = it->second.regionToEvents;
    const RegionToEventToTime expectedBranch3Events = {};
    UNIT_ASSERT_EQUAL(actualBranch3Events, expectedBranch3Events);
}

Y_UNIT_TEST(test_commit_loader)
{
    auto txn = pool().masterWriteableTransaction();

    prepareAclData(*txn);
    prepareBranchAndCommitData(*txn);

    RegionsData regionsData = {
        { CIS1_REGION_ID, CIS1_REGION },
        { TR_REGION_ID, TR_REGION }
    };

    AoiRegionsData aoiRegionsData = {
        { MOSCOW_AOI, geolib3::WKT::read<geolib3::Polygon2>(MOSCOW_AOI_GEOMETRY) }
    };

    const auto startTime = chrono::TimePoint::clock::now();
    const auto interval = 24h;

    CommitLoader commitLoader(
        *txn, //core
        *txn, //social
        *txn, //viewTrunk
        regionsData,
        aoiRegionsData,
        startTime,
        interval);

    commitLoader.loadNextBatch();
    const auto& batch = commitLoader.curBatch();
    UNIT_ASSERT_VALUES_EQUAL(batch.size(), 6);

    //commit 6 in trunk
    checkCommitData(batch[0], {
            { CIS1_REGION, { } },
            { ALL_REGIONS_REGION, { } }
        });

    //commit 5 in branch 3
    checkCommitData(batch[1], {
            { CIS1_REGION, { Event::InStable } },
            { ALL_REGIONS_REGION, { Event::InStable } }
        });

    //commit 4 in branch 2
    checkCommitData(batch[2], {
            { CIS1_REGION, { Event::InStable, Event::Exported } },
            { MOSCOW_AOI, { Event::InStable, Event::Exported } },
            { ALL_REGIONS_REGION, { Event::InStable, Event::Exported } }
        });

    //commit 3 in branch 2
    checkCommitData(batch[3], {
            { CIS1_REGION, { Event::InStable, Event::Exported } },
            { TR_REGION, { Event::InStable, Event::Exported, Event::DeployedRenderer } },
            { ALL_REGIONS_REGION, { Event::InStable, Event::Exported, Event::DeployedRenderer } }
        });

    //commit 2 in branch 2
    checkCommitData(batch[4], {
            { TR_REGION, { Event::InStable, Event::Exported, Event::DeployedRenderer } },
            { ALL_REGIONS_REGION, { Event::InStable, Event::Exported, Event::DeployedRenderer } }
        });

    //commit 1 in branch 2
    checkCommitData(batch[5], {
            { CIS1_REGION, { Event::InStable, Event::Exported } },
            { ALL_REGIONS_REGION, { Event::InStable, Event::Exported } }
        });

    commitLoader.loadNextBatch();
    const auto& batch2 = commitLoader.curBatch();
    UNIT_ASSERT(batch2.empty());
}

} // Y_UNIT_TEST_SUITE

} // namespace maps::wiki::user_edits_metrics::tests
