#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/lib/metrics.h>
#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/lib/metrics_calculator.h>
#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/lib/metrics_calculator_impl.h>
#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/tests/small_tests/metric_creator.h>

#include <maps/libs/introspection/include/comparison.h>
#include <maps/libs/chrono/include/days.h>
#include <library/cpp/testing/unittest/registar.h>

#include <chrono>
#include <tuple>

namespace maps::wiki::user_edits_metrics {

inline auto introspect(const Metric& metric)
{
    return std::tie(
        metric.time,
        metric.value,
        metric.workdaysValue,
        metric.quantile,
        metric.region,
        metric.userType,
        metric.event,
        metric.estimated
    );
}

using introspection::operator<;
using introspection::operator==;

namespace tests {

using namespace std::literals::string_literals;
using namespace std::literals::chrono_literals;
using namespace chrono::literals;

RegionToEvents regionsToDeployedEvents(const std::set<std::string>& regions) {
    RegionToEvents regionToDeployedEvents;
    for (std::string region : regions) {
        regionToDeployedEvents.emplace(std::move(region), DEPLOYED_TO_PRODUCTION_EVENTS);
    }
    return regionToDeployedEvents;
}

const RegionToEvents DEFAULT_REGION_TO_EVENTS = regionsToDeployedEvents({DEFAULT_REGION});


Y_UNIT_TEST_SUITE(metrics_printing)
{
    Y_UNIT_TEST(should_get_name)
    {
        UNIT_ASSERT_STRINGS_EQUAL(
            MetricCreator().quantile(0.13)().name(),
            DEFAULT_REGION + "." + DEFAULT_USER_TYPE + "." + std::string(toString(DEFAULT_EVENT)) + ".in_trunk.13_prc"
        );

        UNIT_ASSERT_STRINGS_EQUAL(
            MetricCreator().quantile(0.13).inTrunk(false)().name(),
            DEFAULT_REGION + "." + DEFAULT_USER_TYPE + "." + std::string(toString(DEFAULT_EVENT)) + ".13_prc"
        );

        UNIT_ASSERT_STRINGS_EQUAL(
            MetricCreator().quantile(0.27).estimated()().name(),
            DEFAULT_REGION + "." + DEFAULT_USER_TYPE + "." + std::string(toString(DEFAULT_EVENT)) + ".in_trunk.27_prc.estimated"
        );
    }
} // Y_UNIT_TEST_SUITE(metrics_printing)

Y_UNIT_TEST_SUITE(metrics_calculation)
{
    Y_UNIT_TEST(should_get_metrics)
    {
        const auto WRONG_QUANTILE = DEFAULT_QUANTILE / 2;

        MetricVec metrics{
            MetricCreator().time("2019-01-25 00:00:00").event(Event::Approved).quantile(WRONG_QUANTILE)(),
            MetricCreator().time("2019-01-26 00:00:00").event(Event::Approved).region("wrong_region")(),
            MetricCreator().time("2019-01-27 00:00:00").event(Event::Approved)(),
            MetricCreator().time("2019-01-28 00:00:00").event(Event::Exported)(),
            MetricCreator().time("2019-01-29 00:00:00").event(Event::Resolved).userType("wrong_user_type")(),
            MetricCreator().time("2019-01-30 00:00:00").event(Event::Resolved)(),
            MetricCreator().time("2019-01-31 00:00:00").event(Event::Resolved).inTrunk(in_trunk::NO)()
        };

        const auto result = impl::getMetrics(
            metrics, DEFAULT_QUANTILE, in_trunk::YES, DEFAULT_REGION, DEFAULT_USER_TYPE, {Event::Approved, Event::Resolved});

        UNIT_ASSERT_EQUAL(result.size(), 2);
        UNIT_ASSERT_EQUAL(result[0], MetricCreator().time("2019-01-27 00:00:00").event(Event::Approved)());
        UNIT_ASSERT_EQUAL(result[1], MetricCreator().time("2019-01-30 00:00:00").event(Event::Resolved)());
    }

    Y_UNIT_TEST(should_sort_and_consolidate_metrics) {
        const MetricVec metrics{
            MetricCreator().time("2019-01-28 12:00:00")(),
            MetricCreator().time("2019-01-27 00:00:00")(),
            MetricCreator().time("2019-01-28 01:00:00")(),
            MetricCreator().time("2019-01-27 23:59:59")(),
            MetricCreator().time("2019-01-28 23:00:00")(),
            MetricCreator().time("2019-01-27 15:00:00")()
        };
        const auto result = impl::sortAndConsolidateMetricsByDays(
            metrics,
            [](auto, auto beginIt, auto) {
                return *beginIt;
            }
        );
        UNIT_ASSERT_EQUAL(result.size(), 2);
        UNIT_ASSERT_EQUAL(result[0].time, chrono::parseSqlDateTime("2019-01-27 00:00:00"));
        UNIT_ASSERT_EQUAL(result[1].time, chrono::parseSqlDateTime("2019-01-28 01:00:00"));
    }

    Y_UNIT_TEST(should_consolidate_needed_metrics_by_max) {
        const MetricVec metrics{
            MetricCreator().time("2019-01-27 00:00:00").value( 5s).workdaysValue(  50s).event(Event::DeployedGeocoder)(),
            MetricCreator().time("2019-01-27 01:01:00").value( 6s).workdaysValue(  20s).event(Event::DeployedCams)(),
            MetricCreator().time("2019-01-27 01:02:00").value( 7s).workdaysValue(  20s).event(Event::DeployedCarparks)(),
            MetricCreator().time("2019-01-27 01:00:00").value( 8s).workdaysValue(  20s).event(Event::DeployedBicycleGraph)(),
            MetricCreator().time("2019-01-27 01:23:45").value( 7s).workdaysValue(  30s).event(Event::DeployedGraph)(),
            MetricCreator().time("2019-01-27 01:24:45").value( 7s).workdaysValue(  30s).event(Event::DeployedMtrExport)(),
            MetricCreator().time("2019-01-27 02:00:00").value( 8s).workdaysValue(  20s).event(Event::DeployedPedestrianGraph)(),
            MetricCreator().time("2019-01-27 23:59:59").value(10s).workdaysValue(  10s).event(Event::DeployedRenderer)(),
            MetricCreator().time("2019-01-28 00:52:00").value( 6s).workdaysValue(  20s).event(Event::DeployedCams)(),
            MetricCreator().time("2019-01-28 00:55:00").value( 7s).workdaysValue(  20s).event(Event::DeployedCarparks)(),
            MetricCreator().time("2019-01-28 01:00:00").value(10s).workdaysValue( 100s).event(Event::DeployedRenderer)(),
            MetricCreator().time("2019-01-28 12:00:00").value( 5s).workdaysValue( 500s).event(Event::DeployedGraph)(),
            MetricCreator().time("2019-01-28 12:01:00").value( 5s).workdaysValue( 500s).event(Event::DeployedMtrExport)(),
            MetricCreator().time("2019-01-28 11:00:00").value( 7s).workdaysValue( 500s).event(Event::DeployedBicycleGraph)(),
            MetricCreator().time("2019-01-28 15:00:00").value( 7s).workdaysValue( 500s).event(Event::DeployedPedestrianGraph)(),
            MetricCreator().time("2019-01-28 23:00:00").value( 1s).workdaysValue(1000s).event(Event::DeployedGeocoder)()
        };
        const auto result = impl::sortAndConsolidateDeployedMetricsByDays(metrics, Event::DeployedTotal, DEPLOYED_TO_PRODUCTION_EVENTS);
        UNIT_ASSERT_EQUAL(result.size(), 2);
        UNIT_ASSERT_EQUAL(
            result[0],
            MetricCreator().time("2019-01-27 00:00:00").value(10s).workdaysValue(50s).event(Event::DeployedTotal)()
        );
        UNIT_ASSERT_EQUAL(
            result[1],
            MetricCreator().time("2019-01-28 00:00:00").value(10s).workdaysValue(1000s).event(Event::DeployedTotal)()
        );
    }

    Y_UNIT_TEST(should_consolidate_only_needed_metrics_by_max) {
        const MetricVec metrics{
            MetricCreator().time("2019-01-27 01:01:00").value( 6s).workdaysValue(  20s).event(Event::DeployedCams)(),
            MetricCreator().time("2019-01-27 01:23:45").value( 7s).workdaysValue(  30s).event(Event::DeployedGraph)(),
            MetricCreator().time("2019-01-28 00:52:00").value( 6s).workdaysValue(  20s).event(Event::DeployedCams)(),
            MetricCreator().time("2019-01-28 12:00:00").value( 5s).workdaysValue( 500s).event(Event::DeployedGraph)()
        };
        const auto result = impl::sortAndConsolidateDeployedMetricsByDays(
            metrics,
            Event::DeployedTotal,
            {Event::DeployedCams, Event::DeployedGraph}
        );
        UNIT_ASSERT_EQUAL(result.size(), 2);
        UNIT_ASSERT_EQUAL(
            result[0],
            MetricCreator().time("2019-01-27 00:00:00").value(7s).workdaysValue(30s).event(Event::DeployedTotal)()
        );
        UNIT_ASSERT_EQUAL(
            result[1],
            MetricCreator().time("2019-01-28 00:00:00").value(6s).workdaysValue(500s).event(Event::DeployedTotal)()
        );
    }

    Y_UNIT_TEST(should_not_consolidate_if_any_needed_metric_absent) {
        UNIT_ASSERT_EQUAL(
            impl::sortAndConsolidateDeployedMetricsByDays(
                {
                    MetricCreator().time("2019-01-25 00:00:00").event(Event::DeployedGraph)(),
                    MetricCreator().time("2019-01-25 00:00:00").event(Event::DeployedRenderer)()
                },
                Event::DeployedTotal,
                DEPLOYED_TO_PRODUCTION_EVENTS)
            .size(),
            0
        );
        UNIT_ASSERT_EQUAL(
            impl::sortAndConsolidateDeployedMetricsByDays(
                {
                    MetricCreator().time("2019-01-25 00:00:00").event(Event::DeployedGraph)(),
                    MetricCreator().time("2019-01-25 00:00:00").event(Event::DeployedRenderer)()
                },
                Event::DeployedTotal,
                {Event::DeployedGeocoder, Event::DeployedGraph, Event::DeployedRenderer})
            .size(),
            0
        );

        UNIT_ASSERT_EQUAL(
            impl::sortAndConsolidateDeployedMetricsByDays(
                {
                    MetricCreator().time("2019-01-25 00:00:00").event(Event::DeployedGeocoder)(),
                    MetricCreator().time("2019-01-25 00:00:00").event(Event::DeployedRenderer)()
                },
                Event::DeployedTotal,
                DEPLOYED_TO_PRODUCTION_EVENTS)
            .size(),
            0
        );

        UNIT_ASSERT_EQUAL(
            impl::sortAndConsolidateDeployedMetricsByDays(
                {
                    MetricCreator().time("2019-01-25 00:00:00").event(Event::DeployedGeocoder)(),
                    MetricCreator().time("2019-01-25 00:00:00").event(Event::DeployedGraph)(),
                },
                Event::DeployedTotal,
                DEPLOYED_TO_PRODUCTION_EVENTS)
            .size(),
            0
        );
    }

    Y_UNIT_TEST(should_not_consolidate_wrong_metrics) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            impl::sortAndConsolidateDeployedMetricsByDays(
                {
                    MetricCreator().time("2019-01-25 00:00:00").event(Event::DeployedGraph)(),
                    MetricCreator().time("2019-01-25 00:00:00").event(Event::Approved)()
                },
                Event::DeployedTotal,
                DEPLOYED_TO_PRODUCTION_EVENTS),
            maps::RuntimeError,
            "Found wrong event approved"
        );
    }

    Y_UNIT_TEST(should_estimate_and_add_missing_metrics) {
        MetricVec metrics{
            MetricCreator().time("2019-01-21 00:00:00")(),
            MetricCreator().time("2019-01-24 00:00:00")(),
            MetricCreator().time("2019-01-26 00:00:00")()
        };

        const auto CALCULATION_DATE = chrono::parseSqlDateTime("2019-01-28 06:00:00");

        const auto result = impl::addEstimatedMissingMetrics(std::move(metrics), CALCULATION_DATE);
        UNIT_ASSERT_EQUAL(result.size(), 8);

        // Week 1 (from Monday to Sunday)
        UNIT_ASSERT_EQUAL(result[0], MetricCreator().time("2019-01-21 00:00:00")());
        UNIT_ASSERT_EQUAL(result[1], MetricCreator().time("2019-01-22 00:00:00").estimated().value(DEFAULT_VALUE + 2_days).workdaysValue(DEFAULT_WORKDAYS_VALUE + 2_days)());
        UNIT_ASSERT_EQUAL(result[2], MetricCreator().time("2019-01-23 00:00:00").estimated().value(DEFAULT_VALUE + 1_days).workdaysValue(DEFAULT_WORKDAYS_VALUE + 1_days)());
        UNIT_ASSERT_EQUAL(result[3], MetricCreator().time("2019-01-24 00:00:00")());
        // 3 hours is subtructed below because of Moscow time zone that is used in workdays calculation.
        UNIT_ASSERT_EQUAL(result[4], MetricCreator().time("2019-01-25 00:00:00").estimated().value(DEFAULT_VALUE + 1_days).workdaysValue(DEFAULT_WORKDAYS_VALUE + 21h)());
        UNIT_ASSERT_EQUAL(result[5], MetricCreator().time("2019-01-26 00:00:00")());
        // Additional 3 hours below because of Moscow time zone that is used in workdays calculation.
        UNIT_ASSERT_EQUAL(result[6], MetricCreator().time("2019-01-27 00:00:00").estimated().value(1_days).workdaysValue(3h)());
        // Week 2
        UNIT_ASSERT_EQUAL(result[7], MetricCreator().time("2019-01-28 00:00:00").estimated().value(0h).workdaysValue(0h)());
    }

    Y_UNIT_TEST(should_drop_excess_estimations) {
        const auto SLA_TIME = 2_days;

        MetricVec metrics{
            MetricCreator().time("2019-01-21 00:00:00").estimated()(),
            MetricCreator().time("2019-01-22 00:00:00").estimated()(),
            MetricCreator().time("2019-01-23 00:00:00").estimated()(),
            MetricCreator().time("2019-01-24 00:00:00").estimated()()
        };

        const auto CALCULATION_DATE = chrono::parseSqlDateTime("2019-01-24 00:00:00");
        const auto result = impl::dropEstimationsInSlaWindow(std::move(metrics), CALCULATION_DATE, SLA_TIME);
        UNIT_ASSERT_EQUAL(metrics.size(), 2);
        UNIT_ASSERT_EQUAL(result[0], MetricCreator().time("2019-01-21 00:00:00").estimated()());
        UNIT_ASSERT_EQUAL(result[1], MetricCreator().time("2019-01-22 00:00:00").estimated()());
    };

    Y_UNIT_TEST(should_not_drop_real_values) {
        const auto SLA_TIME = 2_days;

        MetricVec metrics{
            MetricCreator().time("2019-01-21 00:00:00")(),
            MetricCreator().time("2019-01-22 00:00:00").estimated()(),
            MetricCreator().time("2019-01-23 00:00:00")(),
            MetricCreator().time("2019-01-24 00:00:00").estimated()()
        };

        const auto CALCULATION_DATE = chrono::parseSqlDateTime("2019-01-24 00:00:00");
        const auto result = impl::dropEstimationsInSlaWindow(std::move(metrics), CALCULATION_DATE, SLA_TIME);
        UNIT_ASSERT_EQUAL(metrics.size(), 3);
        UNIT_ASSERT_EQUAL(result[0], MetricCreator().time("2019-01-21 00:00:00")());
        UNIT_ASSERT_EQUAL(result[1], MetricCreator().time("2019-01-22 00:00:00").estimated()());
        UNIT_ASSERT_EQUAL(result[2], MetricCreator().time("2019-01-23 00:00:00")());
    };

    Y_UNIT_TEST(should_consolidate_deployed_metrics_by_time) {
        const MetricVec metrics{
            MetricCreator().time("2019-01-26 00:00:00").value(55s).workdaysValue(7s).event(Event::DeployedGraph)(),
            MetricCreator().time("2019-01-26 01:00:00").value(55s).workdaysValue(6s).event(Event::DeployedPedestrianGraph)(),
            MetricCreator().time("2019-01-26 08:00:00").value(55s).workdaysValue(6s).event(Event::DeployedCams)(),
            MetricCreator().time("2019-01-26 09:00:00").value(55s).workdaysValue(6s).event(Event::DeployedCarparks)(),
            MetricCreator().time("2019-01-26 10:00:00").value(55s).workdaysValue(6s).event(Event::DeployedBicycleGraph)(),
            MetricCreator().time("2019-01-26 12:00:00").value(77s).workdaysValue(5s).event(Event::DeployedGeocoder)(),
            MetricCreator().time("2019-01-26 13:00:00").value(77s).workdaysValue(5s).event(Event::DeployedMtrExport)(),
            MetricCreator().time("2019-01-26 15:00:00").value(33s).workdaysValue(3s).event(Event::DeployedRenderer)(),
            MetricCreator().time("2019-01-27 00:00:00").value(99s).workdaysValue(9s).event(Event::DeployedGraph)(),
            MetricCreator().time("2019-01-27 00:00:00").value(88s).workdaysValue(8s).event(Event::DeployedBicycleGraph)(),
            MetricCreator().time("2019-01-27 00:00:00").value(88s).workdaysValue(8s).event(Event::DeployedCams)(),
            MetricCreator().time("2019-01-27 00:00:00").value(88s).workdaysValue(8s).event(Event::DeployedCarparks)(),
            MetricCreator().time("2019-01-27 00:00:00").value(88s).workdaysValue(8s).event(Event::DeployedMtrExport)(),
            MetricCreator().time("2019-01-27 00:00:00").value(88s).workdaysValue(8s).event(Event::DeployedPedestrianGraph)(),
            MetricCreator().time("2019-01-27 00:00:00").value(99s).workdaysValue(9s).event(Event::DeployedGeocoder)(),
            MetricCreator().time("2019-01-27 00:00:00").value(99s).workdaysValue(9s).event(Event::DeployedRenderer)()
        };

        const auto CALCULATION_DATE = chrono::parseSqlDateTime("2019-01-27 00:00:00");

        const auto result = getConsolidatedDeployedMetricsWithEstimations(
            metrics, CALCULATION_DATE, DEFAULT_QUANTILE, in_trunk::YES, DEFAULT_REGION_TO_EVENTS);
        UNIT_ASSERT_EQUAL(result.size(), 4);
        UNIT_ASSERT(result.at({DEFAULT_REGION, USER_TYPE_TRUSTED}).empty());
        UNIT_ASSERT(result.at({DEFAULT_REGION, USER_TYPE_NON_YANDEX}).empty());
        UNIT_ASSERT(result.at({DEFAULT_REGION, USER_TYPE_YANDEX}).empty());
        const auto& resultMetrics = result.at({DEFAULT_REGION, USER_TYPE_COMMON});
        UNIT_ASSERT_EQUAL(resultMetrics.size(), 2);
        UNIT_ASSERT_EQUAL(
            resultMetrics[0],
            MetricCreator()
                .time("2019-01-26 00:00:00")
                .value(77s)
                .workdaysValue(7s)
                .event(Event::DeployedTotal)()
        );
        UNIT_ASSERT_EQUAL(
            resultMetrics[1],
            MetricCreator()
                .time("2019-01-27 00:00:00")
                .value(99s)
                .workdaysValue(9s)
                .event(Event::DeployedTotal)()
        );
    }

    Y_UNIT_TEST(should_consolidate_by_regions) {
        const MetricVec metrics{
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(33s).workdaysValue(3s).event(Event::DeployedGeocoder)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(55s).workdaysValue(7s).event(Event::DeployedCams)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(55s).workdaysValue(7s).event(Event::DeployedCarparks)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(55s).workdaysValue(7s).event(Event::DeployedGraph)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(44s).workdaysValue(6s).event(Event::DeployedBicycleGraph)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(44s).workdaysValue(6s).event(Event::DeployedPedestrianGraph)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(44s).workdaysValue(6s).event(Event::DeployedMtrExport)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(77s).workdaysValue(5s).event(Event::DeployedRenderer)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(22s).workdaysValue(2s).event(Event::DeployedCams)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(22s).workdaysValue(2s).event(Event::DeployedCarparks)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(33s).workdaysValue(1s).event(Event::DeployedGeocoder)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(22s).workdaysValue(2s).event(Event::DeployedGraph)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(22s).workdaysValue(2s).event(Event::DeployedBicycleGraph)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(22s).workdaysValue(2s).event(Event::DeployedMtrExport)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(22s).workdaysValue(2s).event(Event::DeployedPedestrianGraph)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(11s).workdaysValue(3s).event(Event::DeployedRenderer)()
        };

        const auto CALCULATION_DATE = chrono::parseSqlDateTime("2019-01-26 00:00:00");

        RegionToEvents regionToDeployedEvents = regionsToDeployedEvents({"region1", "region2"});

        const auto result = getConsolidatedDeployedMetricsWithEstimations(
            metrics, CALCULATION_DATE, DEFAULT_QUANTILE, in_trunk::YES, regionToDeployedEvents);
        UNIT_ASSERT_EQUAL(result.size(), 8);
        UNIT_ASSERT(result.at({"region1", USER_TYPE_TRUSTED}).empty());
        UNIT_ASSERT(result.at({"region2", USER_TYPE_TRUSTED}).empty());
        UNIT_ASSERT(result.at({"region1", USER_TYPE_NON_YANDEX}).empty());
        UNIT_ASSERT(result.at({"region2", USER_TYPE_NON_YANDEX}).empty());
        UNIT_ASSERT(result.at({"region1", USER_TYPE_YANDEX}).empty());
        UNIT_ASSERT(result.at({"region2", USER_TYPE_YANDEX}).empty());

        const auto& region1Metrics = result.at({"region1", USER_TYPE_COMMON});
        UNIT_ASSERT_EQUAL(region1Metrics.size(), 1);
        UNIT_ASSERT_EQUAL(
            region1Metrics[0],
            MetricCreator()
                .time("2019-01-26 00:00:00")
                .region("region1")
                .value(77s)
                .workdaysValue(7s)
                .event(Event::DeployedTotal)()
        );

        const auto& region2Metrics = result.at({"region2", USER_TYPE_COMMON});
        UNIT_ASSERT_EQUAL(region2Metrics.size(), 1);
        UNIT_ASSERT_EQUAL(
            region2Metrics[0],
            MetricCreator()
                .time("2019-01-26 00:00:00")
                .region("region2")
                .value(33s)
                .workdaysValue(3s)
                .event(Event::DeployedTotal)()
        );

    }

    Y_UNIT_TEST(should_consolidate_by_single_region) {
        const MetricVec metrics{
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(33s).workdaysValue(3s).event(Event::DeployedGeocoder)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(55s).workdaysValue(7s).event(Event::DeployedCams)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(55s).workdaysValue(7s).event(Event::DeployedGraph)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region1").value(77s).workdaysValue(5s).event(Event::DeployedRenderer)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(22s).workdaysValue(2s).event(Event::DeployedCams)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(33s).workdaysValue(1s).event(Event::DeployedGeocoder)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(88s).workdaysValue(8s).event(Event::DeployedGraph)(),
            MetricCreator().time("2019-01-26 00:00:00").region("region2").value(11s).workdaysValue(3s).event(Event::DeployedRenderer)()
        };
        RegionToEvents region1ToEvents{
            {"region1", {
                Event::DeployedCams,
                Event::DeployedGeocoder,
                Event::DeployedGraph,
            }},
        };
        const auto CALCULATION_DATE = chrono::parseSqlDateTime("2019-01-26 00:00:00");
        const auto result = getConsolidatedDeployedMetricsWithEstimations(
            metrics, CALCULATION_DATE, DEFAULT_QUANTILE, in_trunk::YES, region1ToEvents);
        UNIT_ASSERT_EQUAL(result.size(), 4);
        UNIT_ASSERT(result.at({"region1", USER_TYPE_TRUSTED}).empty());
        UNIT_ASSERT(result.at({"region1", USER_TYPE_NON_YANDEX}).empty());
        UNIT_ASSERT(result.at({"region1", USER_TYPE_YANDEX}).empty());

        UNIT_ASSERT(result.find({"region2", USER_TYPE_COMMON}) == result.end());
        UNIT_ASSERT(result.find({"region2", USER_TYPE_NON_YANDEX}) == result.end());
        UNIT_ASSERT(result.find({"region2", USER_TYPE_TRUSTED}) == result.end());
        UNIT_ASSERT(result.find({"region2", USER_TYPE_YANDEX}) == result.end());

        const auto& region1Metrics = result.at({"region1", USER_TYPE_COMMON});
        UNIT_ASSERT_EQUAL(region1Metrics.size(), 1);
        UNIT_ASSERT_EQUAL(
            region1Metrics[0],
            MetricCreator()
                .time("2019-01-26 00:00:00")
                .region("region1")
                .value(55s)
                .workdaysValue(7s)
                .event(Event::DeployedTotal)()
        );
    }

    Y_UNIT_TEST(should_consolidate_by_user_type) {
        const MetricVec metrics{
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_TRUSTED).value(66s).workdaysValue(6s).event(Event::DeployedCams)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_TRUSTED).value(66s).workdaysValue(6s).event(Event::DeployedCarparks)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_TRUSTED).value(55s).workdaysValue(7s).event(Event::DeployedGeocoder)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_TRUSTED).value(66s).workdaysValue(6s).event(Event::DeployedGraph)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_TRUSTED).value(66s).workdaysValue(6s).event(Event::DeployedBicycleGraph)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_TRUSTED).value(66s).workdaysValue(6s).event(Event::DeployedMtrExport)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_TRUSTED).value(66s).workdaysValue(6s).event(Event::DeployedPedestrianGraph)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_TRUSTED).value(77s).workdaysValue(5s).event(Event::DeployedRenderer)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_NON_YANDEX).value(22s).workdaysValue(2s).event(Event::DeployedCams)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_NON_YANDEX).value(22s).workdaysValue(2s).event(Event::DeployedCarparks)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_NON_YANDEX).value(33s).workdaysValue(1s).event(Event::DeployedGeocoder)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_NON_YANDEX).value(22s).workdaysValue(2s).event(Event::DeployedGraph)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_NON_YANDEX).value(22s).workdaysValue(2s).event(Event::DeployedBicycleGraph)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_NON_YANDEX).value(22s).workdaysValue(2s).event(Event::DeployedMtrExport)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_NON_YANDEX).value(22s).workdaysValue(2s).event(Event::DeployedPedestrianGraph)(),
             MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_NON_YANDEX).value(11s).workdaysValue(3s).event(Event::DeployedRenderer)(),
        };

        const auto CALCULATION_DATE = chrono::parseSqlDateTime("2019-01-26 00:00:00");

        const auto result = getConsolidatedDeployedMetricsWithEstimations(
            metrics, CALCULATION_DATE, DEFAULT_QUANTILE, in_trunk::YES, DEFAULT_REGION_TO_EVENTS);
        UNIT_ASSERT_EQUAL(result.size(), 4);
        UNIT_ASSERT(result.at({DEFAULT_REGION, USER_TYPE_COMMON}).empty());
        UNIT_ASSERT(result.at({DEFAULT_REGION, USER_TYPE_YANDEX}).empty());

        const auto& trustedMetrics = result.at({DEFAULT_REGION, USER_TYPE_TRUSTED});
        UNIT_ASSERT_EQUAL(trustedMetrics.size(), 1);
        UNIT_ASSERT_EQUAL(trustedMetrics[0], MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_TRUSTED).value(77s).workdaysValue(7s).event(Event::DeployedTotal)());

        const auto& nonYandexMetrics = result.at({DEFAULT_REGION, USER_TYPE_NON_YANDEX});
        UNIT_ASSERT_EQUAL(nonYandexMetrics.size(), 1);
        UNIT_ASSERT_EQUAL(nonYandexMetrics[0], MetricCreator().time("2019-01-26 00:00:00").userType(USER_TYPE_NON_YANDEX).value(33s).workdaysValue(3s).event(Event::DeployedTotal)());
    }

    Y_UNIT_TEST(should_consolidate_deployed_metrics_only) {
        const MetricVec metrics{
            MetricCreator().time("2019-01-26 00:00:00").value(44s).workdaysValue(4s).event(Event::DeployedCams)(),
            MetricCreator().time("2019-01-26 00:00:00").value(44s).workdaysValue(4s).event(Event::DeployedCarparks)(),
            MetricCreator().time("2019-01-26 00:00:00").value(11s).workdaysValue(1s).event(Event::DeployedGeocoder)(),
            MetricCreator().time("2019-01-26 00:00:00").value(33s).workdaysValue(3s).event(Event::DeployedGraph)(),
            MetricCreator().time("2019-01-26 00:00:00").value(44s).workdaysValue(4s).event(Event::DeployedBicycleGraph)(),
            MetricCreator().time("2019-01-26 00:00:00").value(44s).workdaysValue(4s).event(Event::DeployedPedestrianGraph)(),
            MetricCreator().time("2019-01-26 00:00:00").value(44s).workdaysValue(4s).event(Event::DeployedMtrExport)(),
            MetricCreator().time("2019-01-26 00:00:00").value(55s).workdaysValue(5s).event(Event::DeployedRenderer)(),
            MetricCreator().time("2019-01-26 00:00:00").value(77s).workdaysValue(7s)()
        };

        const auto CALCULATION_DATE = chrono::parseSqlDateTime("2019-01-26 00:00:00");

        const auto result = getConsolidatedDeployedMetricsWithEstimations(
            metrics, CALCULATION_DATE, DEFAULT_QUANTILE, in_trunk::YES, DEFAULT_REGION_TO_EVENTS);
        UNIT_ASSERT_EQUAL(result.size(), 4);
        UNIT_ASSERT(result.at({DEFAULT_REGION, USER_TYPE_TRUSTED}).empty());
        UNIT_ASSERT(result.at({DEFAULT_REGION, USER_TYPE_NON_YANDEX}).empty());
        UNIT_ASSERT(result.at({DEFAULT_REGION, USER_TYPE_YANDEX}).empty());

        const auto& resultMetrics = result.at({DEFAULT_REGION, USER_TYPE_COMMON});
        UNIT_ASSERT_EQUAL(resultMetrics.size(), 1);
        UNIT_ASSERT_EQUAL(resultMetrics[0], MetricCreator().time("2019-01-26 00:00:00").value(55s).workdaysValue(5s).event(Event::DeployedTotal)());
    }

    Y_UNIT_TEST(should_calculate_days_out_of_sla) {
        const MetricVec metrics{
            MetricCreator().time("2019-01-24 00:00:00").value(12h).workdaysValue(36h).event(Event::DeployedGraph).inTrunk(in_trunk::NO)(),
            MetricCreator().time("2019-01-25 00:00:00").value(12h).workdaysValue(12h).event(Event::DeployedGraph).inTrunk(in_trunk::NO)(),
            MetricCreator().time("2019-01-26 00:00:00").value(12h).workdaysValue(12h).event(Event::DeployedGraph).inTrunk(in_trunk::NO)(),
            MetricCreator().time("2019-01-27 00:00:00").value(36h).workdaysValue(12h).event(Event::DeployedGraph).inTrunk(in_trunk::NO)(),
        };

        const auto WINDOW = 3;

        const auto result = impl::daysOutOfSlaInSlidingWindow(
            metrics, Event::DeployedGraphBrokenSla, WINDOW, DEFAULT_QUANTILE, in_trunk::NO, DEFAULT_REGION, DEFAULT_USER_TYPE);
        UNIT_ASSERT_EQUAL(result.size(), 2);
        UNIT_ASSERT_EQUAL(std::tie(result[0].value, result[0].workdaysValue), std::make_tuple(0_days, 1_days));
        UNIT_ASSERT_EQUAL(std::tie(result[1].value, result[1].workdaysValue), std::make_tuple(1_days, 0_days));
    }

    Y_UNIT_TEST(should_set_date_for_days_out_of_sla_metric) {
        const MetricVec metrics{
            MetricCreator().time("2019-01-21 00:00:00")(),
            MetricCreator().time("2019-01-22 00:00:00")(),
            MetricCreator().time("2019-01-23 00:00:00")(),
            MetricCreator().time("2019-01-24 00:00:00")(),
        };

        const auto WINDOW = 2;

        auto result = impl::daysOutOfSlaInSlidingWindow(
            metrics, Event::DeployedGraphBrokenSla, WINDOW, DEFAULT_QUANTILE, in_trunk::YES, DEFAULT_REGION, DEFAULT_USER_TYPE);

        UNIT_ASSERT_EQUAL(result.size(), 3);
        UNIT_ASSERT_EQUAL(result[0].time, chrono::parseSqlDateTime("2019-01-22 00:00:00"));
        UNIT_ASSERT_EQUAL(result[1].time, chrono::parseSqlDateTime("2019-01-23 00:00:00"));
        UNIT_ASSERT_EQUAL(result[2].time, chrono::parseSqlDateTime("2019-01-24 00:00:00"));
    }

    Y_UNIT_TEST(should_calculate_achievable_sla_in_sliding_window) {
        const MetricVec metrics{
            // In this test, values and workdays values are in reverse sequence.
            MetricCreator().time("2019-01-21 00:00:00").value(3s).workdaysValue(1s)(),
            MetricCreator().time("2019-01-22 00:00:00").value(5s).workdaysValue(1s)(),
            MetricCreator().time("2019-01-23 00:00:00").value(2s).workdaysValue(3s)(),
            MetricCreator().time("2019-01-24 00:00:00").value(1s).workdaysValue(1s)(),
            MetricCreator().time("2019-01-25 00:00:00").value(1s).workdaysValue(1s)(),
            MetricCreator().time("2019-01-26 00:00:00").value(3s).workdaysValue(2s)(),
            MetricCreator().time("2019-01-27 00:00:00").value(1s).workdaysValue(5s)(),
            MetricCreator().time("2019-01-28 00:00:00").value(1s).workdaysValue(3s)()
        };

        const auto WINDOW = 5;
        const auto BROKEN_SLA_BUDGET = 2;
        const auto PREFIX = "prefix"s;

        auto result = impl::achievableSlaInSlidingWindow(
            metrics, Event::DeployedGraphRealSla, WINDOW, BROKEN_SLA_BUDGET, DEFAULT_QUANTILE, in_trunk::YES, DEFAULT_REGION, DEFAULT_USER_TYPE);
        UNIT_ASSERT_EQUAL(result.size(), 4);
        UNIT_ASSERT_EQUAL(result[0], MetricCreator().event(Event::DeployedGraphRealSla).time("2019-01-25 00:00:00").value(2s).workdaysValue(1s)());
        UNIT_ASSERT_EQUAL(result[1], MetricCreator().event(Event::DeployedGraphRealSla).time("2019-01-26 00:00:00").value(2s).workdaysValue(1s)());
        UNIT_ASSERT_EQUAL(result[2], MetricCreator().event(Event::DeployedGraphRealSla).time("2019-01-27 00:00:00").value(1s).workdaysValue(2s)());
        UNIT_ASSERT_EQUAL(result[3], MetricCreator().event(Event::DeployedGraphRealSla).time("2019-01-28 00:00:00").value(1s).workdaysValue(2s)());
    }
} // Y_UNIT_TEST_SUITE(metrics_calculation)


} // namespace tests

} // namespace maps::wiki::user_edits_metrics
