#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/lib/commit_data.h>
#include <maps/wikimap/mapspro/services/tasks_realtime/src/user_edits_metrics/lib/time_computer.h>
#include <maps/libs/geolib/include/serialization.h>

#include <library/cpp/resource/resource.h>
#include <library/cpp/testing/unittest/registar.h>

namespace maps::wiki::user_edits_metrics::tests {

namespace chr = std::chrono;

struct TestCase
{
    std::string from;
    std::string to;
    chr::seconds expected;
    std::string message;
};

const std::vector<TestCase> TEST_CASES{
    {"2018-02-01 12:00:00.0+03", "2018-02-02 12:00:00.0+03", chr::hours(24), "No holidays"},
    {"2018-02-02 12:00:00.0+03", "2018-02-05 12:00:00.0+03", chr::hours(24), "Two holidays gap"},
    {"2018-02-01 12:00:00.0+03", "2018-02-06 12:00:00.0+03", chr::hours(72), "Two holidays gap"},
    {"2018-02-02 20:00:00.0+03", "2018-02-03 12:00:00.0+03", chr::hours(4), "Start on workday, end on holiday"},
    {"2018-02-04 12:00:00.0+03", "2018-02-05 9:00:00.0+03", chr::hours(9), "Start on holiday, end on workday"},
    {"2018-02-03 12:00:00.0+03", "2018-02-04 9:00:00.0+03", chr::hours(0), "All days are holidays"},
    {"2018-02-22 12:00:00.0+03", "2018-02-26 12:00:00.0+03", chr::hours(24), "Holiday February, 23"},
    {"2018-04-27 12:00:00.0+03", "2018-05-3 12:00:00.0+03", chr::hours(48), "Workday April, 28"},
};

const std::string MOSCOW_AOI = "moscow";
const std::string MOSCOW_AOI_GEOMETRY = "POLYGON((50 50, 55 50, 55 55, 50 55, 50 50))";
AoiRegionsData AOI_REGIONS_DATA = {
    { MOSCOW_AOI, geolib3::WKT::read<geolib3::Polygon2>(MOSCOW_AOI_GEOMETRY) }
};

#define CHECK_EVENT_TO_TIME(eventToTime, expected) \
do { \
    UNIT_ASSERT_VALUES_EQUAL(eventToTime.size(), expected.size()); \
    for (const auto& [event, tp] : eventToTime) { \
        UNIT_ASSERT(expected.count(event) > 0); \
        UNIT_ASSERT_VALUES_EQUAL( \
            chrono::formatSqlDateTime(tp), \
            chrono::formatSqlDateTime(expected[event])); \
    } \
} while(false)

#define CHECK_EQUAL_MAP(map1, map2) \
do { \
    UNIT_ASSERT_VALUES_EQUAL(map1.size(), map2.size()); \
    for (const auto& [key, values] : map1) { \
        auto map2It = map2.find(key); \
        UNIT_ASSERT_C(map2It != map2.end(), "Key " << key << " not found"); \
        auto values2 = map2It->second; \
        for (auto it = values.cbegin(); it != values.cend(); ++it) { \
            UNIT_ASSERT_C(values2.count(*it) > 0, "Second map for " << key << " has no value " << *it); \
        } \
        for (auto it = values2.cbegin(); it != values2.cend(); ++it) { \
            UNIT_ASSERT_C(values.count(*it) > 0, "First map for " << key << " has no value " << *it); \
        } \
    } \
} while(false)


Y_UNIT_TEST_SUITE(time_computer_tests)
{

Y_UNIT_TEST(test_duration_computation)
{
    for (const auto& testCase : TEST_CASES) {
        auto from = chrono::parseSqlDateTime(testCase.from);
        auto to = chrono::parseSqlDateTime(testCase.to);
        auto value = computeDurationMinusWeekends(from, to);
        UNIT_ASSERT_VALUES_EQUAL_C(value.count(), testCase.expected.count(), testCase.message);
    }
}

Y_UNIT_TEST(test_date_parsing)
{
    auto dates = parseDates("");
    UNIT_ASSERT(dates.empty());

    dates = parseDates("2018-2-23");
    UNIT_ASSERT_VALUES_EQUAL(dates.size(), 1);
    auto date = *dates.begin();
    UNIT_ASSERT_VALUES_EQUAL(static_cast<unsigned short>(date.year()), 2018);
    UNIT_ASSERT_VALUES_EQUAL(date.month().as_number(), 2);
    UNIT_ASSERT_VALUES_EQUAL(date.day().as_number(), 23);

    dates = parseDates("2018-2-23\n2018-3-8");
    UNIT_ASSERT_VALUES_EQUAL(dates.size(), 2);
    date = *dates.begin();
    UNIT_ASSERT_VALUES_EQUAL(static_cast<unsigned short>(date.year()), 2018);
    UNIT_ASSERT_VALUES_EQUAL(date.month().as_number(), 2);
    UNIT_ASSERT_VALUES_EQUAL(date.day().as_number(), 23);
    date = *(++dates.begin());
    UNIT_ASSERT_VALUES_EQUAL(static_cast<unsigned short>(date.year()), 2018);
    UNIT_ASSERT_VALUES_EQUAL(date.month().as_number(), 3);
    UNIT_ASSERT_VALUES_EQUAL(date.day().as_number(), 8);

    UNIT_CHECK_GENERATED_EXCEPTION(parseDates("2018-2-31"), boost::gregorian::bad_day_of_month);

    UNIT_CHECK_GENERATED_EXCEPTION(parseDates("asdf"), boost::bad_lexical_cast);
}

Y_UNIT_TEST(test_date_files)
{
    const auto holidays = parseDates(NResource::Find(HOLIDAYS_DATA_RESOURCE_KEY));
    const auto workdays = parseDates(NResource::Find(WORKDAYS_DATA_RESOURCE_KEY));

    UNIT_ASSERT_VALUES_EQUAL(holidays.size(), 31);
    UNIT_ASSERT_VALUES_EQUAL(workdays.size(), 3);

    for (const auto& date : workdays) {
        UNIT_ASSERT_VALUES_EQUAL(holidays.count(date), 0);
    }
}

} // Y_UNIT_TEST_SUITE

Y_UNIT_TEST_SUITE(utils_test)
{

const auto BEFORE_TIMEPOINT_1111 = chrono::parseIsoDateTime("2018-04-11 01:00:00+00:00");
const auto TIMEPOINT_1111        = chrono::parseIsoDateTime("2018-05-11 01:00:00+00:00");
const auto MID_TIMEPOINT         = chrono::parseIsoDateTime("2018-05-12 01:00:00+00:00");
const auto TIMEPOINT_1113        = chrono::parseIsoDateTime("2018-05-13 01:00:00+00:00");
const auto TIMEPOINT_1114        = chrono::parseIsoDateTime("2018-05-14 01:00:00+00:00");
const auto AFTER_TIMEPOINT_1114  = chrono::parseIsoDateTime("2018-06-14 01:00:00+00:00");

const BranchesById BRANCHES_BY_ID{
    {1111, BranchInfo{
        TIMEPOINT_1111,
        RegionToEventToTime{
            {"eu3", EventToTime{
                {Event::DeployedGraph,      TIMEPOINT_1111},
                {Event::DeployedCams,       TIMEPOINT_1111},
                {Event::DeployedMtrExport,  TIMEPOINT_1111},
            }},
            {"cis1", EventToTime{
                {Event::DeployedGraph,      TIMEPOINT_1111},
                {Event::DeployedCams,       TIMEPOINT_1111},
            }}
        }
    }},
    {1113, BranchInfo{
        TIMEPOINT_1113,
        RegionToEventToTime{
            {"eu3", EventToTime{
                {Event::DeployedGraph,      TIMEPOINT_1113},
                {Event::DeployedCams,       TIMEPOINT_1113},
                {Event::DeployedMtrExport,  TIMEPOINT_1113},
            }},
            {"eu1", EventToTime{
                {Event::DeployedGraph,      TIMEPOINT_1113},
                {Event::DeployedCams,       TIMEPOINT_1113},
            }},
            {"cis1", EventToTime{
                {Event::DeployedGraph,      TIMEPOINT_1113},
                {Event::DeployedCarparks,   TIMEPOINT_1113},
            }}
        }
    }},
    {1114, BranchInfo{
        TIMEPOINT_1114,
        RegionToEventToTime{
            {"cis1", EventToTime{
                {Event::DeployedGeocoder,   TIMEPOINT_1114}
            }}
        }
    }}
};

// Always add aoi regions and 'all_regions' for each event
EventToRegions EVENTS_TO_REGIONS{
    {Event::Approved,          {"all_regions", "cis1", "eu1", "eu3", "moscow"}},
    {Event::InStable,          {"all_regions", "cis1", "eu1", "eu3", "moscow"}},
    {Event::DeployedCams,      {"all_regions", "cis1", "eu1", "eu3", "moscow"}},
    {Event::DeployedCarparks,  {"all_regions", "cis1",               "moscow"}},
    {Event::DeployedGeocoder,  {"all_regions", "cis1",               "moscow"}},
    {Event::DeployedGraph,     {"all_regions", "cis1", "eu1", "eu3", "moscow"}},
    {Event::DeployedMtrExport, {"all_regions",                "eu3", "moscow"}},
    {Event::Resolved,          {"all_regions", "cis1", "eu1", "eu3", "moscow"}}
};

Y_UNIT_TEST(test_region_events_for_commit)
{
    CHECK_EVENT_TO_TIME(
        calcEventToTimeForBranchAndRegion(BRANCHES_BY_ID, 1111, "eu3"),
        (EventToTime{
            {Event::InStable,           TIMEPOINT_1111},
            {Event::DeployedGraph,      TIMEPOINT_1111},
            {Event::DeployedCams,       TIMEPOINT_1111},
            {Event::DeployedMtrExport,  TIMEPOINT_1111}
        })
    );
    CHECK_EVENT_TO_TIME(
        calcEventToTimeForBranchAndRegion(BRANCHES_BY_ID, 1111, "cis2"),
        (EventToTime{{Event::InStable,  TIMEPOINT_1111}}));
    CHECK_EVENT_TO_TIME(
        calcEventToTimeForBranchAndRegion(BRANCHES_BY_ID, 1111, "eu1"),
        (EventToTime{
            {Event::InStable,           TIMEPOINT_1111},
            {Event::DeployedGraph,      TIMEPOINT_1113},
            {Event::DeployedCams,       TIMEPOINT_1113}
        })
    );
    CHECK_EVENT_TO_TIME(
        calcEventToTimeForBranchAndRegion(BRANCHES_BY_ID, 1111, "cis1"),
        (EventToTime{
            {Event::InStable,           TIMEPOINT_1111},
            {Event::DeployedGraph,      TIMEPOINT_1111},
            {Event::DeployedCams,       TIMEPOINT_1111},
            {Event::DeployedCarparks,   TIMEPOINT_1113},
            {Event::DeployedGeocoder,   TIMEPOINT_1114}
        })
    );
    CHECK_EVENT_TO_TIME(
        calcEventToTimeForBranchAndRegion(BRANCHES_BY_ID, 1114, "eu3"),
        (EventToTime{{Event::InStable,  TIMEPOINT_1114}}));
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        calcEventToTimeForBranchAndRegion(BRANCHES_BY_ID, 1115, "eu3"),
        maps::RuntimeError,
        "branch id: 1115 not found");
}

Y_UNIT_TEST(test_calc_events_to_regions)
{
    auto eventToRegions = calcEventToRegions(BRANCHES_BY_ID, AOI_REGIONS_DATA, TIMEPOINT_1111);
    CHECK_EQUAL_MAP(eventToRegions, EVENTS_TO_REGIONS);
}

Y_UNIT_TEST(test_calc_events_to_regions_early_time)
{
    auto eventToRegions = calcEventToRegions(BRANCHES_BY_ID, AOI_REGIONS_DATA, BEFORE_TIMEPOINT_1111);
    CHECK_EQUAL_MAP(eventToRegions, EVENTS_TO_REGIONS);
}

Y_UNIT_TEST(test_calc_events_to_regions_min_time)
{
    auto eventToRegions = calcEventToRegions(BRANCHES_BY_ID, AOI_REGIONS_DATA, TIMEPOINT_1114);
    EventToRegions expected{
        {Event::Approved,         {"all_regions", "cis1", "moscow"}},
        {Event::InStable,         {"all_regions", "cis1", "moscow"}},
        {Event::DeployedGeocoder, {"all_regions", "cis1", "moscow"}},
        {Event::Resolved,         {"all_regions", "cis1", "moscow"}}
    };
    CHECK_EQUAL_MAP(eventToRegions, expected);
}

Y_UNIT_TEST(test_calc_events_to_regions_mid_time)
{
    EventToRegions expected{
        {Event::Approved,          {"all_regions", "cis1", "eu1", "eu3", "moscow"}},
        {Event::InStable,          {"all_regions", "cis1", "eu1", "eu3", "moscow"}},
        {Event::DeployedCams,      {"all_regions",         "eu1", "eu3", "moscow"}},
        {Event::DeployedCarparks,  {"all_regions", "cis1",               "moscow"}},
        {Event::DeployedGeocoder,  {"all_regions", "cis1",               "moscow"}},
        {Event::DeployedGraph,     {"all_regions", "cis1", "eu1", "eu3", "moscow"}},
        {Event::DeployedMtrExport, {"all_regions",                "eu3", "moscow"}},
        {Event::Resolved,          {"all_regions", "cis1", "eu1", "eu3", "moscow"}}
    };

    auto eventToRegions = calcEventToRegions(BRANCHES_BY_ID, AOI_REGIONS_DATA, MID_TIMEPOINT);
    CHECK_EQUAL_MAP(eventToRegions, expected);
}

Y_UNIT_TEST(test_calc_events_to_regions_late_time)
{
    EventToRegions expected{
        {Event::Approved,         {"all_regions", "moscow"}},
        {Event::InStable,         {"all_regions", "moscow"}},
        {Event::Resolved,         {"all_regions", "moscow"}}
    };
    CHECK_EQUAL_MAP(
        calcEventToRegions(BRANCHES_BY_ID, AOI_REGIONS_DATA, AFTER_TIMEPOINT_1114),
        expected
    );
}

Y_UNIT_TEST(test_calc_events_to_regions_from_commits)
{
    EventToRegions expected{
        {Event::Approved,          {"cis1", "eu1", "eu3"}},
        {Event::InStable,          {"cis1", "eu1", "eu3"}},
        {Event::DeployedCams,      {"cis1", "eu1", "eu3"}},
        {Event::DeployedCarparks,  {"cis1"}},
        {Event::DeployedGeocoder,  {"cis1"}},
        {Event::DeployedGraph,     {"cis1", "eu1", "eu3"}},
        {Event::DeployedMtrExport, {"eu3"}},
        {Event::Resolved,          {"cis1", "eu1", "eu3"}}
    };
    CHECK_EQUAL_MAP(
        calcEventToDeployedRegions(BRANCHES_BY_ID, TIMEPOINT_1111),
        expected
    );
}

Y_UNIT_TEST(test_calc_region_to_deployed_events)
{
    RegionToEvents expected{
        {"all_regions", {
            Event::DeployedCams,
            Event::DeployedCarparks,
            Event::DeployedGeocoder,
            Event::DeployedGraph,
            Event::DeployedMtrExport
        }},
        {"cis1", {
            Event::DeployedCams,
            Event::DeployedCarparks,
            Event::DeployedGeocoder,
            Event::DeployedGraph,
        }},
        {"eu1", {
            Event::DeployedCams,
            Event::DeployedGraph
        }},
        {"eu3", {
            Event::DeployedCams,
            Event::DeployedGraph,
            Event::DeployedMtrExport
        }},
        {"moscow", {
            Event::DeployedCams,
            Event::DeployedCarparks,
            Event::DeployedGeocoder,
            Event::DeployedGraph,
            Event::DeployedMtrExport
        }},
    };
    RegionToEvents regionToDeployedEvents = calcRegionToDeployedEvents(EVENTS_TO_REGIONS);
    CHECK_EQUAL_MAP(regionToDeployedEvents, expected);
}

Y_UNIT_TEST(test_calc_event_to_min_time)
{
    RegionToEventToTime regionEvents{
        {"cis1", EventToTime{
            {Event::DeployedGraph,      TIMEPOINT_1111},
            {Event::DeployedCarparks,   TIMEPOINT_1113},
        }},
        {"cis2", EventToTime{
            {Event::DeployedGraph,      TIMEPOINT_1114},
            {Event::DeployedCams,       TIMEPOINT_1113},
            {Event::DeployedMtrExport,  TIMEPOINT_1113},
        }},
        {"eu1", EventToTime{
            {Event::DeployedGraph,      TIMEPOINT_1113},
            {Event::DeployedCams,       TIMEPOINT_1111},
        }}
    };
    CHECK_EVENT_TO_TIME(
        calcEventToFirstDeployTime(regionEvents),
        (EventToTime{
            {Event::DeployedCams,      TIMEPOINT_1111},
            {Event::DeployedCarparks,  TIMEPOINT_1113},
            {Event::DeployedGraph,     TIMEPOINT_1111},
            {Event::DeployedMtrExport, TIMEPOINT_1113}
        })
    );
}

} // Y_UNIT_TEST_SUITE

} // namespace maps::wiki::user_edits_metrics::tests
