#include "params.h"

#include <maps/libs/chrono/include/days.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/enum_io/include/enum_io_fwd.h>
#include <maps/libs/stringutils/include/join.h>
#include <maps/libs/stringutils/include/to_string.h>

#include <maps/wikimap/mapspro/services/tasks_social/src/assessment_sampler/lib/sample.h>
#include <maps/wikimap/mapspro/services/tasks_social/src/assessment_sampler/lib/common.h>

namespace assessment = maps::wiki::assessment;
namespace sampler = maps::wiki::assessment::sampler;
namespace chrono = maps::chrono;

namespace {

template<typename Enum>
std::string toString()
{
    const auto values = maps::enum_io::enumerateValues<Enum>();

    return maps::stringutils::join(
        maps::stringutils::toStrings(values),
        ", "
    );
}

} // namespace


void Params::log(maps::log8::Level level) const
{
    MAPS_LOG(level) << "Generating assessment sample '" << name << "'";
    MAPS_LOG(level) << "\tdomain: " << domain;
    MAPS_LOG(level) << "\tqualification: " << qualification;
    MAPS_LOG(level) << "\tstaffType: " << staffType;
    MAPS_LOG(level) << "\thypothesesMode: " << bool(hypothesesMode);
    MAPS_LOG(level) << "\tdate: " << chrono::formatIsoDate(date);
    MAPS_LOG(level) << "\tdays: " << days;
    MAPS_LOG(level) << "\tseed: " << seed;
    MAPS_LOG(level) << "\tdryRun: " << dryRun;
    MAPS_LOG(level) << "\tallowDuplicateSample: " << allowDuplicateSample;
}

Params parseCommandLineParams(int argc, char* argv[])
{
    const auto yesterday = std::chrono::system_clock::now() - chrono::Days(1);

    maps::cmdline::Parser parser;

    auto domain = parser.string("domain")
        .help("Entity domain: " + toString<assessment::Entity::Domain>() + ".");
    auto qualification = parser.string("qualification")
        .help("Sample qualification: " + toString<assessment::Qualification>() + ".");
    auto dateStr = parser.string("date")
        .defaultValue(chrono::formatIsoDate(yesterday))
        .help("Date (last date) of tasks in sample.");
    auto days = parser.size_t("days")
        .defaultValue(1)
        .help("Length of sample period.");
    auto dryRun = parser.flag("dry-run")
        .defaultValue(false)
        .help("Skip commit to the database.");
    auto allowDuplicateSample = parser.flag("allow-duplicate-sample")
        .defaultValue(false)
        .help("Create a new sample even if same-named sample already exists.");

    auto seed = parser.size_t("seed")
        .help("Seed for random number generator.");
    auto name = parser.string("name")
        .help("Sample name. Default is '[piecework-][expert-]{domain}-[hypotheses-][{first-date}-]{date}'.");
    auto syslogTag = parser.string("syslog-tag")
        .help("Redirect log output to syslog with given tag.");
    auto workerConfig = parser.file("config")
        .help("Path to worker configuration.");
    auto staffType = parser.string("staff-type")
        .help("Staff type: " + toString<sampler::StaffType>() + ".");
    auto hypothesesMode = parser.flag("hypotheses")
        .defaultValue(false)
        .help("Generate hypotheses sample. Can be used with `feedback` domain only.");

    parser.parse(argc, argv);

    Params params = {
        .domain = maps::enum_io::fromString<assessment::Entity::Domain>(domain),
        .qualification = maps::enum_io::fromString<assessment::Qualification>(qualification),
        .date = chrono::parseIsoDate(dateStr),
        .days = days,
        .dryRun = dryRun,
        .allowDuplicateSample = allowDuplicateSample,
        .seed = static_cast<size_t>(std::time(nullptr)),
        .staffType = maps::enum_io::fromString<sampler::StaffType>(staffType),
        .hypothesesMode = sampler::HypothesesMode(bool(hypothesesMode))
    };
    REQUIRE(
        params.staffType == sampler::StaffType::Staff ||
        params.domain != assessment::Entity::Domain::Tracker,
        "Only 'staff' work with 'tracker' domain.");
    REQUIRE(
        !hypothesesMode ||
        params.domain == assessment::Entity::Domain::Feedback,
        "Hypotheses mode can be used with `feedback` domain only.");
    REQUIRE(
        params.staffType != sampler::StaffType::Outsource ||
        params.domain == assessment::Entity::Domain::Edits || hypothesesMode,
        "Outsourcers work with 'edits' and 'hypotheses' only."
    );

    if (seed.defined()) {
        params.seed = seed;
    }
    if (name.defined()) {
        params.name = name;
    } else {
        params.name = sampler::sampleName(
            params.domain,
            params.qualification,
            params.date,
            params.days,
            params.staffType,
            params.hypothesesMode
        );
    }
    if (syslogTag.defined()) {
        params.syslogTag = syslogTag;
    }
    if (workerConfig.defined()) {
        params.workerConfigPath = workerConfig;
    }

    return params;
}
