#include "feedback.h"

#include <maps/wikimap/mapspro/services/tasks_social/src/assessment_sampler/lib/common.h>
#include <maps/wikimap/mapspro/services/tasks_social/src/assessment_sampler/lib/grouped_units.h>

#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/robot.h>

#include <fmt/format.h>


namespace maps::wiki::assessment::sampler {

namespace feedback = maps::wiki::social::feedback;
using namespace fmt::literals;

namespace {

using TaskFilters = std::vector<feedback::TaskFilter>;

const std::string FBAPI = "fbapi";

const feedback::TaskOperations ASSESSED_OPERATIONS = {
    feedback::TaskOperation::Accept,
    feedback::TaskOperation::Reject,
    feedback::TaskOperation::NeedInfo,
};

} // namespace

namespace impl {

feedback::TaskFilter closedFilter(
    chrono::TimePoint timepointMin,
    chrono::TimePoint timepointMax)
{
    return feedback::TaskFilter()
        .uiFilterStatus(feedback::UIFilterStatus::Resolved)
        .resolvedAt(social::DateTimeCondition(timepointMin, timepointMax))
        .notResolvedBy({common::ALL_ROBOTS_UIDS.begin(), common::ALL_ROBOTS_UIDS.end()});
}

feedback::TaskFilter needInfoFilter(
    chrono::TimePoint timepointMin,
    chrono::TimePoint timepointMax)
{
    return feedback::TaskFilter()
        .uiFilterStatus(feedback::UIFilterStatus::NeedInfo)
        .stateModifiedAt(social::DateTimeCondition(timepointMin, std::nullopt))
        .createdAt(social::DateTimeCondition(std::nullopt, timepointMax));
}

} // namespace impl

namespace {

GroupNameToUnits collectUnits(
    const feedback::Tasks& tasks,
    const feedback::TaskIdsToHistory& taskIdsToHistory,
    chrono::TimePoint timepointMin,
    chrono::TimePoint timepointMax,
    const TUids& allowedUids)
{
    GroupNameToUnits result;

    for (const auto& task: tasks) {
        const auto historyIt = taskIdsToHistory.find(task.id());
        if (historyIt == taskIdsToHistory.end()) {
            continue;
        }
        for (const auto& historyItem: historyIt->second.items()) {
            if (historyItem.modifiedAt() < timepointMin) {
                continue;
            }
            if (historyItem.modifiedAt() >= timepointMax) {
                break;
            }
            if (ASSESSED_OPERATIONS.count(historyItem.operation()) == 0) {
                continue;
            }
            if (!allowedUids.contains(historyItem.modifiedBy())) {
                continue;
            }

            const std::string groupName = fmt::format(
                "{type}-{operation}-{modified_by}",
                "type"_a = toString(task.type()),
                "operation"_a = toString(historyItem.operation()),
                "modified_by"_a = historyItem.modifiedBy()
            );

            result[groupName].emplace_back(Unit{
                .id = UNKNOWN_UNIT_ID,
                .entity = {
                    .id = std::to_string(task.id()),
                    .domain = Entity::Domain::Feedback
                },
                .action = {
                    .name = std::string(feedback::toString(historyItem.operation())),
                    .by = historyItem.modifiedBy(),
                    .at = historyItem.modifiedAt()
                }
            });
        }
    }

    return result;
}

TIds collectIds(const feedback::Tasks& tasks)
{
    TIds retVal;
    for (const auto& task: tasks) {
        retVal.emplace(task.id());
    }
    return retVal;
}

GroupNameToUnits loadFeedbackUnits(
    feedback::GatewayRO& feedbackGatewayRO,
    chrono::TimePoint timepointMin,
    chrono::TimePoint timepointMax,
    const TUids& allowedUids,
    TaskFilters taskFilters)
{
    GroupNameToUnits result;

    for (const auto& filter: taskFilters) {
        auto tasks = feedbackGatewayRO.tasksByFilter(filter);
        auto taskIdsToHistory = feedbackGatewayRO.history(collectIds(tasks));
        result += collectUnits(tasks, taskIdsToHistory, timepointMin, timepointMax, allowedUids);
    }

    return result;
}

} // namespace


GroupNameToUnits loadFeedbackUnits(
    feedback::GatewayRO& fbGw,
    chrono::TimePoint timepointMin,
    chrono::TimePoint timepointMax,
    const TUids& allowedUids,
    HypothesesMode hypothesesMode)
{
    TaskFilters filters;
    switch (hypothesesMode) {
        case HypothesesMode::No:
            filters.emplace_back(impl::closedFilter  (timepointMin, timepointMax).workflow(feedback::Workflow::Feedback));
            filters.emplace_back(impl::needInfoFilter(timepointMin, timepointMax).workflow(feedback::Workflow::Feedback));
            break;
        case HypothesesMode::Yes:
            filters.emplace_back(impl::closedFilter(timepointMin, timepointMax).workflow(feedback::Workflow::Task).experiment(false));
            break;
    }

    return loadFeedbackUnits(fbGw, timepointMin, timepointMax, allowedUids, filters);
}

} // maps::wiki::assessment::sampler
