#include "sample.h"

#include "common.h"
#include "limits.h"
#include "tracker.h"
#include "uids.h"

#include <maps/libs/chrono/include/days.h>
#include <maps/libs/enum_io/include/enum_io.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/assessment/include/gateway.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/robot.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/feedback/gateway_ro.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/gateway.h>

namespace maps::wiki::assessment::sampler {

constexpr maps::enum_io::Representations<StaffType>
STAFF_TYPE_REPRESENTATION {
    {StaffType::Staff,     "staff"},
    {StaffType::Piecework, "piecework"},
    {StaffType::Outsource, "outsource"},
};
DEFINE_ENUM_IO(StaffType, STAFF_TYPE_REPRESENTATION);


void makeSample(
    assessment::Gateway& assessmentGw,
    Entity::Domain domain,
    Qualification qualification,
    const std::string& sampleName,
    const std::vector<TId>& unitIds,
    size_t tasksPerUnit)
{
    INFO() << "Sample size: " << unitIds.size();

    const auto sampleId = assessmentGw.createSample(
        domain,
        qualification,
        sampleName,
        TIds{unitIds.cbegin(), unitIds.cend()},
        tasksPerUnit
    );

    INFO() << "Sample ID: " << sampleId;
}


std::string sampleName(
    Entity::Domain domain,
    Qualification qualification,
    chrono::TimePoint lastDate,
    size_t days,
    StaffType staffType,
    HypothesesMode hypothesesMode)
{
    static const std::string DATE_FORMAT = "%Y%m%d";

    std::stringstream name;

    switch (staffType) {
    case StaffType::Staff:
        // do nothing
        break;
    case StaffType::Piecework:
        name << PIECEWORK_PREFIX;
        break;
    case StaffType::Outsource:
        name << OUTSOURCE_PREFIX;
        break;
    }

    if (qualification != Qualification::Basic) {
        name << qualification << "-";
    }
    name << domain << "-";

    switch (hypothesesMode) {
    case HypothesesMode::Yes:
        name << HYPOTHESES_PREFIX;
        break;
    case HypothesesMode::No:
        // do nothing
        break;
    }

    if (days != 1) {
        auto firstDate = lastDate - chrono::Days(days - 1);
        name << chrono::formatIntegralDateTime(firstDate, DATE_FORMAT) << "-";
    }
    name << chrono::formatIntegralDateTime(lastDate, DATE_FORMAT);

    return name.str();
}

} // maps::wiki::assessment::sampler
