#include "uids.h"

#include "init.h"

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/secrets.h>

#include <maps/libs/log8/include/log8.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/io.h>


namespace maps::wiki::assessment::sampler {

namespace {
const std::string PIECEWORK_ROLE = "cartographer_sdel_tool";
const std::string OUTSOURCE_ROLE = "outsource-role"; // NMAPS-14230 Add outsource to piecework samples
const std::string STAFF_GROUP = "cartographer_staff";

const auto LINKED_ACCOUNTS = "//home/maps/core/nmaps/analytics/tasks_payment/dictionaries/linked_accounts/latest";
const auto IS_PRIMARY_LINK = "is_primary_link";
const auto STAFF_LOGIN = "staff_login";
const auto PUID = "puid";
const auto YT_CLUSTER = "hahn";

std::set<acl::ID>
getRoleIdsByNames(acl::ACLGateway& gateway, const std::set<std::string>& names)
{
    std::set<acl::ID> ids;
    for (const auto& name : names) {
        ids.emplace(gateway.role(name).id());
    }
    return ids;
}

TUids
getUids(const std::vector<acl::User>& users)
{
    TUids uids;
    for (const auto& user : users) {
        uids.emplace(user.uid());
    }
    return uids;
}

TUids
loadRolesUids(acl::ACLGateway& aclGateway, const std::set<std::string>& roleNames)
{
    ASSERT(!roleNames.empty());

    const auto userIds = aclGateway.userIdsByRoles(getRoleIdsByNames(aclGateway, roleNames));
    return getUids(aclGateway.usersByIds({userIds.begin(), userIds.end()}));
}

TUids
loadGroupUids(acl::ACLGateway& aclGateway, const std::string& groupName)
{
    return getUids(aclGateway.group(groupName).users());
}

} // namespace

TUids
loadAllowedUids(
    StaffType staffType,
    const std::optional<std::string>& workerConfigPath)
{
    INFO() << "Loading uids...";

    auto corePoolHolder = initPoolHolder(CORE_DB_ID, workerConfigPath);
    auto coreTxn = corePoolHolder.pool().slaveTransaction();
    acl::ACLGateway aclGateway{*coreTxn};

    TUids result;

    switch (staffType) {
    case StaffType::Staff:
        result = sampler::loadGroupUids(aclGateway, STAFF_GROUP);
        break;
    case StaffType::Piecework:
        result = sampler::loadRolesUids(aclGateway, {PIECEWORK_ROLE});
        break;
    case StaffType::Outsource:
        result = sampler::loadRolesUids(aclGateway, {OUTSOURCE_ROLE});
        break;
    }

    INFO() << "Got " << result.size() << " uids";
    return result;
}

StaffLoginToPuid
loadStaffLoginToPuidFromYt()
{
    using namespace maps::wiki::common::secrets;

    NYT::JoblessInitialize();
    auto ytClient = NYT::CreateClient(
        YT_CLUSTER,
        NYT::TCreateClientOptions().Token(tokenByKey(Key::RobotWikimapYtToken))
    );
    auto ytTxn = ytClient->StartTransaction();
    auto tableReader = ytTxn->CreateTableReader<NYT::TNode>(LINKED_ACCOUNTS);

    StaffLoginToPuid result;
    for (const auto& cursor: *tableReader) {
        const auto& row = cursor.GetRow();
        if (row[IS_PRIMARY_LINK].AsBool()) {
            result[row[STAFF_LOGIN].AsString()] = row[PUID].AsUint64();
        }
    }
    return result;
}

} // maps::wiki::assessment::sampler
